/*******************************************************************************
  GPU sensor
  Copyright (C) 2024 Mavridis Philippe <mavridisf@gmail.com>

  This program is free software: you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation, either version 3 of the License, or (at your option) any later
  version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
  You should have received a copy of the GNU General Public License along with
  this program. If not, see <http://www.gnu.org/licenses/>.

  Improvements and feedback are welcome!
*******************************************************************************/

// TQt
#include <tqregexp.h>

// TDE
#include <tdestandarddirs.h>
#include <kdebug.h>

// Superkaramba
#include "gpusensor.h"

GPUSensor::GPUSensor(TQString gpuDriver, TQString gpuId, int interval)
    : Sensor(interval), m_gpuDriver(gpuDriver), m_gpuId(gpuId)
{
    if (m_gpuDriver.lower() == "nvidia")
    {
        TQString nvsmi = TDEStandardDirs::findExe("nvidia-smi");
        if (nvsmi.isNull())
        {
            kdError() << "The NVidia System Management Interface software is not avaiable." << endl;
            return;
        }

        m_command << nvsmi << "--query-gpu" << "utilization.gpu"
                           << "--format=csv,noheader";

        if (!m_gpuId.isNull())
        {
            m_command << TQString("--id=%1").arg(m_gpuId);
        }
    }

    else
    {
        kdError() << "Unsupported driver specified for GPU sensor (" << m_gpuDriver << ");" << endl
                  << "\tSupported drivers are: nvidia" << endl;
    }

    connect(&m_proc, TQ_SIGNAL(receivedStdout(TDEProcess*, char*, int)),
            this, TQ_SLOT(receivedStdout(TDEProcess*, char*, int)));
    connect(&m_proc, TQ_SIGNAL(processExited(TDEProcess*)),
            this, TQ_SLOT(processExited(TDEProcess*)));
}

GPUSensor::~GPUSensor()
{
}

void GPUSensor::update()
{
    if (m_command.isEmpty()) return;

    m_proc.clearArguments();
    m_proc << m_command;

    m_proc.start(TDEProcess::NotifyOnExit, TDEProcess::Stdout);
}

void GPUSensor::receivedStdout(TDEProcess *proc, char *buffer, int buflen)
{
    buffer[buflen] = 0;
    m_buffer += TQCString(buffer);
}

TQString GPUSensor::getLoad()
{
    if (m_gpuDriver.lower() == "nvidia")
    {
        return m_buffer.left(m_buffer.length() - 3);
    }

    return TQString::null;
}

#define SUB_FORMAT_STR(fstring, value) \
    format.replace(TQRegExp(#fstring, false), value)
void GPUSensor::processExited(TDEProcess *proc)
{
    SensorParams *sp;
    Meter *meter;
    TQString format;
    TQString load = getLoad();
    m_buffer = TQString::null;

    TQObjectListIt it(*objList);
    while (it != 0)
    {
        sp = (SensorParams*)(*it);
        meter = sp->getMeter();
        format = sp->getParam("FORMAT");

        if( format.length() == 0)
        {
            format = "%v";
        }

        SUB_FORMAT_STR(%load, load);
        SUB_FORMAT_STR(%v,    load);

        meter->setValue(format);
        ++it;
    }
}

#undef SUB_FORMAT_STR

void GPUSensor::setMaxValue(SensorParams *sp)
{
    sp->getMeter()->setMax(100);
}

#include "gpusensor.moc"