/***************************************************************************
 *   Copyright (C) 2005 by Pawel Nawrocki                                  *
 *   pnawrocki@interia.pl                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/

#include "netlistviewitem.h"

#include <iostream>

#include <kiconloader.h>
#include <kiconeffect.h>
#include <tdeversion.h>

void NetListViewItem::paintCell(TQPainter *p, const TQColorGroup &cg, int column, int width, int alignment)
{
        /// HACK fixes: higher item (connected) is drawn using regular height upon widget change (widgetStack);
        if ( height()!=mHeight )
                setHeight( mHeight );

        /// PREPARE COLORS ///
        TQColor bgColor, fgColor;
        /// colors of selected item
        if ( listView()->isSelected(this) ) {
                bgColor = cg.color( TQColorGroup::Highlight );	/// settings for selected item;
                fgColor = cg.color( TQColorGroup::HighlightedText );
                /// colors of deselected item`
        } else {
                if (mConnected)
                        bgColor = cg.color( TQColorGroup::Background);
                else {
#if KDE_IS_VERSION(3,4,0)
                        bgColor = ((TDEListViewItem*)this)->backgroundColor(column);
#else

                        bgColor = ((TDEListViewItem*)this)->backgroundColor();
#endif

                }
                fgColor = cg.color( TQColorGroup::Text);
                if (mQuality<8)
                        fgColor = fgColor.light();
        }

        /// DRAW BACKGROUND ///
        p->fillRect(0,0,width,height(),bgColor);
        if (mConnected) {
                /// draw a line separating connectedItem from the rest of the list.
                p->setPen( bgColor.dark(130) );
                p->drawLine(0, height()-1, width, height()-1);
        }

        switch (column) {

                /// DRAW QUALITY ///
        case mQualityColumn: {
                        TQPixmap qualityIcon = SmallIcon("knewstuff");
                        TQPixmap qualityIconGray = TDEIconEffect().apply( qualityIcon, TDEIconEffect::ToGray, 1, TQt::black, true );
                        int barWidth = int(mQuality/8)*8;
                        if (mQuality>0)
                                barWidth+=8;	//add 8 (half a star) b/c int rounds down.
                        if (barWidth>96)
                                barWidth=96;
                        int icoTop = int( ( this->height()-16 )/2 );
                        p->drawTiledPixmap(listView()->itemMargin(),icoTop,6*16, 16, qualityIconGray );
                        p->drawTiledPixmap(listView()->itemMargin(),icoTop,barWidth, 16, qualityIcon );
                        break;
                }

                /// DRAW ENCRYPTION ///
        case mEncColumn: {
                        if (mEnc) {
                                int icoTop = int( ( this->height()-16 )/2 );
				int icoLeft = int( ( width-listView()->itemMargin()-16 )/2 );
                                TQPixmap encIcon = SmallIcon("encrypted");
                                p->drawPixmap(icoLeft,icoTop, encIcon );
                        }
                        break;
                }

                /// DRAW ESSID ///
        case mEssidColumn: {
                        /// draw icon and its shadow.
                        if (mConnected) {
                                TQPixmap connectedIcon;
                                connectedIcon = SmallIcon("forward");
                                int icoTop = int( ( this->height()-16 )/2 );
                                p->drawPixmap(listView()->itemMargin(),icoTop, connectedIcon );
                        }

                        TQFont mFont = listView()->font();
                        if (mConnected)
                                mFont.setBold( true );
                        if (mHidden)
                                mFont.setItalic( true );
                        p->setFont( mFont );
                        /// draw shadow + essid name(not connected)
                        if (mConnected) {
                                p->setPen( bgColor.dark(130) );
                                p->drawText(16+(listView()->itemMargin()*2),0,width, height(), AlignVCenter, mEssid);
                                p->setPen( bgColor.dark(220) );
                                p->drawText(16+(listView()->itemMargin()*2)-1,-1,width, height(), AlignVCenter, mEssid);
                                p->setPen( fgColor );
                                p->drawText(16+(listView()->itemMargin()*2)-2,-2,width, height(), AlignVCenter, mEssid);

                        } else {
                                /// draw essid name (not connected)
                                p->setPen( fgColor );
                                p->drawText(listView()->itemMargin(),0,width, height(), AlignVCenter, mEssid);
                        }
                        break;
                }

                /// DRAW CHANNEL ///
        case mChanColumn: {
                        TQFont mFont = listView()->font();
                        mFont.setItalic(true);
                        if (mConnected)
                                mFont.setBold( true );
                        p->setFont( mFont );
                        if (mConnected) {
                                p->setPen( bgColor.dark(130) );
                                p->drawText(listView()->itemMargin(),0,width, height(), AlignCenter, mChannel);
                                p->setPen( bgColor.dark(220) );
                                p->drawText(listView()->itemMargin()-1,-1,width, height(), AlignCenter, mChannel);
                        } else {
                                p->setPen( bgColor.dark(220) );
                                p->drawText(listView()->itemMargin(),0,width, height(), AlignCenter, mChannel);
                        }
                        break;
                }
                /// DRAW ACCESS POINT ///
        case mAPColumn: {
                        TQFont mFont = listView()->font();
                        if (mHidden)
                                mFont.setItalic( true );
                        if (mConnected)
                                mFont.setBold( true );
                        p->setFont( mFont );
                        if (mConnected) {
                                p->setPen( bgColor.dark(130) );
                                p->drawText(listView()->itemMargin(),0,width, height(), AlignVCenter, mAP);
                                p->setPen( bgColor.dark(220) );
                                p->drawText(listView()->itemMargin()-1,-1,width, height(), AlignVCenter, mAP);

                        } else {
                                p->setPen( bgColor.dark(220) );
                                p->drawText(listView()->itemMargin(),0,width, height(), AlignVCenter, mAP);
                        }
                        break;
                }

        default:
                TDEListViewItem::paintCell(p, cg, column, width, alignment);

        } //switch
}

int NetListViewItem::width(const TQFontMetrics &fm, const TQListView *lv, int column) const
{
        int w;
        TQFont mFont = listView()->font();
        if (mConnected)
                mFont.setBold(true);
        if (mHidden)
                mFont.setItalic(true);
        TQFontMetrics mFm( mFont );

        if (column == mQualityColumn)
                w = 6*16 + (lv->itemMargin()*2);
        else if (column == mEncColumn)
                w = 16 + (lv->itemMargin()*2);
        else if (column == mChanColumn)
                w = mFm.width( mChannel ) + (lv->itemMargin()*2);
        else if (column == mEssidColumn)
                w = mFm.width( mEssid ) + (lv->itemMargin()*2);
        else if (column == mAPColumn)
                w = mFm.width( mAP ) + (lv->itemMargin()*2);
        /*else if (column == mModeColumn)
                w = fm.width( mMode ) + (lv->itemMargin()*2);*/

        else
                w = 0;

        int headerw = fm.width( listView()->columnText(column) ) + (lv->itemMargin()*2);
        if (w < headerw)
                w = headerw;

        return w;
}

TQString NetListViewItem::key( int column, bool ascending ) const
{
        if (mConnected) {	// make sure that connected item is always 1st.
                if (ascending)
                        return "0";
                else
                        return "ZZZ";
        }
        TQString t = TQString();
        if (column == mQualityColumn) {
                t = TQString::number( mQuality );
                if (mQuality < 10)
                        t.prepend("0");
        } else if (column == mEncColumn) {
                if (mEnc)
                        t = "1";
                else
                        t="0";
        } else if (column == mEssidColumn) {
                t = mEssid.upper(); // add .upper() to make it case-insensitive;
        } else if (column == mChanColumn) {
                t = mChannel;
                if ( mChannel.length() == 1 )
                        t.prepend("0");
        } else if (column == mAPColumn) {
                t = mAP.upper();
                /*} else if (column == mModeColumn) {
                        t = mMode.upper();*/

        }


        return t;
}
