/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.apache.causeway.core.metamodel.facets.actions.action;

import java.util.stream.Collectors;

import jakarta.inject.Inject;

import org.apache.causeway.applib.Identifier;
import org.apache.causeway.applib.services.metamodel.BeanSort;
import org.apache.causeway.core.config.progmodel.ProgrammingModelConstants;
import org.apache.causeway.core.metamodel.context.MetaModelContext;
import org.apache.causeway.core.metamodel.spec.ObjectSpecification;
import org.apache.causeway.core.metamodel.spec.feature.MixedIn;
import org.apache.causeway.core.metamodel.spec.feature.ObjectAction;
import org.apache.causeway.core.metamodel.specloader.validator.MetaModelValidatorAbstract;
import org.apache.causeway.core.metamodel.specloader.validator.ValidationFailure;

public class ActionAnnotationShouldEnforceConcreteTypeToBeIncludedWithMetamodelValidator
extends MetaModelValidatorAbstract {

    @Inject
    public ActionAnnotationShouldEnforceConcreteTypeToBeIncludedWithMetamodelValidator(final MetaModelContext mmc) {
        super(mmc, spec->spec.getBeanSort() == BeanSort.UNKNOWN
                && !spec.isAbstract());
    }

    @Override
    public void validateObjectEnter(final ObjectSpecification spec) {

        var actions = spec.streamAnyActions(MixedIn.EXCLUDED).collect(Collectors.toList());

        final int numActions = actions.size();
        if (numActions > 0) {

            var actionIds = actions.stream()
            .map(ObjectAction::getFeatureIdentifier)
            .map(Identifier::toString)
            .collect(Collectors.joining(", "));

            ValidationFailure.raiseFormatted(
                    spec,
                    ProgrammingModelConstants.MessageTemplate.UNKNONW_SORT_WITH_ACTION
                        .builder()
                        .addVariable("type", spec.getCorrespondingClass().getName())
                        .addVariable("actions", actionIds)
                        .addVariable("actionCount", numActions)
                        .buildMessage());

        }

    }

}
