/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.sql.execution.datasources.geoparquet.internal;

import java.math.BigInteger;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.Arrays;
import org.apache.parquet.column.ColumnDescriptor;
import org.apache.parquet.column.Dictionary;
import org.apache.parquet.io.api.Binary;
import org.apache.parquet.schema.LogicalTypeAnnotation;
import org.apache.parquet.schema.PrimitiveType;
import org.apache.spark.sql.catalyst.util.DateTimeUtils;
import org.apache.spark.sql.catalyst.util.RebaseDateTime;
import org.apache.spark.sql.execution.datasources.geoparquet.internal.DataSourceUtils;
import org.apache.spark.sql.execution.datasources.geoparquet.internal.ParquetRowConverter;
import org.apache.spark.sql.execution.datasources.geoparquet.internal.ParquetVectorUpdater;
import org.apache.spark.sql.execution.datasources.geoparquet.internal.VectorizedValuesReader;
import org.apache.spark.sql.execution.vectorized.WritableColumnVector;
import org.apache.spark.sql.types.DataType;
import org.apache.spark.sql.types.DataTypes;
import org.apache.spark.sql.types.DayTimeIntervalType;
import org.apache.spark.sql.types.DecimalType;
import org.apache.spark.sql.types.YearMonthIntervalType;

public class ParquetVectorUpdaterFactory {
    private static final ZoneId UTC = ZoneOffset.UTC;
    private final LogicalTypeAnnotation logicalTypeAnnotation;
    private final ZoneId convertTz;
    private final String datetimeRebaseMode;
    private final String datetimeRebaseTz;
    private final String int96RebaseMode;
    private final String int96RebaseTz;

    ParquetVectorUpdaterFactory(LogicalTypeAnnotation logicalTypeAnnotation, ZoneId convertTz, String datetimeRebaseMode, String datetimeRebaseTz, String int96RebaseMode, String int96RebaseTz) {
        this.logicalTypeAnnotation = logicalTypeAnnotation;
        this.convertTz = convertTz;
        this.datetimeRebaseMode = datetimeRebaseMode;
        this.datetimeRebaseTz = datetimeRebaseTz;
        this.int96RebaseMode = int96RebaseMode;
        this.int96RebaseTz = int96RebaseTz;
    }

    public ParquetVectorUpdater getUpdater(ColumnDescriptor descriptor, DataType sparkType) {
        PrimitiveType.PrimitiveTypeName typeName = descriptor.getPrimitiveType().getPrimitiveTypeName();
        switch (typeName) {
            case BOOLEAN: {
                if (sparkType != DataTypes.BooleanType) break;
                return new BooleanUpdater();
            }
            case INT32: {
                if (sparkType == DataTypes.IntegerType || ParquetVectorUpdaterFactory.canReadAsIntDecimal(descriptor, sparkType)) {
                    return new IntegerUpdater();
                }
                if (sparkType == DataTypes.LongType && this.isUnsignedIntTypeMatched(32)) {
                    return new UnsignedIntegerUpdater();
                }
                if (sparkType == DataTypes.ByteType) {
                    return new ByteUpdater();
                }
                if (sparkType == DataTypes.ShortType) {
                    return new ShortUpdater();
                }
                if (sparkType == DataTypes.DateType) {
                    if ("CORRECTED".equals(this.datetimeRebaseMode)) {
                        return new IntegerUpdater();
                    }
                    boolean failIfRebase = "EXCEPTION".equals(this.datetimeRebaseMode);
                    return new IntegerWithRebaseUpdater(failIfRebase);
                }
                if (!(sparkType instanceof YearMonthIntervalType)) break;
                return new IntegerUpdater();
            }
            case INT64: {
                if (sparkType == DataTypes.LongType || ParquetVectorUpdaterFactory.canReadAsLongDecimal(descriptor, sparkType)) {
                    if (DecimalType.is32BitDecimalType((DataType)sparkType)) {
                        return new DowncastLongUpdater();
                    }
                    return new LongUpdater();
                }
                if (ParquetVectorUpdaterFactory.isLongDecimal(sparkType) && this.isUnsignedIntTypeMatched(64)) {
                    return new UnsignedLongUpdater();
                }
                if (sparkType == DataTypes.TimestampType && this.isTimestampTypeMatched(LogicalTypeAnnotation.TimeUnit.MICROS)) {
                    if ("CORRECTED".equals(this.datetimeRebaseMode)) {
                        return new LongUpdater();
                    }
                    boolean failIfRebase = "EXCEPTION".equals(this.datetimeRebaseMode);
                    return new LongWithRebaseUpdater(failIfRebase, this.datetimeRebaseTz);
                }
                if (sparkType == DataTypes.TimestampType && this.isTimestampTypeMatched(LogicalTypeAnnotation.TimeUnit.MILLIS)) {
                    if ("CORRECTED".equals(this.datetimeRebaseMode)) {
                        return new LongAsMicrosUpdater();
                    }
                    boolean failIfRebase = "EXCEPTION".equals(this.datetimeRebaseMode);
                    return new LongAsMicrosRebaseUpdater(failIfRebase, this.datetimeRebaseTz);
                }
                if (sparkType == DataTypes.TimestampNTZType && this.isTimestampTypeMatched(LogicalTypeAnnotation.TimeUnit.MICROS)) {
                    this.validateTimestampNTZType();
                    return new LongUpdater();
                }
                if (sparkType == DataTypes.TimestampNTZType && this.isTimestampTypeMatched(LogicalTypeAnnotation.TimeUnit.MILLIS)) {
                    this.validateTimestampNTZType();
                    return new LongAsMicrosUpdater();
                }
                if (!(sparkType instanceof DayTimeIntervalType)) break;
                return new LongUpdater();
            }
            case FLOAT: {
                if (sparkType != DataTypes.FloatType) break;
                return new FloatUpdater();
            }
            case DOUBLE: {
                if (sparkType != DataTypes.DoubleType) break;
                return new DoubleUpdater();
            }
            case INT96: {
                if (sparkType == DataTypes.TimestampNTZType) {
                    this.convertErrorForTimestampNTZ(typeName.name());
                    break;
                }
                if (sparkType != DataTypes.TimestampType) break;
                boolean failIfRebase = "EXCEPTION".equals(this.int96RebaseMode);
                if (!this.shouldConvertTimestamps()) {
                    if ("CORRECTED".equals(this.int96RebaseMode)) {
                        return new BinaryToSQLTimestampUpdater();
                    }
                    return new BinaryToSQLTimestampRebaseUpdater(failIfRebase, this.int96RebaseTz);
                }
                if ("CORRECTED".equals(this.int96RebaseMode)) {
                    return new BinaryToSQLTimestampConvertTzUpdater(this.convertTz);
                }
                return new BinaryToSQLTimestampConvertTzRebaseUpdater(failIfRebase, this.convertTz, this.int96RebaseTz);
            }
            case BINARY: {
                if (sparkType != DataTypes.StringType && sparkType != DataTypes.BinaryType && !ParquetVectorUpdaterFactory.canReadAsBinaryDecimal(descriptor, sparkType)) break;
                return new BinaryUpdater();
            }
            case FIXED_LEN_BYTE_ARRAY: {
                int arrayLen = descriptor.getPrimitiveType().getTypeLength();
                if (ParquetVectorUpdaterFactory.canReadAsIntDecimal(descriptor, sparkType)) {
                    return new FixedLenByteArrayAsIntUpdater(arrayLen);
                }
                if (ParquetVectorUpdaterFactory.canReadAsLongDecimal(descriptor, sparkType)) {
                    return new FixedLenByteArrayAsLongUpdater(arrayLen);
                }
                if (ParquetVectorUpdaterFactory.canReadAsBinaryDecimal(descriptor, sparkType)) {
                    return new FixedLenByteArrayUpdater(arrayLen);
                }
                if (sparkType != DataTypes.BinaryType) break;
                return new FixedLenByteArrayUpdater(arrayLen);
            }
        }
        throw this.constructConvertNotSupportedException(descriptor, sparkType);
    }

    boolean isTimestampTypeMatched(LogicalTypeAnnotation.TimeUnit unit) {
        return this.logicalTypeAnnotation instanceof LogicalTypeAnnotation.TimestampLogicalTypeAnnotation && ((LogicalTypeAnnotation.TimestampLogicalTypeAnnotation)this.logicalTypeAnnotation).getUnit() == unit;
    }

    private void validateTimestampNTZType() {
        assert (this.logicalTypeAnnotation instanceof LogicalTypeAnnotation.TimestampLogicalTypeAnnotation);
        if (((LogicalTypeAnnotation.TimestampLogicalTypeAnnotation)this.logicalTypeAnnotation).isAdjustedToUTC()) {
            this.convertErrorForTimestampNTZ("int64 time(" + this.logicalTypeAnnotation + ")");
        }
    }

    void convertErrorForTimestampNTZ(String parquetType) {
        throw new RuntimeException("Unable to create Parquet converter for data type " + DataTypes.TimestampNTZType.json() + " whose Parquet type is " + parquetType);
    }

    boolean isUnsignedIntTypeMatched(int bitWidth) {
        return this.logicalTypeAnnotation instanceof LogicalTypeAnnotation.IntLogicalTypeAnnotation && !((LogicalTypeAnnotation.IntLogicalTypeAnnotation)this.logicalTypeAnnotation).isSigned() && ((LogicalTypeAnnotation.IntLogicalTypeAnnotation)this.logicalTypeAnnotation).getBitWidth() == bitWidth;
    }

    private static int rebaseDays(int julianDays, boolean failIfRebase) {
        if (failIfRebase) {
            if (julianDays < RebaseDateTime.lastSwitchJulianDay()) {
                throw DataSourceUtils.newRebaseExceptionInRead("Parquet");
            }
            return julianDays;
        }
        return RebaseDateTime.rebaseJulianToGregorianDays((int)julianDays);
    }

    private static long rebaseTimestamp(long julianMicros, boolean failIfRebase, String format, String timeZone) {
        if (failIfRebase) {
            if (julianMicros < RebaseDateTime.lastSwitchJulianTs()) {
                throw DataSourceUtils.newRebaseExceptionInRead(format);
            }
            return julianMicros;
        }
        return RebaseDateTime.rebaseJulianToGregorianMicros((String)timeZone, (long)julianMicros);
    }

    private static long rebaseMicros(long julianMicros, boolean failIfRebase, String timeZone) {
        return ParquetVectorUpdaterFactory.rebaseTimestamp(julianMicros, failIfRebase, "Parquet", timeZone);
    }

    private static long rebaseInt96(long julianMicros, boolean failIfRebase, String timeZone) {
        return ParquetVectorUpdaterFactory.rebaseTimestamp(julianMicros, failIfRebase, "Parquet INT96", timeZone);
    }

    private boolean shouldConvertTimestamps() {
        return this.convertTz != null && !this.convertTz.equals(UTC);
    }

    private RuntimeException constructConvertNotSupportedException(ColumnDescriptor descriptor, DataType sparkType) {
        return new RuntimeException("Parquet schema mismatch: " + Arrays.toString(descriptor.getPath()) + "\n" + descriptor.getPrimitiveType().getPrimitiveTypeName().toString() + " and " + sparkType.catalogString());
    }

    private static boolean canReadAsIntDecimal(ColumnDescriptor descriptor, DataType dt) {
        if (!DecimalType.is32BitDecimalType((DataType)dt)) {
            return false;
        }
        return ParquetVectorUpdaterFactory.isDecimalTypeMatched(descriptor, dt);
    }

    private static boolean canReadAsLongDecimal(ColumnDescriptor descriptor, DataType dt) {
        if (!DecimalType.is64BitDecimalType((DataType)dt)) {
            return false;
        }
        return ParquetVectorUpdaterFactory.isDecimalTypeMatched(descriptor, dt);
    }

    private static boolean canReadAsBinaryDecimal(ColumnDescriptor descriptor, DataType dt) {
        if (!DecimalType.isByteArrayDecimalType((DataType)dt)) {
            return false;
        }
        return ParquetVectorUpdaterFactory.isDecimalTypeMatched(descriptor, dt);
    }

    private static boolean isLongDecimal(DataType dt) {
        if (dt instanceof DecimalType) {
            DecimalType d = (DecimalType)dt;
            return d.precision() == 20 && d.scale() == 0;
        }
        return false;
    }

    private static boolean isDecimalTypeMatched(ColumnDescriptor descriptor, DataType dt) {
        DecimalType d = (DecimalType)dt;
        LogicalTypeAnnotation typeAnnotation = descriptor.getPrimitiveType().getLogicalTypeAnnotation();
        if (typeAnnotation instanceof LogicalTypeAnnotation.DecimalLogicalTypeAnnotation) {
            LogicalTypeAnnotation.DecimalLogicalTypeAnnotation decimalType = (LogicalTypeAnnotation.DecimalLogicalTypeAnnotation)typeAnnotation;
            return decimalType.getPrecision() <= d.precision() && decimalType.getScale() == d.scale();
        }
        return false;
    }

    private static class BooleanUpdater
    implements ParquetVectorUpdater {
        private BooleanUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readBooleans(total, values, offset);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipBooleans(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putBoolean(offset, valuesReader.readBoolean());
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            throw new UnsupportedOperationException("Boolean is not supported");
        }
    }

    static class IntegerUpdater
    implements ParquetVectorUpdater {
        IntegerUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readIntegers(total, values, offset);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipIntegers(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putInt(offset, valuesReader.readInteger());
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            values.putInt(offset, dictionary.decodeToInt(dictionaryIds.getDictId(offset)));
        }
    }

    private static class UnsignedIntegerUpdater
    implements ParquetVectorUpdater {
        private UnsignedIntegerUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readUnsignedIntegers(total, values, offset);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipIntegers(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putLong(offset, Integer.toUnsignedLong(valuesReader.readInteger()));
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            values.putLong(offset, Integer.toUnsignedLong(dictionary.decodeToInt(dictionaryIds.getDictId(offset))));
        }
    }

    private static class ByteUpdater
    implements ParquetVectorUpdater {
        private ByteUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readBytes(total, values, offset);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipBytes(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putByte(offset, valuesReader.readByte());
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            values.putByte(offset, (byte)dictionary.decodeToInt(dictionaryIds.getDictId(offset)));
        }
    }

    private static class ShortUpdater
    implements ParquetVectorUpdater {
        private ShortUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readShorts(total, values, offset);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipShorts(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putShort(offset, valuesReader.readShort());
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            values.putShort(offset, (short)dictionary.decodeToInt(dictionaryIds.getDictId(offset)));
        }
    }

    private static class IntegerWithRebaseUpdater
    implements ParquetVectorUpdater {
        private final boolean failIfRebase;

        IntegerWithRebaseUpdater(boolean failIfRebase) {
            this.failIfRebase = failIfRebase;
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readIntegersWithRebase(total, values, offset, this.failIfRebase);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipIntegers(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            int julianDays = valuesReader.readInteger();
            values.putInt(offset, ParquetVectorUpdaterFactory.rebaseDays(julianDays, this.failIfRebase));
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            int julianDays = dictionary.decodeToInt(dictionaryIds.getDictId(offset));
            values.putInt(offset, ParquetVectorUpdaterFactory.rebaseDays(julianDays, this.failIfRebase));
        }
    }

    private static class DowncastLongUpdater
    implements ParquetVectorUpdater {
        private DowncastLongUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            for (int i = 0; i < total; ++i) {
                values.putInt(offset + i, (int)valuesReader.readLong());
            }
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipLongs(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putInt(offset, (int)valuesReader.readLong());
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            values.putLong(offset, dictionary.decodeToLong(dictionaryIds.getDictId(offset)));
        }
    }

    private static class LongUpdater
    implements ParquetVectorUpdater {
        private LongUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readLongs(total, values, offset);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipLongs(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putLong(offset, valuesReader.readLong());
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            values.putLong(offset, dictionary.decodeToLong(dictionaryIds.getDictId(offset)));
        }
    }

    private static class UnsignedLongUpdater
    implements ParquetVectorUpdater {
        private UnsignedLongUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readUnsignedLongs(total, values, offset);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipLongs(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            byte[] bytes = new BigInteger(Long.toUnsignedString(valuesReader.readLong())).toByteArray();
            values.putByteArray(offset, bytes);
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            long signed = dictionary.decodeToLong(dictionaryIds.getDictId(offset));
            byte[] unsigned = new BigInteger(Long.toUnsignedString(signed)).toByteArray();
            values.putByteArray(offset, unsigned);
        }
    }

    private static class LongWithRebaseUpdater
    implements ParquetVectorUpdater {
        private final boolean failIfRebase;
        private final String timeZone;

        LongWithRebaseUpdater(boolean failIfRebase, String timeZone) {
            this.failIfRebase = failIfRebase;
            this.timeZone = timeZone;
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readLongsWithRebase(total, values, offset, this.failIfRebase, this.timeZone);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipLongs(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            long julianMicros = valuesReader.readLong();
            values.putLong(offset, ParquetVectorUpdaterFactory.rebaseMicros(julianMicros, this.failIfRebase, this.timeZone));
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            long julianMicros = dictionary.decodeToLong(dictionaryIds.getDictId(offset));
            values.putLong(offset, ParquetVectorUpdaterFactory.rebaseMicros(julianMicros, this.failIfRebase, this.timeZone));
        }
    }

    private static class LongAsMicrosUpdater
    implements ParquetVectorUpdater {
        private LongAsMicrosUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            for (int i = 0; i < total; ++i) {
                this.readValue(offset + i, values, valuesReader);
            }
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipLongs(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putLong(offset, DateTimeUtils.millisToMicros((long)valuesReader.readLong()));
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            long gregorianMillis = dictionary.decodeToLong(dictionaryIds.getDictId(offset));
            values.putLong(offset, DateTimeUtils.millisToMicros((long)gregorianMillis));
        }
    }

    private static class LongAsMicrosRebaseUpdater
    implements ParquetVectorUpdater {
        private final boolean failIfRebase;
        private final String timeZone;

        LongAsMicrosRebaseUpdater(boolean failIfRebase, String timeZone) {
            this.failIfRebase = failIfRebase;
            this.timeZone = timeZone;
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            for (int i = 0; i < total; ++i) {
                this.readValue(offset + i, values, valuesReader);
            }
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipLongs(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            long julianMicros = DateTimeUtils.millisToMicros((long)valuesReader.readLong());
            values.putLong(offset, ParquetVectorUpdaterFactory.rebaseMicros(julianMicros, this.failIfRebase, this.timeZone));
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            long julianMillis = dictionary.decodeToLong(dictionaryIds.getDictId(offset));
            long julianMicros = DateTimeUtils.millisToMicros((long)julianMillis);
            values.putLong(offset, ParquetVectorUpdaterFactory.rebaseMicros(julianMicros, this.failIfRebase, this.timeZone));
        }
    }

    private static class FloatUpdater
    implements ParquetVectorUpdater {
        private FloatUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readFloats(total, values, offset);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipFloats(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putFloat(offset, valuesReader.readFloat());
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            values.putFloat(offset, dictionary.decodeToFloat(dictionaryIds.getDictId(offset)));
        }
    }

    private static class DoubleUpdater
    implements ParquetVectorUpdater {
        private DoubleUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readDoubles(total, values, offset);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipDoubles(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putDouble(offset, valuesReader.readDouble());
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            values.putDouble(offset, dictionary.decodeToDouble(dictionaryIds.getDictId(offset)));
        }
    }

    private static class BinaryToSQLTimestampUpdater
    implements ParquetVectorUpdater {
        private BinaryToSQLTimestampUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            for (int i = 0; i < total; ++i) {
                this.readValue(offset + i, values, valuesReader);
            }
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipFixedLenByteArray(total, 12);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            long gregorianMicros = ParquetRowConverter.binaryToSQLTimestamp(valuesReader.readBinary(12));
            values.putLong(offset, gregorianMicros);
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            Binary v = dictionary.decodeToBinary(dictionaryIds.getDictId(offset));
            values.putLong(offset, ParquetRowConverter.binaryToSQLTimestamp(v));
        }
    }

    private static class BinaryToSQLTimestampRebaseUpdater
    implements ParquetVectorUpdater {
        private final boolean failIfRebase;
        private final String timeZone;

        BinaryToSQLTimestampRebaseUpdater(boolean failIfRebase, String timeZone) {
            this.failIfRebase = failIfRebase;
            this.timeZone = timeZone;
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            for (int i = 0; i < total; ++i) {
                this.readValue(offset + i, values, valuesReader);
            }
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipFixedLenByteArray(total, 12);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            long julianMicros = ParquetRowConverter.binaryToSQLTimestamp(valuesReader.readBinary(12));
            long gregorianMicros = ParquetVectorUpdaterFactory.rebaseInt96(julianMicros, this.failIfRebase, this.timeZone);
            values.putLong(offset, gregorianMicros);
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            Binary v = dictionary.decodeToBinary(dictionaryIds.getDictId(offset));
            long julianMicros = ParquetRowConverter.binaryToSQLTimestamp(v);
            long gregorianMicros = ParquetVectorUpdaterFactory.rebaseInt96(julianMicros, this.failIfRebase, this.timeZone);
            values.putLong(offset, gregorianMicros);
        }
    }

    private static class BinaryToSQLTimestampConvertTzUpdater
    implements ParquetVectorUpdater {
        private final ZoneId convertTz;

        BinaryToSQLTimestampConvertTzUpdater(ZoneId convertTz) {
            this.convertTz = convertTz;
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            for (int i = 0; i < total; ++i) {
                this.readValue(offset + i, values, valuesReader);
            }
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipFixedLenByteArray(total, 12);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            long gregorianMicros = ParquetRowConverter.binaryToSQLTimestamp(valuesReader.readBinary(12));
            long adjTime = DateTimeUtils.convertTz((long)gregorianMicros, (ZoneId)this.convertTz, (ZoneId)UTC);
            values.putLong(offset, adjTime);
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            Binary v = dictionary.decodeToBinary(dictionaryIds.getDictId(offset));
            long gregorianMicros = ParquetRowConverter.binaryToSQLTimestamp(v);
            long adjTime = DateTimeUtils.convertTz((long)gregorianMicros, (ZoneId)this.convertTz, (ZoneId)UTC);
            values.putLong(offset, adjTime);
        }
    }

    private static class BinaryToSQLTimestampConvertTzRebaseUpdater
    implements ParquetVectorUpdater {
        private final boolean failIfRebase;
        private final ZoneId convertTz;
        private final String timeZone;

        BinaryToSQLTimestampConvertTzRebaseUpdater(boolean failIfRebase, ZoneId convertTz, String timeZone) {
            this.failIfRebase = failIfRebase;
            this.convertTz = convertTz;
            this.timeZone = timeZone;
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            for (int i = 0; i < total; ++i) {
                this.readValue(offset + i, values, valuesReader);
            }
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipFixedLenByteArray(total, 12);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            long julianMicros = ParquetRowConverter.binaryToSQLTimestamp(valuesReader.readBinary(12));
            long gregorianMicros = ParquetVectorUpdaterFactory.rebaseInt96(julianMicros, this.failIfRebase, this.timeZone);
            long adjTime = DateTimeUtils.convertTz((long)gregorianMicros, (ZoneId)this.convertTz, (ZoneId)UTC);
            values.putLong(offset, adjTime);
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            Binary v = dictionary.decodeToBinary(dictionaryIds.getDictId(offset));
            long julianMicros = ParquetRowConverter.binaryToSQLTimestamp(v);
            long gregorianMicros = ParquetVectorUpdaterFactory.rebaseInt96(julianMicros, this.failIfRebase, this.timeZone);
            long adjTime = DateTimeUtils.convertTz((long)gregorianMicros, (ZoneId)this.convertTz, (ZoneId)UTC);
            values.putLong(offset, adjTime);
        }
    }

    private static class BinaryUpdater
    implements ParquetVectorUpdater {
        private BinaryUpdater() {
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readBinary(total, values, offset);
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipBinary(total);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            valuesReader.readBinary(1, values, offset);
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            Binary v = dictionary.decodeToBinary(dictionaryIds.getDictId(offset));
            values.putByteArray(offset, v.getBytes());
        }
    }

    private static class FixedLenByteArrayAsIntUpdater
    implements ParquetVectorUpdater {
        private final int arrayLen;

        FixedLenByteArrayAsIntUpdater(int arrayLen) {
            this.arrayLen = arrayLen;
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            for (int i = 0; i < total; ++i) {
                this.readValue(offset + i, values, valuesReader);
            }
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipFixedLenByteArray(total, this.arrayLen);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            int value = (int)ParquetRowConverter.binaryToUnscaledLong(valuesReader.readBinary(this.arrayLen));
            values.putInt(offset, value);
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            Binary v = dictionary.decodeToBinary(dictionaryIds.getDictId(offset));
            values.putInt(offset, (int)ParquetRowConverter.binaryToUnscaledLong(v));
        }
    }

    private static class FixedLenByteArrayAsLongUpdater
    implements ParquetVectorUpdater {
        private final int arrayLen;

        FixedLenByteArrayAsLongUpdater(int arrayLen) {
            this.arrayLen = arrayLen;
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            for (int i = 0; i < total; ++i) {
                this.readValue(offset + i, values, valuesReader);
            }
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipFixedLenByteArray(total, this.arrayLen);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            long value = ParquetRowConverter.binaryToUnscaledLong(valuesReader.readBinary(this.arrayLen));
            values.putLong(offset, value);
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            Binary v = dictionary.decodeToBinary(dictionaryIds.getDictId(offset));
            values.putLong(offset, ParquetRowConverter.binaryToUnscaledLong(v));
        }
    }

    private static class FixedLenByteArrayUpdater
    implements ParquetVectorUpdater {
        private final int arrayLen;

        FixedLenByteArrayUpdater(int arrayLen) {
            this.arrayLen = arrayLen;
        }

        @Override
        public void readValues(int total, int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            for (int i = 0; i < total; ++i) {
                this.readValue(offset + i, values, valuesReader);
            }
        }

        @Override
        public void skipValues(int total, VectorizedValuesReader valuesReader) {
            valuesReader.skipFixedLenByteArray(total, this.arrayLen);
        }

        @Override
        public void readValue(int offset, WritableColumnVector values, VectorizedValuesReader valuesReader) {
            values.putByteArray(offset, valuesReader.readBinary(this.arrayLen).getBytes());
        }

        @Override
        public void decodeSingleDictionaryId(int offset, WritableColumnVector values, WritableColumnVector dictionaryIds, Dictionary dictionary) {
            Binary v = dictionary.decodeToBinary(dictionaryIds.getDictId(offset));
            values.putByteArray(offset, v.getBytes());
        }
    }
}

