﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Array.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/stream/ResponseStream.h>
#include <aws/datazone/DataZone_EXPORTS.h>
#include <aws/datazone/model/LineageEventProcessingStatus.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace DataZone {
namespace Model {
class GetLineageEventResult {
 public:
  AWS_DATAZONE_API GetLineageEventResult() = default;
  AWS_DATAZONE_API GetLineageEventResult(GetLineageEventResult&&) = default;
  AWS_DATAZONE_API GetLineageEventResult& operator=(GetLineageEventResult&&) = default;
  // we delete these because Microsoft doesn't handle move generation correctly
  // and we therefore don't trust them to get it right here either.
  GetLineageEventResult(const GetLineageEventResult&) = delete;
  GetLineageEventResult& operator=(const GetLineageEventResult&) = delete;

  AWS_DATAZONE_API GetLineageEventResult(Aws::AmazonWebServiceResult<Aws::Utils::Stream::ResponseStream>&& result);
  AWS_DATAZONE_API GetLineageEventResult& operator=(Aws::AmazonWebServiceResult<Aws::Utils::Stream::ResponseStream>&& result);

  ///@{
  /**
   * <p>The timestamp of when the lineage event was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  void SetCreatedAt(CreatedAtT&& value) {
    m_createdAtHasBeenSet = true;
    m_createdAt = std::forward<CreatedAtT>(value);
  }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  GetLineageEventResult& WithCreatedAt(CreatedAtT&& value) {
    SetCreatedAt(std::forward<CreatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user who created the lineage event.</p>
   */
  inline const Aws::String& GetCreatedBy() const { return m_createdBy; }
  template <typename CreatedByT = Aws::String>
  void SetCreatedBy(CreatedByT&& value) {
    m_createdByHasBeenSet = true;
    m_createdBy = std::forward<CreatedByT>(value);
  }
  template <typename CreatedByT = Aws::String>
  GetLineageEventResult& WithCreatedBy(CreatedByT&& value) {
    SetCreatedBy(std::forward<CreatedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the domain.</p>
   */
  inline const Aws::String& GetDomainId() const { return m_domainId; }
  template <typename DomainIdT = Aws::String>
  void SetDomainId(DomainIdT&& value) {
    m_domainIdHasBeenSet = true;
    m_domainId = std::forward<DomainIdT>(value);
  }
  template <typename DomainIdT = Aws::String>
  GetLineageEventResult& WithDomainId(DomainIdT&& value) {
    SetDomainId(std::forward<DomainIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The lineage event details.</p>
   */
  inline Aws::IOStream& GetEvent() const { return m_event.GetUnderlyingStream(); }
  inline void ReplaceBody(Aws::IOStream* body) { m_event = Aws::Utils::Stream::ResponseStream(body); }

  ///@}

  ///@{
  /**
   * <p>The time of the lineage event.</p>
   */
  inline const Aws::Utils::DateTime& GetEventTime() const { return m_eventTime; }
  template <typename EventTimeT = Aws::Utils::DateTime>
  void SetEventTime(EventTimeT&& value) {
    m_eventTimeHasBeenSet = true;
    m_eventTime = std::forward<EventTimeT>(value);
  }
  template <typename EventTimeT = Aws::Utils::DateTime>
  GetLineageEventResult& WithEventTime(EventTimeT&& value) {
    SetEventTime(std::forward<EventTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the lineage event.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  GetLineageEventResult& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The progressing status of the lineage event.</p>
   */
  inline LineageEventProcessingStatus GetProcessingStatus() const { return m_processingStatus; }
  inline void SetProcessingStatus(LineageEventProcessingStatus value) {
    m_processingStatusHasBeenSet = true;
    m_processingStatus = value;
  }
  inline GetLineageEventResult& WithProcessingStatus(LineageEventProcessingStatus value) {
    SetProcessingStatus(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  GetLineageEventResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Utils::DateTime m_createdAt{};
  bool m_createdAtHasBeenSet = false;

  Aws::String m_createdBy;
  bool m_createdByHasBeenSet = false;

  Aws::String m_domainId;
  bool m_domainIdHasBeenSet = false;

  Aws::Utils::Stream::ResponseStream m_event{};
  bool m_eventHasBeenSet = false;

  Aws::Utils::DateTime m_eventTime{};
  bool m_eventTimeHasBeenSet = false;

  Aws::String m_id;
  bool m_idHasBeenSet = false;

  LineageEventProcessingStatus m_processingStatus{LineageEventProcessingStatus::NOT_SET};
  bool m_processingStatusHasBeenSet = false;

  Aws::String m_requestId;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataZone
}  // namespace Aws
