﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/swf/SWFRequest.h>
#include <aws/swf/SWF_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SWF {
namespace Model {

/**
 */
class RequestCancelWorkflowExecutionRequest : public SWFRequest {
 public:
  AWS_SWF_API RequestCancelWorkflowExecutionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RequestCancelWorkflowExecution"; }

  AWS_SWF_API Aws::String SerializePayload() const override;

  AWS_SWF_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the domain containing the workflow execution to cancel.</p>
   */
  inline const Aws::String& GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  template <typename DomainT = Aws::String>
  void SetDomain(DomainT&& value) {
    m_domainHasBeenSet = true;
    m_domain = std::forward<DomainT>(value);
  }
  template <typename DomainT = Aws::String>
  RequestCancelWorkflowExecutionRequest& WithDomain(DomainT&& value) {
    SetDomain(std::forward<DomainT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The workflowId of the workflow execution to cancel.</p>
   */
  inline const Aws::String& GetWorkflowId() const { return m_workflowId; }
  inline bool WorkflowIdHasBeenSet() const { return m_workflowIdHasBeenSet; }
  template <typename WorkflowIdT = Aws::String>
  void SetWorkflowId(WorkflowIdT&& value) {
    m_workflowIdHasBeenSet = true;
    m_workflowId = std::forward<WorkflowIdT>(value);
  }
  template <typename WorkflowIdT = Aws::String>
  RequestCancelWorkflowExecutionRequest& WithWorkflowId(WorkflowIdT&& value) {
    SetWorkflowId(std::forward<WorkflowIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The runId of the workflow execution to cancel.</p>
   */
  inline const Aws::String& GetRunId() const { return m_runId; }
  inline bool RunIdHasBeenSet() const { return m_runIdHasBeenSet; }
  template <typename RunIdT = Aws::String>
  void SetRunId(RunIdT&& value) {
    m_runIdHasBeenSet = true;
    m_runId = std::forward<RunIdT>(value);
  }
  template <typename RunIdT = Aws::String>
  RequestCancelWorkflowExecutionRequest& WithRunId(RunIdT&& value) {
    SetRunId(std::forward<RunIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domain;
  bool m_domainHasBeenSet = false;

  Aws::String m_workflowId;
  bool m_workflowIdHasBeenSet = false;

  Aws::String m_runId;
  bool m_runIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace SWF
}  // namespace Aws
