﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Provides authorization for Amazon to bring an Autonomous System Number (ASN)
 * to a specific Amazon Web Services account using bring your own ASN (BYOASN). For
 * details on the format of the message and signature, see <a
 * href="https://docs.aws.amazon.com/vpc/latest/ipam/tutorials-byoasn.html">Tutorial:
 * Bring your ASN to IPAM</a> in the <i>Amazon VPC IPAM guide</i>.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/AsnAuthorizationContext">AWS
 * API Reference</a></p>
 */
class AsnAuthorizationContext {
 public:
  AWS_EC2_API AsnAuthorizationContext() = default;
  AWS_EC2_API AsnAuthorizationContext(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API AsnAuthorizationContext& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The authorization context's message.</p>
   */
  inline const Aws::String& GetMessage() const { return m_message; }
  inline bool MessageHasBeenSet() const { return m_messageHasBeenSet; }
  template <typename MessageT = Aws::String>
  void SetMessage(MessageT&& value) {
    m_messageHasBeenSet = true;
    m_message = std::forward<MessageT>(value);
  }
  template <typename MessageT = Aws::String>
  AsnAuthorizationContext& WithMessage(MessageT&& value) {
    SetMessage(std::forward<MessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The authorization context's signature.</p>
   */
  inline const Aws::String& GetSignature() const { return m_signature; }
  inline bool SignatureHasBeenSet() const { return m_signatureHasBeenSet; }
  template <typename SignatureT = Aws::String>
  void SetSignature(SignatureT&& value) {
    m_signatureHasBeenSet = true;
    m_signature = std::forward<SignatureT>(value);
  }
  template <typename SignatureT = Aws::String>
  AsnAuthorizationContext& WithSignature(SignatureT&& value) {
    SetSignature(std::forward<SignatureT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_message;

  Aws::String m_signature;
  bool m_messageHasBeenSet = false;
  bool m_signatureHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
