﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/logs/CloudWatchLogsRequest.h>
#include <aws/logs/CloudWatchLogs_EXPORTS.h>
#include <aws/logs/model/SuppressionState.h>

#include <utility>

namespace Aws {
namespace CloudWatchLogs {
namespace Model {

/**
 */
class ListAnomaliesRequest : public CloudWatchLogsRequest {
 public:
  AWS_CLOUDWATCHLOGS_API ListAnomaliesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAnomalies"; }

  AWS_CLOUDWATCHLOGS_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHLOGS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Use this to optionally limit the results to only the anomalies found by a
   * certain anomaly detector.</p>
   */
  inline const Aws::String& GetAnomalyDetectorArn() const { return m_anomalyDetectorArn; }
  inline bool AnomalyDetectorArnHasBeenSet() const { return m_anomalyDetectorArnHasBeenSet; }
  template <typename AnomalyDetectorArnT = Aws::String>
  void SetAnomalyDetectorArn(AnomalyDetectorArnT&& value) {
    m_anomalyDetectorArnHasBeenSet = true;
    m_anomalyDetectorArn = std::forward<AnomalyDetectorArnT>(value);
  }
  template <typename AnomalyDetectorArnT = Aws::String>
  ListAnomaliesRequest& WithAnomalyDetectorArn(AnomalyDetectorArnT&& value) {
    SetAnomalyDetectorArn(std::forward<AnomalyDetectorArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>You can specify this parameter if you want to the operation to return only
   * anomalies that are currently either suppressed or unsuppressed.</p>
   */
  inline SuppressionState GetSuppressionState() const { return m_suppressionState; }
  inline bool SuppressionStateHasBeenSet() const { return m_suppressionStateHasBeenSet; }
  inline void SetSuppressionState(SuppressionState value) {
    m_suppressionStateHasBeenSet = true;
    m_suppressionState = value;
  }
  inline ListAnomaliesRequest& WithSuppressionState(SuppressionState value) {
    SetSuppressionState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to return. If you don't specify a value, the
   * default maximum value of 50 items is used.</p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline ListAnomaliesRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAnomaliesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_anomalyDetectorArn;

  SuppressionState m_suppressionState{SuppressionState::NOT_SET};

  int m_limit{0};

  Aws::String m_nextToken;
  bool m_anomalyDetectorArnHasBeenSet = false;
  bool m_suppressionStateHasBeenSet = false;
  bool m_limitHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchLogs
}  // namespace Aws
