"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystemProvider = void 0;
const promises_1 = __importDefault(require("fs/promises"));
const vscode_css_languageservice_1 = require("vscode-css-languageservice");
const utils_1 = require("../utils");
class FileSystemProvider {
    constructor() {
        this.readDirectory = this.readDirectory.bind(this);
        this.stat = this.stat.bind(this);
    }
    async stat(uri) {
        const path = typeof uri === 'string' ? (0, utils_1.urlToPath)(uri) : uri.fsPath;
        if (!path) {
            return this.unknownStat();
        }
        let stat;
        try {
            stat = await promises_1.default.stat(path);
        }
        catch (error) {
            if (error != null &&
                typeof error === 'object' &&
                'code' in error &&
                error.code === 'ENOENT') {
                return {
                    type: vscode_css_languageservice_1.FileType.Unknown,
                    ctime: -1,
                    mtime: -1,
                    size: -1
                };
            }
            throw error;
        }
        return {
            ctime: stat.ctimeMs,
            mtime: stat.mtimeMs,
            size: stat.size,
            type: this.getFileType(stat)
        };
    }
    unknownStat() {
        return {
            type: vscode_css_languageservice_1.FileType.Unknown,
            ctime: -1,
            mtime: -1,
            size: -1
        };
    }
    getFileType(stat) {
        return stat.isDirectory()
            ? vscode_css_languageservice_1.FileType.Directory
            : stat.isFile()
                ? vscode_css_languageservice_1.FileType.File
                : stat.isSymbolicLink()
                    ? vscode_css_languageservice_1.FileType.SymbolicLink
                    : vscode_css_languageservice_1.FileType.Unknown;
    }
    async readDirectory(uri) {
        const path = (0, utils_1.urlToPath)(uri);
        if (!path) {
            return [];
        }
        const files = await promises_1.default.readdir(path, {
            withFileTypes: true
        });
        return files.map((file) => [file.name, this.getFileType(file)]);
    }
    async readFile(uri) {
        const path = uri.fsPath;
        const data = await promises_1.default.readFile(path);
        return new Uint8Array(data);
    }
}
exports.FileSystemProvider = FileSystemProvider;
//# sourceMappingURL=FileSystemProvider.js.map