% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BuyseTest.R
\name{BuyseTest}
\alias{BuyseTest}
\title{Two-group GPC}
\usage{
BuyseTest(
  formula,
  data,
  scoring.rule = NULL,
  pool.strata = NULL,
  correction.uninf = NULL,
  model.tte = NULL,
  method.inference = NULL,
  n.resampling = NULL,
  strata.resampling = NULL,
  hierarchical = NULL,
  weightEndpoint = NULL,
  weightObs = NULL,
  neutral.as.uninf = NULL,
  add.halfNeutral = NULL,
  keep.pairScore = NULL,
  seed = NULL,
  cpus = NULL,
  trace = NULL,
  treatment = NULL,
  endpoint = NULL,
  type = NULL,
  threshold = NULL,
  status = NULL,
  operator = NULL,
  censoring = NULL,
  restriction = NULL,
  strata = NULL
)
}
\arguments{
\item{formula}{[formula] a symbolic description of the GPC model,
typically \code{type1(endpoint1) + type2(endpoint2, threshold2) ~ treatment + strata(gender)}.
See Details, section "Specification of the GPC model".}

\item{data}{[data.frame] dataset.}

\item{scoring.rule}{[character] method used to compare the observations of a pair in presence of right censoring (i.e. \code{"timeToEvent"} endpoints).
Can be \code{"Gehan"}, \code{"Peron"}, or \code{"Efron"}.
See Details, section "Handling missing values".}

\item{pool.strata}{[character] weights used to combine estimates across strata. Can be
\code{"Buyse"} to weight proportionally to the number of pairs in the strata,
\code{"CMH"} to weight proportionally to the ratio between the number of pairs in the strata and the number of observations in the strata.
\code{"equal"} to weight equally each strata,
\code{"standardisation"} to recover a marginal comparison,
or \code{"var-netBenefit"} to weight each strata proportionally to the precision of its estimated net benefit (similar syntax for the win ratio: \code{"var-winRatio"})}

\item{correction.uninf}{[integer] should a correction be applied to remove the bias due to the presence of uninformative pairs?
0 indicates no correction, 1 impute the average score of the informative pairs, and 2 performs IPCW.
See Details, section "Handling missing values".}

\item{model.tte}{[list] optional survival models relative to each time to each time to event endpoint.
Models must \code{prodlim} objects and stratified on the treatment and strata variable. When used, the uncertainty from the estimates of these survival models is ignored.}

\item{method.inference}{[character] method used to compute confidence intervals and p-values.
Can be \code{"none"}, \code{"u-statistic"}, \code{"permutation"}, \code{"studentized permutation"}, \code{"bootstrap"}, \code{"studentized bootstrap"}, \code{"varExact permutation"}.
See Details, section "Statistical inference".}

\item{n.resampling}{[integer] the number of permutations/samples used for computing the confidence intervals and the p.values. 
See Details, section "Statistical inference".}

\item{strata.resampling}{[character] the variable on which the permutation/sampling should be stratified. 
See Details, section "Statistical inference".}

\item{hierarchical}{[logical] should only the uninformative pairs be analyzed at the lower priority endpoints (hierarchical GPC)?
Otherwise all pairs will be compaired for all endpoint (full GPC).}

\item{weightEndpoint}{[numeric vector] weights used to cumulating the pairwise scores over the endpoints.
Only used when \code{hierarchical=FALSE}. Disregarded if the argument \code{formula} is defined.}

\item{weightObs}{[character or numeric vector] weights or variable in the dataset containing the weight associated to each observation.
These weights are only considered when performing GPC (but not when fitting surival models).}

\item{neutral.as.uninf}{[logical vector] should pairs classified as neutral be re-analyzed using endpoints of lower priority (as it is done for uninformative pairs).
See Details, section "Handling missing values".}

\item{add.halfNeutral}{[logical] should half of the neutral score be added to the favorable and unfavorable scores?}

\item{keep.pairScore}{[logical] should the result of each pairwise comparison be kept?}

\item{seed}{[integer, >0] Random number generator (RNG) state used when starting resampling.
If \code{NULL} no state is set.}

\item{cpus}{[integer, >0] the number of CPU to use.
Only the permutation test can use parallel computation.
See Details, section "Statistical inference".}

\item{trace}{[integer] should the execution of the function be traced ? \code{0} remains silent
and \code{1}-\code{3} correspond to a more and more verbose output in the console.}

\item{treatment, endpoint, type, threshold, status, operator, censoring, restriction, strata}{Alternative to \code{formula} for describing the GPC model.
See Details, section "Specification of the GPC model".}
}
\value{
An \R object of class \code{\linkS4class{S4BuyseTest}}.
}
\description{
Performs Generalized Pairwise Comparisons (GPC) between two groups.
Can handle one or several binary, continuous and time-to-event endpoints.
}
\details{
\bold{Specification of the GPC model} \cr
There are two way to specify the GPC model in \code{BuyseTest}. 
A \emph{Formula interface} via the argument \code{formula} with:
\itemize{
\item on one side of the \code{~} symbol the endpoints by order of priority, surrounded by parentheses and a character string indicating the type of variable (binary,continuous,timetoevent). Additional argument may be included in the parenthesis: threshold of clinical relevance (\code{threshold}), restriction time for time to event endpoints (\code{restriction}), ...
\item in the other side a binary variable defining the two treatment arms.
\item the right-hand side of the formula may also contain a strata variable(s) surrounded by parenthese and the character strata. The argument \code{match} should be set to \code{TRUE} when strata are independent but not observations (e.g. cross-over trial with strata as patients).
}

A \emph{Vector interface} using  the following arguments \itemize{
  \item \code{treatment}: [character] name of the treatment variable identifying the control and the experimental group.
Must have only two levels (e.g. \code{0} and \code{1}).
  \item \code{endpoint}: [character vector] the name of the endpoint variable(s).
  \item \code{threshold}: [numeric vector] critical values used to compare the pairs (threshold of minimal important difference).
A pair will be classified as neutral if the difference in endpoint is strictly below this threshold.
There must be one threshold for each endpoint variable; it must be \code{NA} for binary endpoints and positive for continuous or time to event endpoints. 
  \item \code{status}: [character vector] the name of the binary variable(s) indicating whether the endpoint was observed or censored.
Must value \code{NA} when the endpoint is not a time to event.
  \item \code{operator}: [character vector] the sign defining a favorable endpoint.
\code{">0"} indicates that higher values are favorable while "<0" indicates the opposite.
  \item \code{type}: [character vector] indicates whether it is
a binary outcome  (\code{"b"}, \code{"bin"}, or \code{"binary"}),
a continuous outcome  (\code{"c"}, \code{"cont"}, or \code{"continuous"}),
or a time to event outcome  (\code{"t"}, \code{"tte"}, \code{"time"}, or \code{"timetoevent"})
  \item \code{censoring}: [character vector] is the endpoint subject to right or left censoring (\code{"left"} or \code{"right"}). The default is right-censoring and left-censoring is only implemented with the Gehan's scoring rule.
  \item \code{restriction}: [numeric vector] value above which any difference is classified as neutral.
  \item \code{strata}: [character vector] if not \code{NULL}, the GPC will be applied within each group of patient defined by the strata variable(s).
}
The formula interface can be more concise, especially when considering few outcomes, but may be more difficult to apprehend for new users.
Note that arguments \code{endpoint}, \code{threshold}, \code{status}, \code{operator},  \code{type}, and \code{censoring} must have the same length. \cr \cr \cr


\bold{GPC procedure} \cr
The GPC procedure form all pairs of observations, one belonging to the experimental group and the other to the control group, and class them in 4 categories: \itemize{
 \item \emph{Favorable pair}: the endpoint is better for the observation in the experimental group.
 \item \emph{Unfavorable pair}: the endpoint is better for the observation in the control group.
 \item \emph{Neutral pair}: the difference between the endpoints of the two observations is (in absolute value) below the threshold. When \code{threshold=0}, neutral pairs correspond to pairs with equal endpoint. Lower-priority outcomes (if any) are then used to classified the pair into favorable/unfavorable.
 \item \emph{Uninformative pair}: censoring/missingness prevents from classifying into favorable, unfavorable or neutral.
}
With complete data, pairs can be decidely classified as favorable/unfavorable/neutral.
In presence of missing values, the GPC procedure uses the scoring rule (argument \code{scoring.rule}) and the correction for uninformative pairs (argument \code{correction.uninf}) to classify the pairs.
The classification may not be 0,1, e.g. the probability that the pair is favorable/unfavorable/neutral with the Peron's/Efron's scoring rule.
To export the classification of each pair set the argument \code{keep.pairScore} to \code{TRUE} and call the function \code{getPairScore} on the result of the \code{BuyseTest} function. \cr \cr \cr


\bold{Handling missing values}: the recommended default approach is to use the Peron's scoring rule with a restriction time if a non-neglectable part of the survival is unknown and otherwise analyse uniformative pairs using the following endpoint(s) if any. 
\itemize{
  \item \code{scoring.rule}: indicates how to handle right-censoring in time to event endpoints using information from the survival curves. \describe{
   \item{\code{scoring.rule="Gehan"}}{When no observations is censored or only the pair with the largest timepoint is censored, the pair is decidedly classified as favorable, unfavorable, or neutral. Otherwise pairs are classified as uninformative.}
   \item{\code{scoring.rule="Peron"}}{Score pairs involving censored observations using the (group-specific) survival curves. It may still lead to uninformative pairs when the survival curve is only partially known.}
   \item{\code{scoring.rule="Efron"}}{Same as the Peron's scoring rule except that the survival curve is extrapolated to 0 when its tail is unknown. Only relevant when using a (stratified) Kaplan-Meier estimator and no competing risks.}
}

  \item \code{correction.uninf}: indicates how to handle pairs that were classified as uninformative by the scoring rule.  \describe{
   \item{\code{correction.uninf=0}}{ treat them as uninformative: this is an equivalent to complete case analysis when \code{neutral.as.uninf=FALSE}, while when \code{neutral.as.uninf=TRUE}, uninformative pairs are treated as neutral, i.e., analyzed at the following endpoint (if any). This approach will (generally) lead to biased estimates for the proportion of favorable, unfavorable, or neutral pairs.}
   \item{\code{correction.uninf=1}}{ imputes to the uninformative pairs the average score of the informative pairs, i.e. assumes that uninformative pairs would on average behave like informative pairs. This is therefore the recommanded approach when this assumption is resonnable, typically when the the tail of the survival function estimated by the Kaplan–Meier method is close to 0.}
   \item{\code{correction.uninf=2}}{ up-weight informative pairs to represent uninformative pairs. It also assumes that uninformative pairs would on average behave like informative pairs and is only recommanded when the analysis is stopped after the first endpoint with uninformative pairs.}
}
}


\bold{Statistical inference} \cr
The argument \code{method.inference} defines how to approximate the distribution of the GPC estimators and so how standard errors, confidence intervals, and p-values are computed.
Available methods are:
\itemize{
  \item argument \code{method.inference="none"}: only the point estimate is computed which makes the execution of the \code{BuyseTest} faster than with the other methods.
  \item argument \code{method.inference="u-statistic"}: compute the variance of the estimate using a H-projection of order 1 (default option) or 2 (see \code{BuyseTest.options}). The first order is downward biased but consistent. When considering the Gehan scoring rule, no transformation nor correction, the second order is unbiased and equivalent to the variance of the (stratified) bootstrap distribution. P-values and confidence intervals are then evaluated assuming that the estimates follow a Gaussian distribution. In presence of strata, the weights used to combine estimates across strata are assumed known. This is the case the type of weights used is pre-defined and the covariate distribution is fixed by design as in a blocked randomized trial.
\bold{WARNING}: the current implementation of the H-projection is not valid when using corrections for uninformative pairs (\code{correction.uninf=1}, or \code{correction.uninf=2}).
  \item argument \code{method.inference="permutation"}: perform a permutation test, estimating in each sample the summary statistics (net benefit, win ratio).
  \item argument \code{method.inference="studentized permutation"}: perform a permutation test, estimating in each sample the summary statistics (net benefit, win ratio) and the variance-covariance matrix of the estimate.
  \item argument \code{method.inference="varExact permutation"}: compute the variance of the permutation distribution using a closed-form formula (Anderson and Verbeeck 2023). P-values and confidence intervals are then evaluated assuming that the estimates follow a Gaussian distribution.
\bold{WARNING}: the current implementation of the variance estimator for the permutation distribution is not valid when using the Peron scoring rule or corrections for uninformative pairs.
  \item argument \code{method.inference="bootstrap"}: perform a non-parametric boostrap, estimating in each sample the summary statistics (net benefit, win ratio).
  \item argument \code{method.inference="studentized bootstrap"}: perform a non-parametric boostrap, estimating in each sample the summary statistics (net benefit, win ratio) and the variance-covariance matrix of the estimator.
}
Additional arguments for permutation and bootstrap resampling:
\itemize{
   \item \code{strata.resampling} If \code{NA} or of length 0, the permutation/non-parametric boostrap will be performed by resampling in the whole sample.
Otherwise, the permutation/non-parametric boostrap will be performed separately for each level that the variable defined in \code{strata.resampling} take.
   \item \code{n.resampling} set the number of permutations/samples used.
A large number of permutations (e.g. \code{n.resampling=10000}) are needed to obtain accurate CI and p.value. See (Buyse et al., 2010) for more details.
   \item \code{seed}: the seed is used to generate one seed per sample. These seeds are the same whether one or several CPUs are used.
   \item \code{cpus} indicates whether the resampling procedure can be splitted on several cpus to save time. Can be set to \code{"all"} to use all available cpus.
The detection of the number of cpus relies on the \code{detectCores} function from the \emph{parallel} package. \cr \cr
}

\bold{Pooling results across strata} \cr Consider \eqn{K} strata and denote by \eqn{m_k} and \eqn{n_k} the sample size in the control and active arm (respectively) for strata \eqn{k}. Let \eqn{\sigma_k} be the standard error of the strata-specific summary statistic (e.g. net benefit). The strata specific weights, \eqn{w_k}, are given by:
\itemize{
\item \code{"CMH"}: \eqn{w_k=\frac{\frac{m_k \times n_k}{m_k + n_k}}{\sum_{l=1}^K \frac{m_l \times n_l}{m_l + n_l}}}. Optimal if the if the odds ratios are constant across strata.
\item \code{"equal"}:  \eqn{w_k=\frac{1}{K}}
\item \code{"Buyse"}:  \eqn{w_k=\frac{m_k \times n_k}{\sum_{l=1}^K m_l \times n_l}}. Optimal if the risk difference is constant across strata
\item \code{"var-*"} (e.g. \code{"var-netBenefit"}): . \eqn{w_k=\frac{1/\sigma^2_k}{\sum_{l=1}^K 1/\sigma^2_k}}
}
Only when using \code{"var-winRatio"}, the pooled Win Ratio is computed by pooling the strata-specific win-ratios. Otherwise the pooled Win Ratio is obtained by dividing the pooled number of favorable pairs divided by the pooled number of unfavorable pairs, possibly adding half the pooled neutral pairs, according to formula (1) in Dong et al. (2018). \cr \cr

\bold{Default values} \cr
The default of the arguments
\code{scoring.rule}, \code{correction.uninf}, \code{method.inference}, \code{n.resampling},
\code{hierarchical}, \code{neutral.as.uninf}, \code{keep.pairScore}, \code{strata.resampling},
\code{cpus}, \code{trace} is read from \code{BuyseTest.options()}. \cr
Additional (hidden) arguments are \itemize{
 \item \code{alternative} [character] the alternative hypothesis. Must be one of "two.sided", "greater" or "less" (used by \code{confint}).
 \item \code{conf.level} [numeric] level for the confidence intervals (used by \code{confint}).
 \item \code{keep.survival} [logical] export the survival values used by the Peron's scoring rule.
 \item \code{order.Hprojection} [1 or 2] the order of the H-projection used to compute the variance when \code{method.inference="u-statistic"}. 
}
}
\examples{
library(data.table)

#### simulate some data ####
set.seed(10)
df.data <- simBuyseTest(1e2, n.strata = 2)

## display 
if(require(prodlim)){
   resKM_tempo <- prodlim(Hist(eventtime,status)~treatment, data = df.data)
   plot(resKM_tempo)
}

#### one time to event endpoint ####
BT <- BuyseTest(treatment ~ TTE(eventtime, status = status), data= df.data)

summary(BT) ## net benefit
model.tables(BT) ## export the table at the end of summary
summary(BT, percentage = FALSE)  
summary(BT, statistic = "winRatio") ## win Ratio

## permutation instead of asymptotics to compute the p-value
\dontrun{
    BTperm <- BuyseTest(treatment ~ TTE(eventtime, status = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e3)
}
\dontshow{
    BTperm <- BuyseTest(treatment ~ TTE(eventtime, status = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e1, trace = 0)
}
summary(BTperm)
summary(BTperm, statistic = "winRatio") 

## same with parallel calculations
\dontrun{
    BTperm <- BuyseTest(treatment ~ TTE(eventtime, status = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e3, cpus = 8)
    summary(BTperm)
}

## method Gehan is much faster but does not optimally handle censored observations
BT <- BuyseTest(treatment ~ TTE(eventtime, status = status), data=df.data,
                scoring.rule = "Gehan", trace = 0)
summary(BT)

#### one time to event endpoint: only differences in survival over 1 unit ####
BT <- BuyseTest(treatment ~ TTE(eventtime, threshold = 1, status = status), data=df.data)
summary(BT)

#### one time to event endpoint with a strata variable
BTS <- BuyseTest(treatment ~ strata + TTE(eventtime, status = status), data=df.data)
summary(BTS)

#### several endpoints with a strata variable
ff <- treatment ~ strata + T(eventtime, status, 1) + B(toxicity) 
ff <- update(ff, 
            ~. + T(eventtime, status, 0.5) + C(score, 1) + T(eventtime, status, 0.25))

BTM <- BuyseTest(ff, data=df.data)
summary(BTM)
plot(BTM)

#### real example : veteran dataset of the survival package ####
## Only one endpoint. Type = Time-to-event. Thresold = 0. Stratfication by histological subtype
## scoring.rule = "Gehan"

if(require(survival)){
\dontrun{
  data(cancer, package = "survival") ## import veteran
 
  ## scoring.rule = "Gehan"
  BT_Gehan <- BuyseTest(trt ~ celltype + TTE(time,threshold=0,status=status), 
                        data=veteran, scoring.rule="Gehan")
  
  summary_Gehan <- summary(BT_Gehan)
  summary_Gehan <- summary(BT_Gehan, statistic = "winRatio")
  
  ## scoring.rule = "Peron"
  BT_Peron <- BuyseTest(trt ~ celltype + TTE(time,threshold=0,status=status), 
                        data=veteran, scoring.rule="Peron")

  summary(BT_Peron)
}
}
}
\references{
On the GPC procedure: Marc Buyse (2010). \bold{Generalized pairwise comparisons of prioritized endpoints in the two-sample problem}. \emph{Statistics in Medicine} 29:3245-3257 \cr
On the win ratio: D. Wang, S. Pocock (2016). \bold{A win ratio approach to comparing continuous non-normal outcomes in clinical trials}. \emph{Pharmaceutical Statistics} 15:238-245 \cr
On the stratified win ratio: G. Dong et al. (2018). \bold{The stratified win ratio}. \emph{Journal of biopharmaceutical statistics}. 28(4):778-796 \cr
On the Peron's scoring rule: J. Peron, M. Buyse, B. Ozenne, L. Roche and P. Roy (2018). \bold{An extension of generalized pairwise comparisons for prioritized outcomes in the presence of censoring}. \emph{Statistical Methods in Medical Research} 27: 1230-1239. \cr
On the Peron's scoring rule with competing risks: E. Cantagallo, M. De Backer, M. Kicinski, B. Ozenne, L Collette, C. Legrand, M. Buyse, J. Péron. \bold{A new measure of treatment effect in clinical trials involving competing risks based on generalized pairwise comparisons}. \emph{Biometrical Journal} 2021 Feb;63(2):272-288. doi: 10.1002/bimj.201900354.
On the Gehan's scoring rule: Gehan EA (1965). \bold{A generalized two-sample Wilcoxon test for doubly censored data}. \emph{Biometrika}  52(3):650-653 \cr
On inference in GPC using the U-statistic theory: Ozenne B, Budtz-Jorgensen E, Peron J (2021). \bold{The asymptotic distribution of the Net Benefit estimator in presence of right-censoring}. \emph{Statistical Methods in Medical Research} 2021 doi:10.1177/09622802211037067 \cr
On how using a restriction time: Piffoux M, Ozenne B, De Backer M, Buyse M, Chiem JC, Péron J (2024). \bold{Restricted Net Treatment Benefit in oncology}. \emph{Journal of Clinical Epidemiology}. Jun;170:111340. \cr
On the argument \code{correction.uninf}: J. Peron, M. Idlhaj, D. Maucort-Boulch, et al. (2021) \bold{Correcting the bias of the net benefit estimator due to right-censored observations}. \emph{Biometrical Journal} 63: 893–906. \cr
On closed-form formula for permutation variance:  W.N. Anderson and J. Verbeeck (2023). \bold{Exact Permutation and Bootstrap Distribution of Generalized Pairwise Comparisons Statistics}. \emph{Mathematics} , 11, 1502. doi:10.3390/math11061502.
}
\seealso{
\code{\link{S4BuyseTest-summary}} for a summary of the results of generalized pairwise comparison. \cr
\code{\link{S4BuyseTest-confint}} for exporting estimates with confidence intervals and p-values. \cr
\code{\link{S4BuyseTest-model.tables}} for exporting the number or percentage of favorable/unfavorable/neutral/uninformative pairs. \cr
\code{\link{sensitivity}} for performing a sensitivity analysis on the choice of the threshold(s). \cr
\code{\link{S4BuyseTest-plot}} for graphical display of the pairs across endpoints. \cr
\code{\link{getIid}} for exporting the first order H-decomposition. \cr
\code{\link{getPairScore}} for exporting the scoring of each pair.
}
\author{
Brice Ozenne
}
\keyword{models}
