% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metric_graph.R
\name{metric_graph}
\alias{metric_graph}
\title{Metric graph}
\value{
Object of \code{\link[R6]{R6Class}} for creating metric graphs.
}
\description{
Class representing a general metric graph.
}
\details{
A graph object created from vertex and edge matrices, or from an
\code{sp::SpatialLines} object where each line is representing and edge. For more details,
see the vignette:
\code{vignette("metric_graph", package = "MetricGraph")}
}
\examples{
library(sp)
line1 <- Line(rbind(c(0, 0), c(2, 0)))
line2 <- Line(rbind(c(2, 0), c(1, 1)))
line3 <- Line(rbind(c(1, 1), c(0, 0)))
lines <-  SpatialLines(list(Lines(list(line1), ID = "1"),
                           Lines(list(line2), ID = "2"),
                           Lines(list(line3), ID = "3")))
graph <- metric_graph$new(lines)
graph$plot()

}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{V}}{Matrix with positions in Euclidean space of the vertices of the
graph.}

\item{\code{nV}}{The number of vertices.}

\item{\code{E}}{Matrix with the edges of the graph, where each row represents an
edge, \code{E[i,1]} is the vertex at the start of the ith edge and \code{E[i,2]} is
the vertex at the end of the edge.}

\item{\code{nE}}{The number of edges.}

\item{\code{edge_lengths}}{Vector with the lengths of the edges in the graph.}

\item{\code{EID}}{Vector with the IDs of the edges in the graph.}

\item{\code{LtE}}{Matrix with edge positions on the lines.}

\item{\code{ELend}}{Vector with the locations of the end points of the edges on
the lines in the graph. The locations are normalized on the line.}

\item{\code{ELstart}}{Vector with the locations of the starting points of the
edges on the lines in the graph. The locations are normalized on the line.}

\item{\code{C}}{Constraint matrix used to set Kirchhoff constraints.}

\item{\code{CoB}}{Change-of-basis object used for Kirchhoff constraints.}

\item{\code{data}}{List containing data on the metric graph.}

\item{\code{PtV}}{Vector with the indices of the vertices which are observation
locations.}

\item{\code{mesh}}{Mesh object used for plotting.}

\item{\code{lines}}{The lines in the graph.}

\item{\code{geo_dist}}{Geodesic distances between the vertices in the graph.}

\item{\code{res_dist}}{Resistance distances between the observation locations.}

\item{\code{Laplacian}}{The weighted graph Laplacian of the vertices in the
graph. The weights are given by the edge lengths.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-metric_graph-new}{\code{metric_graph$new()}}
\item \href{#method-metric_graph-compute_geodist}{\code{metric_graph$compute_geodist()}}
\item \href{#method-metric_graph-compute_geodist_PtE}{\code{metric_graph$compute_geodist_PtE()}}
\item \href{#method-metric_graph-compute_geodist_mesh}{\code{metric_graph$compute_geodist_mesh()}}
\item \href{#method-metric_graph-compute_resdist}{\code{metric_graph$compute_resdist()}}
\item \href{#method-metric_graph-compute_resdist_PtE}{\code{metric_graph$compute_resdist_PtE()}}
\item \href{#method-metric_graph-get_degrees}{\code{metric_graph$get_degrees()}}
\item \href{#method-metric_graph-compute_resdist_mesh}{\code{metric_graph$compute_resdist_mesh()}}
\item \href{#method-metric_graph-compute_laplacian}{\code{metric_graph$compute_laplacian()}}
\item \href{#method-metric_graph-prune_vertices}{\code{metric_graph$prune_vertices()}}
\item \href{#method-metric_graph-get_PtE}{\code{metric_graph$get_PtE()}}
\item \href{#method-metric_graph-get_Spoints}{\code{metric_graph$get_Spoints()}}
\item \href{#method-metric_graph-observation_to_vertex}{\code{metric_graph$observation_to_vertex()}}
\item \href{#method-metric_graph-get_mesh_locations}{\code{metric_graph$get_mesh_locations()}}
\item \href{#method-metric_graph-clear_observations}{\code{metric_graph$clear_observations()}}
\item \href{#method-metric_graph-add_observations}{\code{metric_graph$add_observations()}}
\item \href{#method-metric_graph-buildC}{\code{metric_graph$buildC()}}
\item \href{#method-metric_graph-build_mesh}{\code{metric_graph$build_mesh()}}
\item \href{#method-metric_graph-compute_fem}{\code{metric_graph$compute_fem()}}
\item \href{#method-metric_graph-mesh_A}{\code{metric_graph$mesh_A()}}
\item \href{#method-metric_graph-VtEfirst}{\code{metric_graph$VtEfirst()}}
\item \href{#method-metric_graph-plot}{\code{metric_graph$plot()}}
\item \href{#method-metric_graph-plot_connections}{\code{metric_graph$plot_connections()}}
\item \href{#method-metric_graph-plot_function}{\code{metric_graph$plot_function()}}
\item \href{#method-metric_graph-plot_movie}{\code{metric_graph$plot_movie()}}
\item \href{#method-metric_graph-add_mesh_observations}{\code{metric_graph$add_mesh_observations()}}
\item \href{#method-metric_graph-get_initial_graph}{\code{metric_graph$get_initial_graph()}}
\item \href{#method-metric_graph-A}{\code{metric_graph$A()}}
\item \href{#method-metric_graph-coordinates}{\code{metric_graph$coordinates()}}
\item \href{#method-metric_graph-clone}{\code{metric_graph$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-new"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{metric_graph} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$new(
  lines = NULL,
  V = NULL,
  E = NULL,
  longlat = FALSE,
  tolerance = list(vertex_vertex = 1e-07, vertex_line = 1e-07, line_line = 0),
  check_connected = TRUE,
  adjust_lines = NULL,
  remove_deg2 = FALSE,
  remove_circles = TRUE,
  verbose = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{lines}}{Object of type \code{SpatialLinesDataFrame} or \code{SpatialLines}.}

\item{\code{V}}{n x 2 matrix with Euclidean coordinates of the n vertices.}

\item{\code{E}}{m x 2 matrix where each row represents one of the m edges.}

\item{\code{longlat}}{If TRUE, then it is assumed that the coordinates are given.
in Longitude/Latitude and that distances should be computed in km.}

\item{\code{tolerance}}{List that provides tolerances during the construction of
the graph:
\itemize{
\item \code{vertex_vertex} Vertices that are closer than this number are merged
(default = 1e-7).
\item \code{vertex_line} If a vertex at the end of one line is closer than this
number to another line, this vertex is connected to that line
(default = 1e-7).
\item \code{line_line} If two lines at some point are closer than this number, a new
vertex is added at that point and the two lines are connected (default = 0).
}

If \code{longlat = TRUE}, the tolerances are given in km.}

\item{\code{check_connected}}{If \code{TRUE}, it is checked whether the graph is
connected and a warning is given if this is not the case.}

\item{\code{adjust_lines}}{Set to \code{TRUE} to adjust the lines object to match the graph
connections. This can take some time for large graphs, so by default it is \code{TRUE}
for graphs with at most 100 lines, and \code{FALSE} for larger graphs.}

\item{\code{remove_deg2}}{Set to \code{TRUE} to remove all vertices of degree 2 in the
initialization. Default is \code{FALSE}.}

\item{\code{remove_circles}}{All circlular edges with a length smaller than this number
are removed. The default is the \code{vertex_vertex} tolerance.}

\item{\code{verbose}}{Print progress of graph creation.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
A graph object can be initialized in two ways. The first method
is to specify V and E. In this case, all edges are assumed to be straight
lines. The second option is to specify the graph via the \code{lines} input.
In this case, the vertices are set by the end points of the lines.
Thus, if two lines are intersecting somewhere else, this will not be
viewed as a vertex.
}

\subsection{Returns}{
A \code{metric_graph} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-compute_geodist"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-compute_geodist}{}}}
\subsection{Method \code{compute_geodist()}}{
Computes shortest path distances between the vertices in the
graph
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$compute_geodist(full = FALSE, obs = TRUE, group = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{full}}{Should the geodesic distances be computed for all
the available locations? If \code{FALSE}, it will be computed
separately for the locations of each group.}

\item{\code{obs}}{Should the geodesic distances be computed at the observation
locations?}

\item{\code{group}}{Vector or list containing which groups to compute the distance
for. If \code{NULL}, it will be computed for all groups.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No return value. Called for its side effects. The computed geodesic
distances are stored in the \code{geo_dist} element of the \code{metric_graph} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-compute_geodist_PtE"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-compute_geodist_PtE}{}}}
\subsection{Method \code{compute_geodist_PtE()}}{
Computes shortest path distances between the vertices in the
graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$compute_geodist_PtE(
  PtE,
  normalized = TRUE,
  include_vertices = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{PtE}}{Points to compute the metric for.}

\item{\code{normalized}}{are the locations in PtE in normalized distance?}

\item{\code{include_vertices}}{Should the original vertices be included in the
distance matrix?}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A matrix containing the geodesic distances.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-compute_geodist_mesh"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-compute_geodist_mesh}{}}}
\subsection{Method \code{compute_geodist_mesh()}}{
Computes shortest path distances between the vertices in the
mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$compute_geodist_mesh()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
No return value. Called for its side effects. The geodesic distances
on the mesh are stored in \code{mesh$geo_dist} in the \code{metric_graph} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-compute_resdist"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-compute_resdist}{}}}
\subsection{Method \code{compute_resdist()}}{
Computes the resistance distance between the observation
locations.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$compute_resdist(full = FALSE, obs = TRUE, group = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{full}}{Should the resistance distances be computed for all
the available locations. If \code{FALSE}, it will be computed
separately for the locations of each group.}

\item{\code{obs}}{Should the resistance distances be computed at the observation
locations?}

\item{\code{group}}{Vector or list containing which groups to compute the distance
for. If \code{NULL}, it will be computed for all groups.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No return value. Called for its side effects. The geodesic distances
are stored in the \code{res_dist} element of the \code{metric_graph} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-compute_resdist_PtE"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-compute_resdist_PtE}{}}}
\subsection{Method \code{compute_resdist_PtE()}}{
Computes the resistance distance between the observation
locations.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$compute_resdist_PtE(
  PtE,
  normalized = TRUE,
  include_vertices = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{PtE}}{Points to compute the metric for.}

\item{\code{normalized}}{Are the locations in PtE in normalized distance?}

\item{\code{include_vertices}}{Should the original vertices be included in the
Laplacian matrix?}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A matrix containing the resistance distances.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-get_degrees"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-get_degrees}{}}}
\subsection{Method \code{get_degrees()}}{
Returns the degrees of the vertices in the metric graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$get_degrees()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A vector containing the degrees of the vertices.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-compute_resdist_mesh"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-compute_resdist_mesh}{}}}
\subsection{Method \code{compute_resdist_mesh()}}{
Computes the resistance metric between the vertices in the
mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$compute_resdist_mesh()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
No return value. Called for its side effects. The geodesic distances
on the mesh are stored in the \code{mesh$res_dist} element in the \code{metric_graph}
object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-compute_laplacian"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-compute_laplacian}{}}}
\subsection{Method \code{compute_laplacian()}}{
Computes the weigthed graph Laplacian for the graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$compute_laplacian(full = FALSE, obs = TRUE, group = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{full}}{Should the resistance distances be computed for all
the available locations. If \code{FALSE}, it will be computed
separately for the locations of each group.}

\item{\code{obs}}{Should the resistance distances be computed at the observation
locations?}

\item{\code{group}}{Vector or list containing which groups to compute the
Laplacian for. If \code{NULL}, it will be computed for all groups.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No reutrn value. Called for its side effects. The Laplacian is stored
in the \code{Laplacian} element in the \code{metric_graph} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-prune_vertices"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-prune_vertices}{}}}
\subsection{Method \code{prune_vertices()}}{
Removes vertices of degree 2 from the metric graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$prune_vertices()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
No return value. Called for its side effects.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-get_PtE"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-get_PtE}{}}}
\subsection{Method \code{get_PtE()}}{
Gets PtE from the data.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$get_PtE()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A matrix with two columns, where the first column contains the edge
number and the second column contains the distance on edge of the
observation locations.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-get_Spoints"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-get_Spoints}{}}}
\subsection{Method \code{get_Spoints()}}{
Gets the spatial points from the data.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$get_Spoints()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A \code{SpatialPoints} object of the observation locations.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-observation_to_vertex"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-observation_to_vertex}{}}}
\subsection{Method \code{observation_to_vertex()}}{
Adds observation locations as vertices in the graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$observation_to_vertex(tolerance = 1e-15)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{tolerance}}{Observations locations are merged to a single vertex if
they are closer than this number (given in relative edge distance between
0 and 1). The default is \code{1e-15}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No return value. Called for its side effects.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-get_mesh_locations"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-get_mesh_locations}{}}}
\subsection{Method \code{get_mesh_locations()}}{
Returns a list or a matrix with the mesh locations.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$get_mesh_locations(bru = FALSE, loc = NULL, normalized = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{bru}}{Should an 'inlabru'-friendly list be returned?}

\item{\code{loc}}{If \code{bru} is set to \code{TRUE}, the name of the location variable.
The default name is 'loc'.}

\item{\code{normalized}}{If TRUE, then the distances in \code{distance_on_edge} are
assumed to be normalized to (0,1). Default TRUE.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A list or a matrix containing the mesh locations.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-clear_observations"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-clear_observations}{}}}
\subsection{Method \code{clear_observations()}}{
Clear all observations from the \code{metric_graph} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$clear_observations()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
No return value. Called for its side effects.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-add_observations"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-add_observations}{}}}
\subsection{Method \code{add_observations()}}{
Add observations to the metric graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$add_observations(
  Spoints = NULL,
  data = NULL,
  edge_number = "edge_number",
  distance_on_edge = "distance_on_edge",
  coord_x = "coord_x",
  coord_y = "coord_y",
  data_coords = c("PtE", "euclidean"),
  group = NULL,
  normalized = FALSE,
  tolerance = max(self$edge_lengths)/2
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{Spoints}}{\code{SpatialPoints} or \code{SpatialPointsDataFrame} containing the
observations. It may include the coordinates of the observations only, or
the coordinates as well as the observations.}

\item{\code{data}}{A \code{data.frame} or named list containing the observations. In
case of groups, the data.frames for the groups should be stacked vertically,
with a column indicating the index of the group. If \code{data} is not \code{NULL},
it takes priority over any eventual data in \code{Spoints}.}

\item{\code{edge_number}}{Column (or entry on the list) of the \code{data} that
contains the edge numbers. If not supplied, the column with name
"edge_number" will be chosen. Will not be used if \code{Spoints} is not \code{NULL}.}

\item{\code{distance_on_edge}}{Column (or entry on the list) of the \code{data} that
contains the edge numbers. If not supplied, the column with name
"distance_on_edge" will be chosen.  Will not be used if \code{Spoints} is not
\code{NULL}.}

\item{\code{coord_x}}{Column (or entry on the list) of the \code{data} that contains
the x coordinate. If not supplied, the column with name "coord_x" will be
chosen. Will not be used if \code{Spoints} is not \code{NULL} or if \code{data_coords} is
\code{PtE}.}

\item{\code{coord_y}}{Column (or entry on the list) of the \code{data} that contains
the y coordinate. If not supplied, the column with name "coord_x" will be
chosen. Will not be used if \code{Spoints} is not \code{NULL} or if \code{data_coords} is
\code{PtE}.}

\item{\code{data_coords}}{To be used only if \code{Spoints} is \code{NULL}. It decides which
coordinate system to use. If \code{PtE}, the user must provide \code{edge_number} and
\code{distance_on_edge}, otherwise if \code{euclidean}, the user must provide
\code{coord_x} and \code{coord_y}.}

\item{\code{group}}{If the data is grouped (for example measured at different time
points), this argument specifies the the column (or entry on the list) in
which the group varialbe is stored.}

\item{\code{normalized}}{if TRUE, then the distances in \code{distance_on_edge} are
assumed to be normalized to (0,1). Default FALSE. Will not be used if
\code{Spoints} is not \code{NULL}.}

\item{\code{tolerance}}{Parameter to control a warning when adding observations.
If the distance of some location and the closest point on the graph is
greater than the tolerance, the function will display a warning.
This helps detecting mistakes on the input locations when adding new data.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No return value. Called for its side effects. The observations are
stored in the \code{data} element of the \code{metric_graph} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-buildC"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-buildC}{}}}
\subsection{Method \code{buildC()}}{
Build Kirchoff constraint matrix from edges.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$buildC(alpha = 2, edge_constraint = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{alpha}}{the type of constraint (currently only supports 2)}

\item{\code{edge_constraint}}{if TRUE, add constraints on vertices of degree 1}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
Currently not implemented for circles (edges that start and end
in the same vertex)
}

\subsection{Returns}{
No return value. Called for its side effects.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-build_mesh"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-build_mesh}{}}}
\subsection{Method \code{build_mesh()}}{
Builds mesh object for graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$build_mesh(h = NULL, n = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{h}}{Maximum distance between mesh nodes (should be provided if n is
not provided).}

\item{\code{n}}{Maximum number of nodes per edge (should be provided if h is not
provided).}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The mesh is a list with the objects:
\itemize{
\item \code{PtE} The mesh locations excluding the original vertices;
\item \code{V} The verties of the mesh;
\item \code{E} The edges of the mesh;
\item \code{n_e} The number of vertices in the mesh per original edge in the graph;
\item \code{h_e} The mesh width per edge in the graph;
\item \code{ind} The indices of the vertices in the mesh;
\item \code{VtE} All mesh locations including the original vertices.
}
}

\subsection{Returns}{
No return value. Called for its side effects. The mesh is stored in
the \code{mesh} element of the \code{metric_graph} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-compute_fem"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-compute_fem}{}}}
\subsection{Method \code{compute_fem()}}{
Build mass and stiffness matrices for given mesh object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$compute_fem()}\if{html}{\out{</div>}}
}

\subsection{Details}{
The function builds: The matrix \code{C} which is the mass matrix with
elements \eqn{C_{ij} = <\phi_i, \phi_j>}, the matrix \code{G} which is the stiffness
matrix with elements \eqn{G_{ij} = <d\phi_i, d\phi_j>}, the matrix \code{B} with
elements \eqn{B_{ij} = <d\phi_i, \phi_j>}, and the vector with weights
\eqn{<\phi_i, 1>}.
}

\subsection{Returns}{
No return value. Called for its side effects. The finite element
matrices \code{C}, \code{G} and \code{B} are stored in the \code{mesh} element in the
\code{metric_graph} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-mesh_A"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-mesh_A}{}}}
\subsection{Method \code{mesh_A()}}{
Computes observation matrix for mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$mesh_A(PtE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{PtE}}{Locations given as (edge number in graph, normalized location on
edge)}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
For n locations and a mesh with m nodes, \code{A} is an n x m matrix with
elements \eqn{A_{ij} = \phi_j(s_i)}{A_{ij} = \phi_j(s_i)}.
}

\subsection{Returns}{
The observation matrix.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-VtEfirst"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-VtEfirst}{}}}
\subsection{Method \code{VtEfirst()}}{
Find one edge corresponding to each vertex.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$VtEfirst()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A nV x 2 matrix the first element of the \code{i}th row is the edge
number corresponding to the \code{i}th vertex and the second value is 0
if the vertex is at the start of the edge and 1 if the vertex
is at the end of the edge.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-plot"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-plot}{}}}
\subsection{Method \code{plot()}}{
Plots the metric graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$plot(
  data = NULL,
  group = 1,
  plotly = FALSE,
  vertex_size = 3,
  vertex_color = "black",
  edge_width = 0.3,
  edge_color = "black",
  data_size = 1,
  mesh = FALSE,
  X = NULL,
  X_loc = NULL,
  p = NULL,
  degree = FALSE,
  direction = FALSE,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{Which column of the data to plot? If \code{NULL}, no data will be
plotted.}

\item{\code{group}}{If there are groups, which group to plot? If \code{group} is a
number, it will be the index of the group as stored internally. If \code{group}
is a character, then the group will be chosen by its name.}

\item{\code{plotly}}{Use plot_ly for 3D plot (default \code{FALSE}). This option
requires the 'plotly' package.}

\item{\code{vertex_size}}{Size of the vertices.}

\item{\code{vertex_color}}{Color of vertices.}

\item{\code{edge_width}}{Line width for edges.}

\item{\code{edge_color}}{Color of edges.}

\item{\code{data_size}}{Size of markers for data.}

\item{\code{mesh}}{Plot the mesh locations?}

\item{\code{X}}{Additional values to plot.}

\item{\code{X_loc}}{Locations of the additional values in the format
(edge, normalized distance on edge).}

\item{\code{p}}{Existing objects obtained from 'ggplot2' or 'plotly' to add the graph to}

\item{\code{degree}}{Show the degrees of the vertices?}

\item{\code{direction}}{Show the direction of the edges?}

\item{\code{...}}{Additional arguments to pass to \code{ggplot()} or \code{plot_ly()}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{plot_ly} (if \code{plotly = TRUE}) or \code{ggplot} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-plot_connections"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-plot_connections}{}}}
\subsection{Method \code{plot_connections()}}{
Plots the connections in the graph
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$plot_connections()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
No return value. Called for its side effects.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-plot_function"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-plot_function}{}}}
\subsection{Method \code{plot_function()}}{
Plots continuous function on the graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$plot_function(
  X,
  plotly = FALSE,
  vertex_size = 5,
  vertex_color = "black",
  edge_width = 1,
  edge_color = "black",
  line_width = NULL,
  line_color = "rgb(0,0,200)",
  support_width = 0.5,
  support_color = "gray",
  p = NULL,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{X}}{Either an m x 3 matrix with (edge number, position on
curve (in length), value) or a vector with values for the function
evaluated at the mesh in the graph}

\item{\code{plotly}}{If \code{TRUE}, then the plot is shown in 3D. This option requires
the package 'plotly'.}

\item{\code{vertex_size}}{Size of the vertices.}

\item{\code{vertex_color}}{Color of vertices.}

\item{\code{edge_width}}{Width for edges.}

\item{\code{edge_color}}{For 3D plot, color of edges.}

\item{\code{line_width}}{For 3D plot, line width of the function curve.}

\item{\code{line_color}}{Color of the function curve.}

\item{\code{support_width}}{For 3D plot, width of support lines.}

\item{\code{support_color}}{For 3D plot, color of support lines.}

\item{\code{p}}{Previous plot to which the new plot should be added.}

\item{\code{...}}{Additional arguments for \code{ggplot()} or \code{plot_ly()}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Either a \code{ggplot} (if \code{plotly = FALSE}) or a \code{plot_ly} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-plot_movie"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-plot_movie}{}}}
\subsection{Method \code{plot_movie()}}{
Plots a movie of a continuous function evolving on the graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$plot_movie(
  X,
  plotly = TRUE,
  vertex_size = 5,
  vertex_color = "black",
  edge_width = 1,
  edge_color = "black",
  line_width = NULL,
  line_color = "rgb(0,0,200)",
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{X}}{A m x T matrix where the ith column represents the function at the
ith time, evaluated at the mesh locations.}

\item{\code{plotly}}{If \code{TRUE}, then plot is shown in 3D. This option requires the
package 'plotly'.}

\item{\code{vertex_size}}{Size of the vertices.}

\item{\code{vertex_color}}{Color of vertices.}

\item{\code{edge_width}}{Width for edges.}

\item{\code{edge_color}}{For 3D plot, color of edges.}

\item{\code{line_width}}{For 3D plot, line width of the function curve.}

\item{\code{line_color}}{Color of the function curve.}

\item{\code{...}}{Additional arguments for ggplot or plot_ly.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Either a \code{ggplot} (if \code{plotly=FALSE}) or a \code{plot_ly} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-add_mesh_observations"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-add_mesh_observations}{}}}
\subsection{Method \code{add_mesh_observations()}}{
Add observations on mesh to the object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$add_mesh_observations(data = NULL, group = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{A \code{data.frame} or named list containing the observations.
In case of groups, the data.frames for the groups should be stacked vertically,
with a column indicating the index of the group. If \code{data_frame} is not
\code{NULL}, it takes priority over any eventual data in \code{Spoints}.}

\item{\code{group}}{If the data_frame contains groups, one must provide the column
in which the group indices are stored.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No return value. Called for its side effects. The observations are
stored in the \code{data} element in the \code{metric_graph} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-get_initial_graph"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-get_initial_graph}{}}}
\subsection{Method \code{get_initial_graph()}}{
Returns a copy of the initial metric graph.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$get_initial_graph()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A \code{metric_graph} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-A"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-A}{}}}
\subsection{Method \code{A()}}{
Get the observation/prediction matrix A
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$A(group = NULL, obs_to_vert = FALSE, include_NA = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{group}}{A vector. If \code{NULL}, the A matrix for the first group will be
returned. One can use all groups by simply setting the \code{group} variable
to \verb{__all}. Otherwise, the A matrix for the groups in the vector will be
returned.}

\item{\code{obs_to_vert}}{Should the observations be turned into vertices?}

\item{\code{include_NA}}{Should the locations for which all observations are NA be
included?}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The observation or prediction matrix.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-coordinates"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-coordinates}{}}}
\subsection{Method \code{coordinates()}}{
Convert between locations on the graph and Euclidean
coordinates.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$coordinates(PtE = NULL, XY = NULL, normalized = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{PtE}}{Matrix with locations on the graph (edge number and normalized
position on the edge).}

\item{\code{XY}}{Matrix with locations in Euclidean space}

\item{\code{normalized}}{If \code{TRUE}, it is assumed that the positions in \code{PtE} are
normalized to (0,1), and the object returned if \code{XY} is specified contains
normalized locations.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
If \code{PtE} is specified, then a matrix with Euclidean coordinates of
the locations is returned. If \code{XY} is provided, then a matrix with the
closest locations on the graph is returned.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-metric_graph-clone"></a>}}
\if{latex}{\out{\hypertarget{method-metric_graph-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{metric_graph$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
