#' Create "Marker_list" from Seurat object
#'
#' @param df Dataframe generated by "FindAllMarkers" function, recommend
#'     to use parameter "group.by = "Cell_type"" and "only.pos = TRUE".
#' @param sort_by Marker sorting parameter, select "avg_log2FC" or "p_val_adj".
#'     Default parameters use "sort_by = "avg_log2FC"".
#' @param gene_filter The number of markers left for each cell type based
#'     on the "sort_by" parameter's level of difference. Default parameters use
#'     "gene_fliter = 20"
#'
#' @returns The standardized "Marker_list" in the SlimR package.
#' @export
#'
#' @importFrom utils head
#'
#' @examples library(Seurat)
#'           data("pbmc_small")
#'           sce <- pbmc_small
#'           seurat_markers <- FindAllMarkers(sce,
#'                                            group.by = "RNA_snn_res.1",
#'                                            only.pos = TRUE)
#'           Markers_list_Seurat <- read_seurat_markers(seurat_markers,
#'           sort_by = "avg_log2FC",
#'           gene_filter = 10
#'           )
#'
read_seurat_markers <- function(df, sort_by = "avg_log2FC", gene_filter = 20) {
  if (!sort_by %in% c("avg_log2FC", "p_val_adj")) {
    stop("sort_by must be either 'avg_log2FC' or 'p_val_adj'")
  }

  clusters <- split(df, df$cluster)

  processed <- lapply(clusters, function(cluster_df) {
    if (sort_by == "avg_log2FC") {
      sorted_df <- cluster_df[order(-cluster_df$avg_log2FC), , drop = FALSE]
    } else {
      sorted_df <- cluster_df[order(cluster_df$p_val_adj), , drop = FALSE]
    }

    filtered_df <- head(sorted_df, gene_filter)
    desired_order <- c("gene", "avg_log2FC", "p_val_adj", "p_val", "pct.1", "pct.2")
    reordered_df <- filtered_df[, desired_order, drop = FALSE]

    return(reordered_df)
  })

  names(processed) <- names(clusters)

  return(processed)
}
