# test.varmod.R

library(earth)
options(warn=2)
expect.err <- function(obj) # test that we got an error as expected from a try() call
{
    if(class(obj)[1] == "try-error")
        cat("Got error as expected\n")
    else
        stop("did not get expected try error")
}
printh <- function(caption)
    cat("===", caption, "\n", sep="")

CAPTION <- NULL

multifigure <- function(caption, nrow=3, ncol=3)
{
    CAPTION <<- caption
    printh(caption)
    par(mfrow=c(nrow, ncol))
    par(cex = 0.8)
    par(mar = c(3, 3, 5, 0.5)) # small margins but space for right hand axis
    par(mgp = c(1.6, 0.6, 0))  # flatten axis elements
    oma <- par("oma") # make space for caption
    oma[3] <- 2
    par(oma=oma)
}
do.caption <- function() # must be called _after_ first plot on new page
    mtext(CAPTION, outer=TRUE, font=2, line=1, cex=1)
if(!interactive())
    postscript(paper="letter")
old.par <- par(no.readonly=TRUE)

multifigure("test predict.earth with pints", 2, 2)

set.seed(2)
earth.trees <- earth(Volume~Girth, data=trees, nfold=3, ncross=3, varmod.method="earth")

printh("print.default(earth.trees$varmod)")
print.default(earth.trees$varmod)

printh("summary(earth.trees)")
print(summary(earth.trees))

printh("predict(earth.trees, interval=\"se\")")
stderrs <- predict(earth.trees, interval="se")
print(stderrs)

printh("predict(earth.trees, interval=\"abs.residual\")")
stderrs <- predict(earth.trees, interval="abs.residual")
print(stderrs)

expect.err(try(predict(earth.trees, newdata=trees, interval="training.cint")))

printh("predict(earth.trees, interval=\"training.cint\")")
cints <- predict(earth.trees, interval="training.cint")
print(cints)

printh("predict(earth.trees, interval=\"pint\", level=.80)")
news <- predict(earth.trees, interval="pint", level=.80)
print(news)

printh("predict(earth.trees, interval=\"training.pint\", level=.90)")
pints <- predict(earth.trees, interval="training.pint", level=.90)
print(pints)

printh("print.default(earth.trees$varmod$residmod)")
# have to modify earth.trees because terms field stores the environment
# as a hex address which messes up the diffs
earth.trees$varmod$residmod$terms <- NULL
print.default(earth.trees$varmod$residmod)
# prevent mistakes later where we try to use a modified earth.trees
remove(earth.trees)

multifigure("test example for varmod help page", 2, 2)

data(ozone1)
set.seed(1) # optional, for cross validation reproducibility

a <- earth(O3~temp, data=ozone1, nfold=10, ncross=3, varmod.method="earth")

print(summary(a)) # note additional info on the variance model

old.mfrow <- par(mfrow=c(2,2))

# the variance model assumes residuals are symmetric, which is not
# quite true in this example, so the lower band is a bit too big
plotmo(a, do.par=FALSE, col.response=1, level=.95, main="earth model: O3~temp")

plot(a, which=1)            # model selection plot, same as ever
plot(a, which=3, level=.95) # residual plot: note 95% pred and darker conf intervals
plot(a, which=3, level=.95, pearson=TRUE) # standardized resids are approx homoscedastic

par(par=old.mfrow)

plot(a$varmod)              # plot the embedded variance model (this calls plot.varmod)

multifigure("test example for plot.varmod help page", 2, 2)

# multivariate example (for univariate, see the example on the varmod help page)

data(ozone1)
set.seed(1) # optional, for cross validation reproducibility

mod.temp.vh.doy <- earth(O3~temp+vh+vis+doy, data=ozone1, nfold=5, ncross=3, varmod.method="x.earth")

print(summary(mod.temp.vh.doy))  # note additional info on the variance model

plot(mod.temp.vh.doy, level=.95) # note 95% pred and darker conf intervals in resids plot

plot(mod.temp.vh.doy$varmod)     # plot the variance model (this calls plot.varmod)

plot(mod.temp.vh.doy, versus="", level=.9, caption="plot.earth versus=\"\"")

plot(mod.temp.vh.doy, versus="v", level=.9, caption="plot.earth versus=\"v\" and versus=\"temp\"", do.par=2)
plot(mod.temp.vh.doy, versus="temp", level=.9, caption="", main="temp on same page")

# plot.earth will silently not plots it cannot plot below, so 1:8 becomes c(3,5)
plot(mod.temp.vh.doy, which=1:8, versus="v", info=T, caption='which=c(3,5) versus="v" info=T')

plot(mod.temp.vh.doy, versus="*", level=.9, caption="plot.earth versus=\"*\"")

plot(mod.temp.vh.doy, versus="*doy", level=.9, caption="plot.earth versus=\"*doy\"")

multifigure("test example in (very old) earth vignette", 2, 2)

data(ozone1)
x <- ozone1$temp
y <- ozone1$O3

set.seed(1) # optional, for cross validation reproducibility
earth.mod <- earth(y~x, nfold=10, ncross=3, varmod.method="earth", trace=.1)
predict <- predict(earth.mod, interval="training.pint")

order <- order(x)
x <- x[order]
y <- y[order]
predict <- predict[order,]

inconf <- y >= predict$lwr & y <= predict$upr

plot(x, y, pch=20, col=ifelse(inconf, 1, 2), main=sprintf(
    "Prediction intervals\n%.0f%% of the points are in the estimated band",
    100 * sum(inconf) / length(y)))
do.caption()

lines(x, predict$fit)
lines(x, predict$lwr, lty=2)
lines(x, predict$upr, lty=2)

# Plot the Residuals vs Fitted graph
plot(earth.mod, which=3, level=.95)

# Plot the embedded residual model
plot(earth.mod$varmod, do.par=F, which=1:2)

cat('head(residuals(earth.mod))\n')
print(head(residuals(earth.mod)))
cat('head(residuals(earth.mod, type="pearson"))\n')
print(head(residuals(earth.mod, type="pearson")))

multifigure("plot.earth varmod options", 2, 2)

plot(earth.mod, which=3, level=.95, shade.pints=0, main="plot.earth varmod options")
do.caption()
plot(earth.mod, which=3, shade.pints="orange", shade.cints="darkgray", level=.99)
plot(earth.mod, which=3, level=.95, shade.pints=0, shade.cints="mistyrose4")

multifigure("plot.earth delever and pearson", 2, 2)

set.seed(4)
earth.mod1 <- earth(O3~temp, data=ozone1, nfold=5, ncross=3, varmod.method="lm", keepxy=T, trace=.1)
plot(earth.mod1, which=3, ylim=c(-16,20), info=TRUE, level=.95)
do.caption()
plot(earth.mod1, which=3, ylim=c(-16,20), delever=TRUE, level=.95)
plot(earth.mod1, which=3, pearson=TRUE, info=TRUE,    level=.95)
plot(earth.mod1, which=3, pearson=TRUE, delever=TRUE, level=.95)

multifigure("plot.earth which=5 and which=6", 2, 3)
plot(earth.mod1, which=5, info=T,            main="which=5, info=T")
plot(earth.mod1, which=5, pearson=T, info=T, main="which=5, pearson=T, info=T")
plot(earth.mod1, which=5, pearson=T,         main="which=5, pearson=T")
do.caption()
plot(earth.mod1, which=6, info=T,            main="which=6, info=T")
plot(earth.mod1, which=6, pearson=T, info=T, main="which=6, pearson=T, info=T")
plot(earth.mod1, which=6, pearson=T,         main="which=6, pearson=T")

multifigure("plot.earth which=7 and which=8", 2, 3)
plot(earth.mod1, which=7, info=T,            main="which=7, info=T")
plot(earth.mod1, which=7, pearson=T, info=T, main="which=7, pearson=T, info=T")
plot(earth.mod1, which=7, pearson=T,         main="which=7, pearson=T")
do.caption()
plot(earth.mod1, which=8, info=T,            main="which=8, info=T")
plot(earth.mod1, which=8, pearson=T, info=T, main="which=8, pearson=T, info=T")
plot(earth.mod1, which=8, pearson=T,         main="which=8, pearson=T")

cat("summary(earth.mod1, newdata=ozone1)\n")
print(summary(earth.mod1, newdata=ozone1))

cat("summary(earth.mod1, newdata=ozone1[1:100,]:)\n")
print(summary(earth.mod1, newdata=ozone1[1:100,]))

# TODO the following give err msg as expected, but do not give a try error
# expect.err(try(summary(earth.mod1, newdata=c(1,2,3))))
# expect.err(try(summary(earth.mod1, newdata=ozone1[1:100,1:3])))

multifigure("plot(earth.mod1)", 2, 2)
set.seed(5)
earth.mod2 <- earth(y~x, nfold=10, ncross=5, varmod.method="earth")
plot(earth.mod2, caption="plot(earth.mod2)", level=.95)
do.caption()

multifigure("plot(earth.mod2) with pearson=TRUE", 2, 2)
plot(earth.mod2, pearson=TRUE, level=.95,
     caption="plot(earth.mod2, pearson=TRUE, level=.95)")
do.caption()

multifigure("plot.varmod by calling plot(earth.mod2$varmod)", 2, 2)
plot(earth.mod2$varmod)

multifigure("embedded earth model by calling plot(earth.mod2$varmod$residmod)", 2, 2)
plot(earth.mod2$varmod$residmod, caption="embedded earth model")
do.caption()

# test varmod.* args like varmod.conv

# cat("test varmod.exponent=.5\n")
# set.seed(1)
# (earth(Volume~Girth, data=trees, nfold=3, ncross=3, varmod.method="lm", trace=.3, varmod.exponent=.5))

# cat("test varmod.lambda=2/3\n")
# set.seed(1)
# (earth(Volume~Girth, data=trees, nfold=3, ncross=3, varmod.method="lm", trace=.3, varmod.lambda=2/3))

cat("test varmod.conv=50%\n")
set.seed(1)
(earth(Volume~Girth, data=trees, nfold=3, ncross=3, varmod.method="lm", trace=.3, varmod.conv=50))

cat("test varmod.conv=-5\n")
set.seed(1)
(earth(Volume~Girth, data=trees, nfold=3, ncross=3, varmod.method="lm", trace=.3, varmod.conv=-5))

cat("test varmod.clamp\n")
set.seed(1)
a.noclamp <- earth(Volume~Girth, data=trees, nfold=3, ncross=3, varmod.method="lm")
plot(a.noclamp$varmod, which=1:2, caption="a.noclamp and a.clamp", do.par=FALSE)
set.seed(1)
a.clamp <- earth(Volume~Girth, data=trees, nfold=3, ncross=3, varmod.method="lm", varmod.clamp=.6)
plot(a.clamp$varmod, which=1:2, caption="", do.par=FALSE)

cat("test varmod.minspan=-5\n")
set.seed(1)
a.varmod.minspan.minus5 <- earth(Volume~Girth, data=trees, nfold=3, ncross=3, varmod.method="earth", trace=.3, varmod.minspan=-5)
print(coef(a.varmod.minspan.minus5$varmod))
cat("test varmod.minspan=1\n")
set.seed(1)
a.varmod.minspan1 <- earth(Volume~Girth, data=trees, nfold=3, ncross=3, varmod.method="earth", trace=.3, varmod.minspan=1)
print(coef(a.varmod.minspan1$varmod))

# gam and y.gam are repeated below and on the repeat we will use the mgcv not gam package
use.mgcv.package <- FALSE

for(varmod.method in earth:::VARMOD.METHODS) {

    multifigure(sprintf("varmod.method=\"%s\"", varmod.method), 2, 3)
    par(mar = c(3, 3, 2, 3)) # space for right margin axis

    set.seed(6)
    earth.mod <- earth(Volume~Girth, data=trees, nfold=3, ncross=3,
                       varmod.method=varmod.method,
                       trace=if(varmod.method %in% c("const", "lm", "power")) .3 else 0)
    printh(sprintf("varmod.method %s: summary(earth.mod)", varmod.method))
    printh("summary(earth.mod)")
    print(summary(earth.mod))

    if(use.mgcv.package && (varmod.method == "x.gam" || varmod.method == "gam")) {
        # summary(mgcv) prints environment as hex address which messes up the diffs
        printh("skipping summary(mgcv::gam) etc.\n")
    } else {
        printh("earth.mod$varmod")
        print(earth.mod$varmod, style="unit")

        printh("summary(earth.mod$varmod)")
        print(summary(earth.mod$varmod))

        printh("summary(earth.mod$varmod$residmod)")
        print(summary(earth.mod$varmod$residmod))
    }
    printh(sprintf("varmod.method %s: predict(earth.mod, interval=\"pint\")", varmod.method))
    pints <- predict(earth.mod, interval="pint")
    print(pints)

    plotmo(earth.mod$varmod, do.par=FALSE, col.response=2,
           main="plotmo residual model",
           xlab="x", ylab="varmod residuals")

    plotmo(earth.mod, level=.90, do.par=FALSE, col.response=1,
           main="main model plotmo Girth")
    do.caption()

    plot(earth.mod, which=3, do.par=FALSE, level=.95)

    # plot.varmod
    plot(earth.mod$varmod, do.par=FALSE, which=1:3, info=(varmod.method=="earth"))

    # on second use of gam and y.gam we want to use the mgcv package
    if(varmod.method == "y.gam" && !use.mgcv.package) {
        use.mgcv.package <- TRUE
        detach("package:gam", unload=TRUE)
        library("mgcv")
    }
}
# test varmod.exponent
set.seed(6)
earth.exponent <- earth(Volume~Girth, data=trees, nfold=3, ncross=3,
                        varmod.method="lm", varmod.exponent=.5)
printh("summary(earth.exponent)")
print(summary(earth.exponent))

par(old.par)

if(!interactive()) {
    dev.off()         # finish postscript plot
    q(runLast=FALSE)  # needed else R prints the time on exit (R2.5 and higher) which messes up the diffs
}
