% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/emuR-add_signal.R
\name{add_signalViaMatlab}
\alias{add_signalViaMatlab}
\title{add_signalViaMatlab}
\usage{
add_signalViaMatlab(
  emuDBhandle,
  matlabFunctionName,
  outputFileExtension,
  trackNames,
  trackColumns,
  oneMatlabFunctionCallPerFile = TRUE,
  inputFileExtension = NULL,
  matlabFunctionParameters = NULL,
  paths_to_add = NULL,
  ...
)
}
\arguments{
\item{emuDBhandle}{The Emu database to work on.}

\item{matlabFunctionName}{Name of a Matlab function to use for signal processing.
Must be available on Matlab’s search path; see \code{paths_to_add}.}

\item{outputFileExtension}{The file extension for the new derived signal file
to be created within each bundle.}

\item{trackNames}{The names of the new tracks that will be created automatically.
Should reflect the signal data produced by the Matlab function.}

\item{trackColumns}{The column of data to be used from the result files generated
by Matlab. Each value should usually start with \verb{data[} or \verb{data$}, depending
on the output produced by the Matlab function.}

\item{oneMatlabFunctionCallPerFile}{Whether to call \code{matlabFunctionName} once
per file (TRUE) or once for the entire database (FALSE). \code{FALSE} will
be necessary if you want Matlab to process bundles in parallel.}

\item{inputFileExtension}{The file extension of the files to operate on. Defaults
to the standard media file extension of the current Emu database.}

\item{matlabFunctionParameters}{Data frame with parameters for \code{matlabFunctionName}.
Needs to contain the columns \code{session} and \code{bundle} plus one column for
each function parameter. The column names will be used as parameter names.
Must contain \emph{one row for every bundle, without exception}.}

\item{paths_to_add}{List of paths where Matlab will look for functions. This
will be combined with a path to a small number of Matlab functions
bundled with emuR.}

\item{...}{Other parameters are passed on to \link[matlabr:run_matlab_code]{matlabr::run_matlab_code}.}
}
\description{
Use a Matlab function to derive an extra signal file for each
bundle of the Emu database. A new track definition will be added to the
database automatically.
}
\details{
This function enables EMU-SDMS users you take advantage of tool boxes
and signal processing functions written in Matlab. The Matlab function must meet
certain requirements as detailed below, and it will always be run against the
entire database (either one bundle at a time or the whole database at a time,
but never only a part of the database).

The Matlab function must:
\itemize{
\item Be defined in a file of its own.
\item Accept named parameters.
\item Accept at least the parameters \code{inputFilename} and \code{outputFilename}, both
strings.
\item Use the file at \code{inputFilename} and produce a new file \code{outputFilename};
the new file must be a \code{.mat} file containing the variables \code{data},
\code{sampleRate}, \code{startTime}, \code{units}, and \code{comment}.
}

You can find examples of Matlab functions that meet these requirements by running
\code{\link[=create_emuRdemoData]{create_emuRdemoData()}} and then looking at the subdirectory \verb{add_signal_scripts/matlab/}.

The Matlab function can accept more parameters to influence the signal
processing. These parameters need not be the same values for the entire
database. They can be used, for example, to modify the signal processing
algorithms in a speaker-specific way.

If \code{oneMatlabFunctionCallPerFile} is \code{TRUE}, the function will be called once
for every bundle of the database; in that case, all parameters
to the Matlab function will be 1x1 matrices. If \code{oneMatlabFunctionCallPerFile}
is \code{FALSE}, the Matlab function will only be called once for the entire database;
in that case, all parameters will be 1xN matrices with N equal to the number
of bundles in the database. \code{add_signalViaMatlab} will create a temporary \code{.m}
script. That script may, for example, contain code like this:

\if{html}{\out{<div class="sourceCode matlab">}}\preformatted{demoSignalScalerForOneFile(inputFilename="msajc003.wav", outputFilename="/tmp/RtmpRwjz5Q/add_signalViaMatlab/0fc618dc-8980-414d-8c7a-144a649ce199/0000_ses/msajc003.mat", scalingFactor=1);
demoSignalScalerForOneFile(inputFilename="msajc010.wav", outputFilename="/tmp/RtmpRwjz5Q/add_signalViaMatlab/0fc618dc-8980-414d-8c7a-144a649ce199/0000_ses/msajc010.mat", scalingFactor=4);
}\if{html}{\out{</div>}}

Or like this:

\if{html}{\out{<div class="sourceCode matlab">}}\preformatted{demoSignalScalerForManyFiles(inputFilename=["msajc003.wav", "msajc010.wav",], outputFilename=["/tmp/RtmpRwjz5Q/add_signalViaMatlab/0fc618dc-8980-414d-8c7a-144a649ce199/0000_ses/msajc003.mat", "/tmp/RtmpRwjz5Q/add_signalViaMatlab/0fc618dc-8980-414d-8c7a-144a649ce199/0000_ses/msajc010.mat], scalingFactor=[1, 4]);
}\if{html}{\out{</div>}}

In both cases, \code{scalingFactor} is a parameter that \code{demoSignalScalerForOneFile}
and \code{demoSignalScalerForManyFiles} happen to accept. These are the demo functions
you can find by running \link{create_emuRdemoData}.

The input file will typically be the media file of the bundle, but can be one
of the other files stored in the bundle. If you need that, use the \code{inputFileExtension}
parameter.

The output \code{.mat} files that need to be written by the Matlab function will
be converted – by \code{emuR} – to \code{.Rda} files and saved in each bundle folder with
the file extension \code{outputFileExtension}.

The working directory of the Matlab function will be the same as that of the
current R session, see \code{\link[base:getwd]{base::getwd()}}.

You need a working and licensed Matlab instance on your computer. It will be
called via \code{\link[matlabr:run_matlab_code]{matlabr::run_matlab_code()}}.

Matlab is a trademark of The MathWorks, Inc.
}
\examples{

\dontrun{
###########################
# Setting up some demo data

library(dplyr)
library(ggplot2)
library(emuR)
base_dir = tempdir()
emuR::create_emuRdemoData(base_dir)
emuDBhandle = emuR::load_emuDB(file.path(base_dir,
                                         "emuR_demoData",
                                         "ae_emuDB"))
segmentList = query(emuDBhandle, "Phonetic == ei")

#########################################################
# Calling a Matlab function without additional parameters

add_signalViaMatlab(emuDBhandle = emuDBhandle,
                    matlabFunctionName = "demoSignalScalerForOneFile",
                    outputFileExtension = "sound",
                    trackNames = c("unchangedSound"),
                    trackColumns = c("data[,1]"),
                    paths_to_add = c(file.path(base_dir,
                                               "emuR_demoData",
                                               "add_signal_scripts",
                                               "matlab")))
                    
# paths_to_add tells Matlab where to find the demoSignalScalerForOneFile function.
# This will create a new track definition called unchangedSound. The track’s
# file format will be Rda. All files for this track will have the extension
# .sound and will contain the new signal within the variable data[,1].

list_ssffTrackDefinitions(emuDBhandle)

# The "new" signal will just be a copy of the sound signal, because we have not
# included a scalingFactor parameter. Therefore, demoSignalScalerForOneFile will
# read the wav files and output them mostly unchanged (the values may be on a
# different scale). You can check it like this:

td_media = get_trackdata(emuDBhandle, segmentList, "MEDIAFILE_SAMPLES")
td_new   = get_trackdata(emuDBhandle, segmentList, "unchangedSound")

ggplot(td_media) +
  aes(x = times_rel, y = T1) +
  facet_grid(vars(paste(session, bundle))) +
  geom_line() +
  ggtitle("Three sound signals, original")
ggplot(td_new) +
  aes(x = times_rel, y = T1) +
  facet_grid(vars(paste(session, bundle))) +
  geom_line() +
  ggtitle("Three sound signals, output by Matlab at new scale")
  
# Observe that the two graphs look the same except for the scale.

###########################################
# Calling a Matlab function with parameters

bundleList = 
  emuR::list_bundles(emuDBhandle = emuDBhandle) \%>\% 
  dplyr::rename(bundle = name)
parameterList =
  bundleList \%>\% 
  mutate(scalingFactor = case_match(bundle,
                                    "msajc022" ~ 4,
                                    "msajc023" ~ 2,
                                    .default = 1))
add_signalViaMatlab(emuDBhandle = emuDBhandle,
                    matlabFunctionName = "demoSignalScalerForOneFile",
                    outputFileExtension = "sound2",
                    trackNames = c("scaledSound"),
                    trackColumns = c("data[,1]"),
                    matlabFunctionParameters = parameterList,
                    paths_to_add = c(file.path(base_dir,
                                               "emuR_demoData",
                                               "add_signal_scripts",
                                               "matlab")))

# This will create a new track definition called scaledSound:

list_ssffTrackDefinitions(emuDBhandle)

# The "new" signal will be a copy of the original sound signals, but two bundles
# will be scaled up (multiplied by a given factor). The scaling factor was determined
# through the parameterList data frame, which contained a column scalingFactor.
# If the Matlab function expected other parameters, the data frame would have to
# contain columns accordingly. You can see that two of the bundles have changed
# their scale, but the shape is still the same:

td_media  = get_trackdata(emuDBhandle, segmentList, "MEDIAFILE_SAMPLES")
td_scaled = get_trackdata(emuDBhandle, segmentList, "scaledSound")

ggplot(td_media) +
  aes(x = times_rel, y = T1) +
  facet_grid(vars(paste(session, bundle))) +
  geom_line() +
  ggtitle("Three sound signals, original")
ggplot(td_scaled) +
  aes(x = times_rel, y = T1) +
  facet_grid(vars(paste(session, bundle))) +
  geom_line() +
  ggtitle("Three sound signals, with different scaling factors applied")
}


}
