# WARNING - Generated by {fusen} from dev/flat_sepuku.Rmd: do not edit by hand # nolint: line_length_linter.

#' Clean a package from fusen-related files and tags
#'
#' This function will delete all the flat files (i.e files listed in the fusen configuration file, as well as Rmd or qmd files starting with "flat" in the "dev/" and "dev/flat_history" folders). It will also remove the fusen-related tags added by calls to `fusen::inflate()` in files located in the "R/", "tests/" and "vignettes/" folders. Finally, it will also remove the fusen configuration file if it exists.
#'
#' @param pkg Character. Path of the current package
#' @param force logical. whether to force the cleaning or not, without asking if the user is sure about making this operation (default: FALSE)
#' @param verbose logical. whether to display the files that will be deleted or modified (default: FALSE)
#' @return side effect. A package cleaned from fusen-related files or tags
#' @importFrom cli cli_alert_info cli_alert_danger cli_abort cli_alert_success
#' @export
#' @examples
#' \dontrun{
#' sepuku()
#' # If you want to force the cleaning, you can use the force argument
#' sepuku(force = TRUE)
#'
#' # Example with a dummy package
#' dummypackage <- tempfile("sepuku.example")
#' dir.create(dummypackage)
#' fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
#'
#' usethis::with_project(dummypackage, {
#'   # Add licence
#'   usethis::use_mit_license("John Doe")
#'
#'   dir.create(file.path(dummypackage, "dev"))
#'   dir.create(file.path(dummypackage, "dev", "flat_history"))
#'
#'   # We add 2 flat files in the package and inflate them
#'   dev_file1 <- add_minimal_flat(
#'     pkg = dummypackage,
#'     flat_name = "flat1.Rmd",
#'     open = FALSE
#'   )
#'
#'   dev_file2 <- add_minimal_flat(
#'     pkg = dummypackage,
#'     flat_name = "flat2.Rmd",
#'     open = FALSE
#'   )
#'
#'   inflate(
#'     pkg = dummypackage,
#'     flat_file = dev_file1,
#'     vignette_name = "Get started",
#'     check = FALSE,
#'     open_vignette = FALSE,
#'     document = TRUE,
#'     overwrite = "yes"
#'   )
#'
#'   inflate(
#'     pkg = dummypackage,
#'     flat_file = dev_file2,
#'     vignette_name = "Get started 2",
#'     check = FALSE,
#'     open_vignette = FALSE,
#'     document = TRUE,
#'     overwrite = "yes"
#'   )
#'
#'   # We deprecate the first flat file, which will be moved to the flat_history folder
#'   deprecate_flat_file(
#'     file.path(dummypackage, "dev", "flat_flat1.Rmd")
#'   )
#'
#'   # We create 2 flat files with the qmd extension
#'   file.create(file.path(dummypackage, "dev", "flat_history", "flat_old.qmd"))
#'   file.create(file.path(dummypackage, "dev", "flat_qmd.qmd"))
#'
#'   sepuku(force = TRUE)
#'
#'   # We check that the fusen configuration file has been deleted
#'   file.exists(
#'     file.path(dummypackage, "dev", "config_fusen.yaml")
#'   )
#'
#'   # We check that all the flat files have been deleted
#'   length(
#'     list.files(
#'       file.path(dummypackage, "dev"),
#'       pattern = "^flat.*\\.Rmd"
#'     )
#'   )
#'
#'   length(
#'     list.files(
#'       file.path(dummypackage, "dev"),
#'       pattern = "^flat.*\\.qmd"
#'     )
#'   )
#'
#'
#'   length(
#'     list.files(
#'       file.path(dummypackage, "dev", "flat_history"),
#'       pattern = "^flat.*\\.Rmd"
#'     )
#'   )
#'
#'
#'   length(
#'     list.files(
#'       file.path(dummypackage, "dev", "flat_history"),
#'       pattern = "^flat.*\\.qmd"
#'     )
#'   )
#'
#'   # We check that all the files with fusen tags have been cleaned
#'   length(fusen:::find_files_with_fusen_tags(pkg = dummypackage))
#' })
#'
#' # Clean the temporary directory
#' unlink(dummypackage, recursive = TRUE)
#' }
sepuku <- function(
  pkg = ".",
  force = FALSE,
  verbose = FALSE
) {
  if (!dir.exists(file.path(pkg, "dev"))) {
    cli_abort("No dev/ folder have been found. Are you sure that your package has been initiated with fusen ?")
  }

  config_file <- getOption("fusen.config_file", default = "dev/config_fusen.yaml")

  if (!file.exists(config_file)) {
    cli_alert_info("No fusen configuration file found. The flat files to be deleted will be identified as Rmd or qmd files starting with 'flat' in the dev/ and dev/flat_history folders.")
  } else {
    cli_alert_info("A fusen configuration file was found. The flat files to be deleted will be identified as files listed in this configuration file as well as Rmd or qmd files starting with 'flat' in the dev/ and dev/flat_history folders. The configuration file will also be deleted.")
  }

  flat_files <- list_flat_files(pkg = pkg)

  if (length(flat_files) == 0) {
    cli_alert_info("No flat files were detected.")
  } else {
    if (verbose) {
      cli_alert_info(
        paste0(
          "The following flat files were detected and will therefore be deleted from your package:\n",
          paste0(flat_files, collapse = "\n")
        )
      )
    }
  }

  files_to_be_modified <- find_files_with_fusen_tags(pkg = pkg)
  if (length(files_to_be_modified) == 0) {
    cli_alert_info("No fusen-related tags have been found in any files located in R/, tests/ and vignettes/ folders.")
  } else {
    if (verbose) {
      cli_alert_info(
        paste0(
          "The following files have been identified as containing fusen-related tags and will therefore be modified:\n",
          paste0(files_to_be_modified, collapse = "\n")
        )
      )
    }
  }

  if (length(flat_files) == 0 && length(files_to_be_modified) == 0) {
    return(invisible(NULL))
  }

  do_it <- force

  if (!force) {
    cli_alert_danger("Some files are about to be deleted or modified. This operation is irreversible.")
    clean <- readline(
      prompt = "Are you sure of what you are doing? (y/n)"
    )
    if (tolower(clean) %in% c("yes", "y")) {
      do_it <- TRUE
    } else if (tolower(clean) %in% c("no", "n")) {
      do_it <- FALSE
    } else {
      stop("clean should be TRUE, FALSE, 'yes'or 'no'")
    }
  }

  if (isTRUE(do_it)) {
    if (length(flat_files) > 0) {
      invisible(
        lapply(
          flat_files,
          function(f) {
            file.remove(file.path(pkg, f))
          }
        )
      )
      try(
        unlink(
          file.path(
            pkg,
            "dev",
            "flat_history"
          ),
          recursive = TRUE
        ),
        silent = TRUE
      )
    }

    if (length(files_to_be_modified) > 0) {
      invisible(
        lapply(
          files_to_be_modified,
          function(f) {
            clean_fusen_tags_in_files(pkg = pkg, files_to_clean = f)
          }
        )
      )
    }
    if (file.exists(config_file)) {
      file.remove(config_file)
    }
    cli_alert_success("Cleaning is done !")
  }
  return(invisible(TRUE))
}
