/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5ACprivate.h" 
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5Gpkg.h"      
#include "H5HLprivate.h" 
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5Ppublic.h"   

typedef struct H5G_node_key_t {
    size_t offset; 
} H5G_node_key_t;

static H5UC_t   *H5G__node_get_shared(const H5F_t *f, const void *_udata);
static herr_t    H5G__node_create(H5F_t *f, H5B_ins_t op, void *_lt_key, void *_udata, void *_rt_key,
                                  haddr_t *addr_p );
static int       H5G__node_cmp2(void *_lt_key, void *_udata, void *_rt_key);
static int       H5G__node_cmp3(void *_lt_key, void *_udata, void *_rt_key);
static herr_t    H5G__node_found(H5F_t *f, haddr_t addr, const void *_lt_key, bool *found, void *_udata);
static H5B_ins_t H5G__node_insert(H5F_t *f, haddr_t addr, void *_lt_key, bool *lt_key_changed, void *_md_key,
                                  void *_udata, void *_rt_key, bool *rt_key_changed,
                                  haddr_t *new_node_p );
static H5B_ins_t H5G__node_remove(H5F_t *f, haddr_t addr, void *lt_key, bool *lt_key_changed, void *udata,
                                  void *rt_key, bool *rt_key_changed);
static herr_t    H5G__node_decode_key(const H5B_shared_t *shared, const uint8_t *raw, void *_key);
static herr_t    H5G__node_encode_key(const H5B_shared_t *shared, uint8_t *raw, const void *_key);
static herr_t H5G__node_debug_key(FILE *stream, int indent, int fwidth, const void *key, const void *udata);

H5B_class_t H5B_SNODE[1] = {{
    H5B_SNODE_ID,           
    sizeof(H5G_node_key_t), 
    H5G__node_get_shared,   
    H5G__node_create,       
    H5G__node_cmp2,         
    H5G__node_cmp3,         
    H5G__node_found,        
    H5G__node_insert,       
    true,                   
    true,                   
    H5B_RIGHT,              
    H5G__node_remove,       
    H5G__node_decode_key,   
    H5G__node_encode_key,   
    H5G__node_debug_key     
}};

H5FL_DEFINE(H5G_node_t);

H5FL_SEQ_DEFINE(H5G_entry_t);

static H5UC_t *
H5G__node_get_shared(const H5F_t *f, const void H5_ATTR_UNUSED *_udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(f);

    
    FUNC_LEAVE_NOAPI(H5F_GRP_BTREE_SHARED(f))
} 

static herr_t
H5G__node_decode_key(const H5B_shared_t *shared, const uint8_t *raw, void *_key)
{
    H5G_node_key_t *key = (H5G_node_key_t *)_key;

    FUNC_ENTER_PACKAGE_NOERR

    assert(shared);
    assert(raw);
    assert(key);

    H5_DECODE_LENGTH_LEN(raw, key->offset, shared->sizeof_len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5G__node_encode_key(const H5B_shared_t *shared, uint8_t *raw, const void *_key)
{
    const H5G_node_key_t *key = (const H5G_node_key_t *)_key;

    FUNC_ENTER_PACKAGE_NOERR

    assert(shared);
    assert(raw);
    assert(key);

    H5_ENCODE_LENGTH_LEN(raw, key->offset, shared->sizeof_len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5G__node_debug_key(FILE *stream, int indent, int fwidth, const void *_key, const void *_udata)
{
    const H5G_node_key_t  *key   = (const H5G_node_key_t *)_key;
    const H5G_bt_common_t *udata = (const H5G_bt_common_t *)_udata;

    FUNC_ENTER_PACKAGE_NOERR

    assert(key);

    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Heap offset:", (unsigned)key->offset);

    if (udata->heap) {
        const char *s;

        Rfprintf(stream, "%*s%-*s ", indent, "", fwidth, "Name:");

        if ((s = (const char *)H5HL_offset_into(udata->heap, key->offset)) != NULL)
            Rfprintf(stream, "%s\n", s);
    } 
    else
        Rfprintf(stream, "%*s%-*s ", indent, "", fwidth, "Cannot get name; heap address not specified\n");

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5G__node_free(H5G_node_t *sym)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(sym);

    
    assert(sym->cache_info.is_dirty == false);

    if (sym->entry)
        sym->entry = H5FL_SEQ_FREE(H5G_entry_t, sym->entry);
    sym = H5FL_FREE(H5G_node_t, sym);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5G__node_create(H5F_t *f, H5B_ins_t H5_ATTR_UNUSED op, void *_lt_key, void H5_ATTR_UNUSED *_udata,
                 void *_rt_key, haddr_t *addr_p )
{
    H5G_node_key_t *lt_key    = (H5G_node_key_t *)_lt_key;
    H5G_node_key_t *rt_key    = (H5G_node_key_t *)_rt_key;
    H5G_node_t     *sym       = NULL;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5B_INS_FIRST == op);

    if (NULL == (sym = H5FL_CALLOC(H5G_node_t)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTALLOC, FAIL, "memory allocation failed");
    sym->node_size = H5G_NODE_SIZE(f);
    if (HADDR_UNDEF == (*addr_p = H5MF_alloc(f, H5FD_MEM_BTREE, (hsize_t)sym->node_size)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to allocate file space");
    if (NULL == (sym->entry = H5FL_SEQ_CALLOC(H5G_entry_t, (size_t)(2 * H5F_SYM_LEAF_K(f)))))
        HGOTO_ERROR(H5E_SYM, H5E_CANTALLOC, FAIL, "memory allocation failed");

    if (H5AC_insert_entry(f, H5AC_SNODE, *addr_p, sym, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to cache symbol table leaf node");
    
    if (lt_key)
        lt_key->offset = 0;
    if (rt_key)
        rt_key->offset = 0;

done:
    if (ret_value < 0)
        if (sym != NULL) {
            if (sym->entry != NULL)
                sym->entry = H5FL_SEQ_FREE(H5G_entry_t, sym->entry);
            sym = H5FL_FREE(H5G_node_t, sym);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__node_cmp2(void *_lt_key, void *_udata, void *_rt_key)
{
    H5G_bt_common_t *udata  = (H5G_bt_common_t *)_udata;
    H5G_node_key_t  *lt_key = (H5G_node_key_t *)_lt_key;
    H5G_node_key_t  *rt_key = (H5G_node_key_t *)_rt_key;
    const char      *s1, *s2;
    size_t           max_len;
    int              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(udata && udata->heap);
    assert(lt_key);
    assert(rt_key);

    
    if ((s1 = (const char *)H5HL_offset_into(udata->heap, lt_key->offset)) == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get key name");
    if ((s2 = (const char *)H5HL_offset_into(udata->heap, rt_key->offset)) == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get key name");

    
    if (rt_key->offset > lt_key->offset)
        max_len = udata->block_size - rt_key->offset;
    else
        max_len = udata->block_size - lt_key->offset;

    
    ret_value = strncmp(s1, s2, max_len);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__node_cmp3(void *_lt_key, void *_udata, void *_rt_key)
{
    H5G_bt_common_t *udata  = (H5G_bt_common_t *)_udata;
    H5G_node_key_t  *lt_key = (H5G_node_key_t *)_lt_key;
    H5G_node_key_t  *rt_key = (H5G_node_key_t *)_rt_key;
    const char      *s;
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(udata && udata->heap);
    assert(lt_key);
    assert(rt_key);

    
    if ((s = (const char *)H5HL_offset_into(udata->heap, lt_key->offset)) == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get key name");
    if (strncmp(udata->name, s, (udata->block_size - lt_key->offset)) <= 0)
        ret_value = (-1);
    else {
        
        if ((s = (const char *)H5HL_offset_into(udata->heap, rt_key->offset)) == NULL)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get key name");
        if (strncmp(udata->name, s, (udata->block_size - rt_key->offset)) > 0)
            ret_value = 1;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__node_found(H5F_t *f, haddr_t addr, const void H5_ATTR_UNUSED *_lt_key, bool *found, void *_udata)
{
    H5G_bt_lkp_t *udata = (H5G_bt_lkp_t *)_udata;
    H5G_node_t   *sn    = NULL;
    unsigned      lt = 0, idx = 0, rt;
    int           cmp = 1;
    const char   *s;
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(found);
    assert(udata && udata->common.heap);

    
    if (NULL == (sn = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, addr, f, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table node");

    
    rt = sn->nsyms;
    while (lt < rt && cmp) {
        idx = (lt + rt) / 2;

        if ((s = (const char *)H5HL_offset_into(udata->common.heap, sn->entry[idx].name_off)) == NULL)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get symbol table name");
        cmp = strncmp(udata->common.name, s, (udata->common.block_size - sn->entry[idx].name_off));

        if (cmp < 0)
            rt = idx;
        else
            lt = idx + 1;
    } 

    if (cmp)
        *found = false;
    else {
        
        *found = true;

        
        if ((udata->op)(&sn->entry[idx], udata->op_data) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "iterator callback failed");
    } 

done:
    if (sn && H5AC_unprotect(f, H5AC_SNODE, addr, sn, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to release symbol table node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5B_ins_t
H5G__node_insert(H5F_t *f, haddr_t addr, void H5_ATTR_UNUSED *_lt_key, bool H5_ATTR_UNUSED *lt_key_changed,
                 void *_md_key, void *_udata, void *_rt_key, bool *rt_key_changed, haddr_t *new_node_p)
{
    H5G_node_key_t *md_key = (H5G_node_key_t *)_md_key;
    H5G_node_key_t *rt_key = (H5G_node_key_t *)_rt_key;
    H5G_bt_ins_t   *udata  = (H5G_bt_ins_t *)_udata;
    H5G_node_t     *sn = NULL, *snrt = NULL;
    unsigned        sn_flags = H5AC__NO_FLAGS_SET, snrt_flags = H5AC__NO_FLAGS_SET;
    const char     *s;
    unsigned        lt  = 0, rt; 
    int             cmp = 1, idx = -1;
    H5G_node_t     *insert_into = NULL; 
    H5G_entry_t     ent;                
    H5B_ins_t       ret_value = H5B_INS_ERROR;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(md_key);
    assert(rt_key);
    assert(udata && udata->common.heap);
    assert(new_node_p);

    
    if (NULL == (sn = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, addr, f, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, H5B_INS_ERROR, "unable to protect symbol table node");

    
    rt = sn->nsyms;
    while (lt < rt) {
        idx = (int)((lt + rt) / 2);
        if ((s = (const char *)H5HL_offset_into(udata->common.heap, sn->entry[idx].name_off)) == NULL)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5B_INS_ERROR, "unable to get symbol table name");

        
        if (0 == (cmp = strncmp(udata->common.name, s, (udata->common.block_size - sn->entry[idx].name_off))))
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, H5B_INS_ERROR, "symbol is already present in symbol table");

        if (cmp < 0)
            rt = (unsigned)idx;
        else
            lt = (unsigned)(idx + 1);
    } 
    idx += cmp > 0 ? 1 : 0;

    
    if (H5G__link_to_ent(f, udata->common.heap, udata->lnk, udata->obj_type, udata->crt_info, &ent) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCONVERT, H5B_INS_ERROR, "unable to convert link");

    
    if (sn->nsyms >= 2 * H5F_SYM_LEAF_K(f)) {
        
        ret_value = H5B_INS_RIGHT;

        
        if (H5G__node_create(f, H5B_INS_FIRST, NULL, NULL, NULL, new_node_p ) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, H5B_INS_ERROR, "unable to split symbol table node");

        if (NULL == (snrt = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, *new_node_p, f, H5AC__NO_FLAGS_SET)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, H5B_INS_ERROR, "unable to split symbol table node");

        H5MM_memcpy(snrt->entry, sn->entry + H5F_SYM_LEAF_K(f), H5F_SYM_LEAF_K(f) * sizeof(H5G_entry_t));
        snrt->nsyms = H5F_SYM_LEAF_K(f);
        snrt_flags |= H5AC__DIRTIED_FLAG;

        
        memset(sn->entry + H5F_SYM_LEAF_K(f), 0, H5F_SYM_LEAF_K(f) * sizeof(H5G_entry_t));
        sn->nsyms = H5F_SYM_LEAF_K(f);
        sn_flags |= H5AC__DIRTIED_FLAG;

        
        md_key->offset = sn->entry[sn->nsyms - 1].name_off;

        
        if (idx <= (int)H5F_SYM_LEAF_K(f)) {
            insert_into = sn;
            if (idx == (int)H5F_SYM_LEAF_K(f))
                md_key->offset = ent.name_off;
        } 
        else {
            idx -= (int)H5F_SYM_LEAF_K(f);
            insert_into = snrt;
            if (idx == (int)H5F_SYM_LEAF_K(f)) {
                rt_key->offset  = ent.name_off;
                *rt_key_changed = true;
            } 
        }     
    }         
    else {
        
        ret_value = H5B_INS_NOOP;
        sn_flags |= H5AC__DIRTIED_FLAG;
        insert_into = sn;
        if (idx == (int)sn->nsyms) {
            rt_key->offset  = ent.name_off;
            *rt_key_changed = true;
        } 
    }     

    
    assert(idx >= 0);
    memmove(insert_into->entry + idx + 1, insert_into->entry + idx,
            (insert_into->nsyms - (unsigned)idx) * sizeof(H5G_entry_t));

    
    H5G__ent_copy(&(insert_into->entry[idx]), &ent, H5_COPY_SHALLOW);

    
    insert_into->nsyms += 1;

done:
    if (snrt && H5AC_unprotect(f, H5AC_SNODE, *new_node_p, snrt, snrt_flags) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, H5B_INS_ERROR, "unable to release symbol table node");
    if (sn && H5AC_unprotect(f, H5AC_SNODE, addr, sn, sn_flags) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, H5B_INS_ERROR, "unable to release symbol table node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5B_ins_t
H5G__node_remove(H5F_t *f, haddr_t addr, void H5_ATTR_NDEBUG_UNUSED *_lt_key ,
                 bool H5_ATTR_UNUSED *lt_key_changed , void *_udata ,
                 void *_rt_key , bool *rt_key_changed )
{
    H5G_node_key_t *rt_key   = (H5G_node_key_t *)_rt_key;
    H5G_bt_rm_t    *udata    = (H5G_bt_rm_t *)_udata;
    H5G_node_t     *sn       = NULL;
    unsigned        sn_flags = H5AC__NO_FLAGS_SET;
    unsigned        lt = 0, rt, idx = 0;
    int             cmp       = 1;
    H5B_ins_t       ret_value = H5B_INS_ERROR;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert((H5G_node_key_t *)_lt_key);
    assert(rt_key);
    assert(udata && udata->common.heap);

    
    if (NULL == (sn = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, addr, f, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, H5B_INS_ERROR, "unable to protect symbol table node");

    
    if (udata->common.name != NULL) {
        H5O_link_t lnk;           
        size_t     link_name_len; 

        
        rt = sn->nsyms;
        while (lt < rt && cmp) {
            const char *s; 

            idx = (lt + rt) / 2;
            if ((s = (const char *)H5HL_offset_into(udata->common.heap, sn->entry[idx].name_off)) == NULL)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5B_INS_ERROR, "unable to get symbol table name");
            cmp = strncmp(udata->common.name, s, (udata->common.block_size - sn->entry[idx].name_off));
            if (cmp < 0)
                rt = idx;
            else
                lt = idx + 1;
        } 

        if (cmp)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, H5B_INS_ERROR, "name not found");

        
        if (NULL == (lnk.name = (char *)H5HL_offset_into(udata->common.heap, sn->entry[idx].name_off)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5B_INS_ERROR, "unable to get link name");

        
        link_name_len = strnlen(lnk.name, (udata->common.block_size - sn->entry[idx].name_off)) + 1;
        if (link_name_len > (udata->common.block_size - sn->entry[idx].name_off))
            link_name_len = (udata->common.block_size - sn->entry[idx].name_off);

        
        lnk.corder_valid = false;
        lnk.corder       = 0;
        lnk.cset         = H5T_CSET_ASCII;
        if (sn->entry[idx].type == H5G_CACHED_SLINK) {
            lnk.type = H5L_TYPE_SOFT;
            if (NULL == (lnk.u.soft.name = (char *)H5HL_offset_into(udata->common.heap,
                                                                    sn->entry[idx].cache.slink.lval_offset)))
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5B_INS_ERROR, "unable to get link name");
        } 
        else {
            lnk.type = H5L_TYPE_HARD;
            assert(H5_addr_defined(sn->entry[idx].header));
            lnk.u.hard.addr = sn->entry[idx].header;
        } 

        
        if (H5G__link_name_replace(f, udata->grp_full_path_r, &lnk) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5B_INS_ERROR, "unable to get object type");

        
        if (lnk.type == H5L_TYPE_HARD) {
            H5O_loc_t tmp_oloc; 

            
            tmp_oloc.file = f;
            tmp_oloc.addr = lnk.u.hard.addr;

            if (H5O_link(&tmp_oloc, -1) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, H5B_INS_ERROR, "unable to decrement object link count");
        } 
        else {
            
            if (lnk.u.soft.name) {
                size_t soft_link_len; 

                
                soft_link_len = strnlen(lnk.u.soft.name,
                                        (udata->common.block_size - sn->entry[idx].cache.slink.lval_offset)) +
                                1;
                if (soft_link_len > (udata->common.block_size - sn->entry[idx].cache.slink.lval_offset))
                    soft_link_len = (udata->common.block_size - sn->entry[idx].cache.slink.lval_offset);

                if (H5HL_remove(f, udata->common.heap, sn->entry[idx].cache.slink.lval_offset,
                                soft_link_len) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, H5B_INS_ERROR,
                                "unable to remove soft link from local heap");
            } 
        }     

        
        if (H5HL_remove(f, udata->common.heap, sn->entry[idx].name_off, link_name_len) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, H5B_INS_ERROR, "unable to remove link name from local heap");

        
        if (1 == sn->nsyms) {
            
            assert(0 == idx);
            sn->nsyms = 0;
            sn_flags |= H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;
            ret_value = H5B_INS_REMOVE;
        }
        else if (0 == idx) {
            
            sn->nsyms -= 1;
            sn_flags |= H5AC__DIRTIED_FLAG;
            memmove(sn->entry + idx, sn->entry + idx + 1, (sn->nsyms - idx) * sizeof(H5G_entry_t));
            ret_value = H5B_INS_NOOP;
        }
        else if (idx + 1 == sn->nsyms) {
            
            sn->nsyms -= 1;
            sn_flags |= H5AC__DIRTIED_FLAG;
            rt_key->offset  = sn->entry[sn->nsyms - 1].name_off;
            *rt_key_changed = true;
            ret_value       = H5B_INS_NOOP;
        }
        else {
            
            sn->nsyms -= 1;
            sn_flags |= H5AC__DIRTIED_FLAG;
            memmove(sn->entry + idx, sn->entry + idx + 1, (sn->nsyms - idx) * sizeof(H5G_entry_t));
            ret_value = H5B_INS_NOOP;
        } 
    }     
    
    else {
        H5O_loc_t tmp_oloc; 

        
        tmp_oloc.file = f;

        
        for (idx = 0; idx < sn->nsyms; idx++)
            if (!(H5G_CACHED_SLINK == sn->entry[idx].type)) {
                
                assert(H5_addr_defined(sn->entry[idx].header));
                tmp_oloc.addr = sn->entry[idx].header;

                if (H5O_link(&tmp_oloc, -1) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, H5B_INS_ERROR,
                                "unable to decrement object link count");
            } 

        
        sn->nsyms = 0;
        sn_flags |= H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;
        ret_value = H5B_INS_REMOVE;
    } 

done:
    if (sn && H5AC_unprotect(f, H5AC_SNODE, addr, sn, sn_flags) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, H5B_INS_ERROR, "unable to release symbol table node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5G__node_iterate(H5F_t *f, const void H5_ATTR_UNUSED *_lt_key, haddr_t addr,
                  const void H5_ATTR_UNUSED *_rt_key, void *_udata)
{
    H5G_bt_it_it_t *udata = (H5G_bt_it_it_t *)_udata;
    H5G_node_t     *sn    = NULL;
    H5G_entry_t    *ents; 
    unsigned        u;    
    int             ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(udata && udata->heap);

    
    if (NULL == (sn = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, addr, f, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, H5_ITER_ERROR, "unable to load symbol table node");

    
    for (u = 0, ents = sn->entry; u < sn->nsyms && ret_value == H5_ITER_CONT; u++) {
        if (udata->skip > 0)
            --udata->skip;
        else {
            H5O_link_t lnk; 

            
            if (H5G__ent_to_link(&ents[u], udata->heap, &lnk) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTCONVERT, H5_ITER_ERROR,
                            "unable to convert symbol table entry to link");

            
            ret_value = (udata->op)(&lnk, udata->op_data);

            
            if (H5O_msg_reset(H5O_LINK_ID, &lnk) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTFREE, H5_ITER_ERROR, "unable to release link message");
        } 

        
        
        if (udata->final_ent)
            (*udata->final_ent)++;
    } 
    if (ret_value < 0)
        HERROR(H5E_SYM, H5E_CANTNEXT, "iteration operator failed");

done:
    
    if (sn && H5AC_unprotect(f, H5AC_SNODE, addr, sn, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, H5_ITER_ERROR, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5G__node_sumup(H5F_t *f, const void H5_ATTR_UNUSED *_lt_key, haddr_t addr,
                const void H5_ATTR_UNUSED *_rt_key, void *_udata)
{
    hsize_t    *num_objs  = (hsize_t *)_udata;
    H5G_node_t *sn        = NULL;
    int         ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(num_objs);

    
    if (NULL == (sn = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, addr, f, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, H5_ITER_ERROR, "unable to load symbol table node");

    *num_objs += sn->nsyms;

done:
    if (sn && H5AC_unprotect(f, H5AC_SNODE, addr, sn, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, H5_ITER_ERROR, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5G__node_by_idx(H5F_t *f, const void H5_ATTR_UNUSED *_lt_key, haddr_t addr,
                 const void H5_ATTR_UNUSED *_rt_key, void *_udata)
{
    H5G_bt_it_idx_common_t *udata     = (H5G_bt_it_idx_common_t *)_udata;
    H5G_node_t             *sn        = NULL;
    int                     ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(udata);

    
    if (NULL == (sn = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, addr, f, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, H5_ITER_ERROR, "unable to load symbol table node");

    
    if (udata->idx >= udata->num_objs && udata->idx < (udata->num_objs + sn->nsyms)) {
        hsize_t ent_idx; 

        
        ent_idx = udata->idx - udata->num_objs;

        
        assert(udata->op);
        if ((udata->op)(&sn->entry[ent_idx], udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5B_INS_ERROR, "'by index' callback failed");

        
        ret_value = H5_ITER_STOP;
    } 
    else
        udata->num_objs += sn->nsyms;

done:
    if (sn && H5AC_unprotect(f, H5AC_SNODE, addr, sn, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, H5_ITER_ERROR, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__node_init(H5F_t *f)
{
    H5B_shared_t *shared;              
    size_t        sizeof_rkey;         
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);

    
    sizeof_rkey = H5F_SIZEOF_SIZE(f); 

    
    if (NULL == (shared = H5B_shared_new(f, H5B_SNODE, sizeof_rkey)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "memory allocation failed for shared B-tree info");

    
    

    
    if (H5F_SET_GRP_BTREE_SHARED(f, H5UC_create(shared, H5B_shared_free)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create ref-count wrapper for shared B-tree info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_node_close(const H5F_t *f)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(f);

    
    if (H5F_GRP_BTREE_SHARED(f))
        H5UC_DEC(H5F_GRP_BTREE_SHARED(f));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

int
H5G__node_copy(H5F_t *f, const void H5_ATTR_UNUSED *_lt_key, haddr_t addr, const void H5_ATTR_UNUSED *_rt_key,
               void *_udata)
{
    H5G_bt_it_cpy_t *udata    = (H5G_bt_it_cpy_t *)_udata;
    const H5O_loc_t *src_oloc = udata->src_oloc;
    H5O_copy_t      *cpy_info = udata->cpy_info;
    H5G_node_t      *sn       = NULL;
    unsigned int     i; 
    int              ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(udata);
    assert(udata->src_heap);

    
    if (NULL == (sn = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, addr, f, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, H5_ITER_ERROR, "unable to load symbol table node");

    
    for (i = 0; i < sn->nsyms; i++) {
        H5G_entry_t *src_ent =
            &(sn->entry[i]);             
        H5O_link_t          lnk;         
        char               *name;        
        H5G_entry_t         tmp_src_ent; 
        H5O_type_t          obj_type = H5O_TYPE_UNKNOWN; 
        H5G_copy_file_ud_t *cpy_udata;                   
        H5G_obj_create_t    gcrt_info;                   
        size_t              max_link_len;                

        
        if (H5G_CACHED_SLINK == src_ent->type && cpy_info->expand_soft_link) {
            haddr_t    obj_addr = HADDR_UNDEF; 
            H5G_loc_t  grp_loc;                
            H5G_name_t grp_path;               
            char      *link_name;              

            
            H5MM_memcpy(&tmp_src_ent, src_ent, sizeof(H5G_entry_t));

            
            H5G_name_reset(&grp_path);
            grp_loc.path = &grp_path;
            H5_WARN_CAST_AWAY_CONST_OFF
            grp_loc.oloc = (H5O_loc_t *)src_oloc;
            H5_WARN_CAST_AWAY_CONST_ON

            
            if ((link_name =
                     (char *)H5HL_offset_into(udata->src_heap, tmp_src_ent.cache.slink.lval_offset)) == NULL)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5_ITER_ERROR, "unable to get link name");

            
            max_link_len = udata->src_block_size - tmp_src_ent.cache.slink.lval_offset;
            if (strnlen(link_name, max_link_len) == max_link_len)
                HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, H5_ITER_ERROR, "invalid link name offset");

            
            if (H5G__loc_addr(&grp_loc, link_name, &obj_addr) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTFIND, H5_ITER_ERROR,
                            "unable to check if soft link resolves to an object");
            if (H5_addr_defined(obj_addr)) {
                tmp_src_ent.header = obj_addr;
                src_ent            = &tmp_src_ent;
            } 
        }     

        
        if (H5_addr_defined(src_ent->header)) {
            H5O_loc_t new_dst_oloc; 
            H5O_loc_t tmp_src_oloc; 

            
            H5O_loc_reset(&new_dst_oloc);
            new_dst_oloc.file = udata->dst_file;

            
            H5O_loc_reset(&tmp_src_oloc);
            tmp_src_oloc.file = f;
            tmp_src_oloc.addr = src_ent->header;

            
            if (H5O_copy_header_map(&tmp_src_oloc, &new_dst_oloc, cpy_info, true, &obj_type,
                                    (void **)&cpy_udata) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, H5_ITER_ERROR, "unable to copy object");

            
            if (obj_type == H5O_TYPE_GROUP) {
                gcrt_info.gcpl_id    = H5P_DEFAULT;
                gcrt_info.cache_type = cpy_udata->cache_type;
                gcrt_info.cache      = cpy_udata->cache;
            } 

            
            lnk.type        = H5L_TYPE_HARD;
            lnk.u.hard.addr = new_dst_oloc.addr;
        } 
        else if (H5G_CACHED_SLINK == src_ent->type) {
            
            
            obj_type = H5O_TYPE_UNKNOWN;

            
            lnk.type = H5L_TYPE_SOFT;
            if ((lnk.u.soft.name =
                     (char *)H5HL_offset_into(udata->src_heap, src_ent->cache.slink.lval_offset)) == NULL)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5_ITER_ERROR, "unable to get link name");

            
            max_link_len = udata->src_block_size - src_ent->cache.slink.lval_offset;
            if (strnlen(lnk.u.soft.name, max_link_len) == max_link_len)
                HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, H5_ITER_ERROR, "invalid link name offset");
        } 
        else
            assert(0 && "Unknown entry type");

        
        if ((name = (char *)H5HL_offset_into(udata->src_heap, src_ent->name_off)) == NULL)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5_ITER_ERROR, "unable to get source object name");

        
        max_link_len = udata->src_block_size - src_ent->name_off;
        if (strnlen(name, max_link_len) == max_link_len)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, H5_ITER_ERROR, "invalid link name offset");

        
        lnk.cset         = H5F_DEFAULT_CSET; 
        lnk.corder       = 0;                
        lnk.corder_valid = false;            
        lnk.name         = name;             

        
        H5_BEGIN_TAG(H5AC__COPIED_TAG)

        
        
        if (H5G__stab_insert_real(udata->dst_file, udata->dst_stab, &lnk, obj_type,
                                  (obj_type == H5O_TYPE_GROUP ? &gcrt_info : NULL)) < 0)
            HGOTO_ERROR_TAG(H5E_DATATYPE, H5E_CANTINIT, H5_ITER_ERROR, "unable to insert the name");

        
        H5_END_TAG

    } 

done:
    if (sn && H5AC_unprotect(f, H5AC_SNODE, addr, sn, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, H5_ITER_ERROR, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5G__node_build_table(H5F_t *f, const void H5_ATTR_UNUSED *_lt_key, haddr_t addr,
                      const void H5_ATTR_UNUSED *_rt_key, void *_udata)
{
    H5G_bt_it_bt_t *udata = (H5G_bt_it_bt_t *)_udata;
    H5G_node_t     *sn    = NULL; 
    unsigned        u;            
    int             ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(udata && udata->heap);

    
    if (NULL == (sn = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, addr, f, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, H5_ITER_ERROR, "unable to load symbol table node");

    
    if ((udata->ltable->nlinks + sn->nsyms) >= udata->alloc_nlinks) {
        size_t      na = MAX((udata->ltable->nlinks + sn->nsyms),
                             (udata->alloc_nlinks * 2)); 
        H5O_link_t *x;                                   

        
        if (NULL == (x = (H5O_link_t *)H5MM_realloc(udata->ltable->lnks, sizeof(H5O_link_t) * na)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTALLOC, H5_ITER_ERROR, "memory allocation failed");
        udata->ltable->lnks = x;
    } 

    
    for (u = 0; u < sn->nsyms; u++) {
        size_t linkno; 

        
        linkno = udata->ltable->nlinks++;

        
        if (H5G__ent_to_link(&sn->entry[u], udata->heap, &udata->ltable->lnks[linkno]) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTCONVERT, H5_ITER_ERROR,
                        "unable to convert symbol table entry to link");
    } 

done:
    
    if (sn && H5AC_unprotect(f, H5AC_SNODE, addr, sn, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, H5_ITER_ERROR, "unable to release object header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__node_iterate_size(H5F_t *f, const void H5_ATTR_UNUSED *_lt_key, haddr_t H5_ATTR_UNUSED addr,
                       const void H5_ATTR_UNUSED *_rt_key, void *_udata)
{
    hsize_t *stab_size = (hsize_t *)_udata; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(stab_size);

    *stab_size += H5G_NODE_SIZE(f);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5G_node_debug(H5F_t *f, haddr_t addr, FILE *stream, int indent, int fwidth, haddr_t heap_addr)
{
    H5G_node_t *sn        = NULL;
    H5HL_t     *heap      = NULL;
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(H5_addr_defined(addr));
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    
    if (heap_addr > 0 && H5_addr_defined(heap_addr))
        if (NULL == (heap = H5HL_protect(f, heap_addr, H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table heap");

    
    H5E_PAUSE_ERRORS
        {
            sn = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, addr, f, H5AC__READ_ONLY_FLAG);
        }
    H5E_RESUME_ERRORS
    if (sn) {
        unsigned u; 

        Rfprintf(stream, "%*sSymbol Table Node...\n", indent, "");
        Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Dirty:", sn->cache_info.is_dirty ? "Yes" : "No");
        Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
                "Size of Node (in bytes):", (unsigned)sn->node_size);
        Rfprintf(stream, "%*s%-*s %u of %u\n", indent, "", fwidth, "Number of Symbols:", sn->nsyms,
                (unsigned)(2 * H5F_SYM_LEAF_K(f)));

        indent += 3;
        fwidth = MAX(0, fwidth - 3);
        for (u = 0; u < sn->nsyms; u++) {
            Rfprintf(stream, "%*sSymbol %u:\n", indent - 3, "", u);

            if (heap) {
                const char *s = (const char *)H5HL_offset_into(heap, sn->entry[u].name_off);

                if (s)
                    Rfprintf(stream, "%*s%-*s `%s'\n", indent, "", fwidth, "Name:", s);
            } 
            else
                Rfprintf(stream, "%*s%-*s\n", indent, "", fwidth,
                        "Warning: Invalid heap address given, name not displayed!");

            H5G__ent_debug(sn->entry + u, stream, indent, fwidth, heap);
        } 
    }     
    
    else {
        H5G_bt_common_t udata; 

        udata.heap       = heap;
        udata.block_size = H5HL_heap_get_size(heap);
        if (H5B_debug(f, addr, stream, indent, fwidth, H5B_SNODE, &udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "unable to debug B-tree node");
    } 

done:
    if (sn && H5AC_unprotect(f, H5AC_SNODE, addr, sn, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to release symbol table node");
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 
