/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5TSmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5TSpkg.h"    

#ifdef H5_HAVE_THREADS

#if defined(_WIN32)

herr_t
H5TS_semaphore_init(H5TS_semaphore_t *sem, unsigned initial_count)
{
    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    if (H5_UNLIKELY(NULL == (*sem = CreateSemaphore(NULL, (LONG)initial_count, LONG_MAX, NULL))))
        return FAIL;

    return SUCCEED;
} 

herr_t
H5TS_semaphore_destroy(H5TS_semaphore_t *sem)
{
    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    if (H5_UNLIKELY(0 == CloseHandle(*sem)))
        return FAIL;

    return SUCCEED;
} 

#elif defined(__unix__) && !defined(__MACH__)

herr_t
H5TS_semaphore_init(H5TS_semaphore_t *sem, unsigned initial_count)
{
    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    if (H5_UNLIKELY(0 != sem_init(sem, 0, initial_count)))
        return FAIL;

    return SUCCEED;
} 

herr_t
H5TS_semaphore_destroy(H5TS_semaphore_t *sem)
{
    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    if (H5_UNLIKELY(0 != sem_destroy(sem)))
        return FAIL;

    return SUCCEED;
} 
#else

herr_t
H5TS_semaphore_init(H5TS_semaphore_t *sem, unsigned initial_count)
{
    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    if (H5_UNLIKELY(H5TS_mutex_init(&sem->mutex, H5TS_MUTEX_TYPE_PLAIN) < 0))
        return FAIL;
    if (H5_UNLIKELY(H5TS_cond_init(&sem->cond) < 0)) {
        H5TS_mutex_destroy(&sem->mutex);
        return FAIL;
    }
    sem->waiters = 0;
    sem->counter = (int)initial_count;

    return SUCCEED;
} 

herr_t
H5TS_semaphore_destroy(H5TS_semaphore_t *sem)
{
    herr_t ret_value = SUCCEED;

    
    if (H5_UNLIKELY(NULL == sem))
        return FAIL;

    if (H5_UNLIKELY(H5TS_mutex_destroy(&sem->mutex) < 0))
        ret_value = FAIL;
    if (H5_UNLIKELY(H5TS_cond_destroy(&sem->cond) < 0))
        return FAIL;

    return ret_value;
} 
#endif

#endif 
