% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ctstm.R
\name{IndivCtstm}
\alias{IndivCtstm}
\title{Individual-level continuous time state transition model}
\format{
An \link[R6:R6Class]{R6::R6Class} object.
}
\description{
Simulate outcomes from an individual-level continuous time state transition
model (CTSTM). The class supports "clock-reset"
(i.e., semi-Markov), "clock-forward" (i.e., Markov), and mixtures of
clock-reset and clock-forward multi-state models as described in
\code{\link{IndivCtstmTrans}}.
}
\examples{
library("flexsurv")

# Treatment strategies, target population, and model structure
strategies <- data.frame(strategy_id = c(1, 2))
patients <- data.frame(patient_id = seq(1, 3),
                       age = c(45, 50, 60),
                       female = c(0, 0, 1))
states <- data.frame(state_id = c(1, 2))
hesim_dat <- hesim_data(strategies = strategies,
                        patients = patients,
                        states = states)

# Parameter estimation
## Multi-state model
tmat <- rbind(c(NA, 1, 2),
              c(3, NA, 4),
              c(NA, NA, NA))
fits <- vector(length = max(tmat, na.rm = TRUE), mode = "list")
surv_dat <- data.frame(mstate3_exdata$transitions)
for (i in 1:length(fits)){
  fits[[i]] <- flexsurvreg(Surv(years, status) ~ factor(strategy_id),
                           data = surv_dat,
                           subset = (trans == i),
                           dist = "weibull")
}
fits <- flexsurvreg_list(fits)

## Utility
utility_tbl <- stateval_tbl(data.frame(state_id = states$state_id,
                                       mean = mstate3_exdata$utility$mean,
                                       se = mstate3_exdata$utility$se),
                            dist = "beta")
## Costs
drugcost_tbl <- stateval_tbl(data.frame(strategy_id = strategies$strategy_id,
                                        est = mstate3_exdata$costs$drugs$costs),
                             dist = "fixed")
medcost_tbl <- stateval_tbl(data.frame(state_id = states$state_id,
                                       mean = mstate3_exdata$costs$medical$mean,
                                       se = mstate3_exdata$costs$medical$se),
                            dist = "gamma")

# Economic model
n_samples = 2

## Construct model
### Transitions
transmod_data <- expand(hesim_dat)
transmod <- create_IndivCtstmTrans(fits, input_data = transmod_data,
                                   trans_mat = tmat,
                                   n = n_samples)

### Utility
utilitymod <- create_StateVals(utility_tbl, n = n_samples, hesim_data = hesim_dat)

### Costs
drugcostmod <- create_StateVals(drugcost_tbl, n = n_samples, hesim_data = hesim_dat)
medcostmod <- create_StateVals(medcost_tbl, n = n_samples, hesim_data = hesim_dat)
costmods <- list(drugs = drugcostmod,
                 medical = medcostmod)

### Combine
ictstm <- IndivCtstm$new(trans_model = transmod,
                         utility_model = utilitymod,
                         cost_models = costmods)


## Simulate outcomes
head(ictstm$sim_disease()$disprog_)
head(ictstm$sim_stateprobs(t = c(0, 5, 10))$stateprobs_[t == 5])
ictstm$sim_qalys(dr = .03)
ictstm$sim_costs(dr = .03)

### Summarize cost-effectiveness
ce <- ictstm$summarize()
head(ce)
format(summary(ce), pivot_from = "strategy")
}
\references{
\href{https://arxiv.org/abs/2102.09437}{Incerti and Jansen (2021)}.
See Section 2.2 for a mathematical description of an individual-level CTSTM and Section 4.1 for
an example in oncology.
}
\seealso{
The \code{\link{IndivCtstmTrans}} documentation
describes the class for the transition model and the \code{\link{StateVals}} documentation
describes the class for the cost and utility models. An \code{\link{IndivCtstmTrans}}
object is typically created using \code{\link[=create_IndivCtstmTrans]{create_IndivCtstmTrans()}}.

There are currently two relevant vignettes. \code{vignette("mstate")} shows how to
parameterize \code{IndivCtstmTrans} objects in cases where patient-level data is
available by fitting a multi-state models. \code{vignette("markov-inhomogeneous-indiv")}
shows how the time inhomogeneous Markov cohort model in
\code{vignette("markov-inhomogeneous-cohort")} can be developed as an individual
patient simulation; in doing so, it shows how \code{IndivCtstm} models can be
used even without access to patient-level data.
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{trans_model}}{The model for health state transitions. Must be an object
of class \code{\link{IndivCtstmTrans}}.}

\item{\code{utility_model}}{The model for health state utility. Must be an object of
class \code{\link{StateVals}}.}

\item{\code{cost_models}}{The models used to predict costs by health state.
Must be a list of objects of class \code{\link{StateVals}}, where each element of the
list represents a different cost category.}

\item{\code{disprog_}}{An object of class \code{\link{disprog}}.}

\item{\code{stateprobs_}}{An object of class \code{\link{stateprobs}} simulated using \verb{$sim_stateprobs()}.}

\item{\code{qalys_}}{An object of class \code{\link{qalys}} simulated using \verb{$sim_qalys()}.}

\item{\code{costs_}}{An object of class \code{\link{costs}} simulated using \verb{$sim_costs()}.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-IndivCtstm-new}{\code{IndivCtstm$new()}}
\item \href{#method-IndivCtstm-sim_disease}{\code{IndivCtstm$sim_disease()}}
\item \href{#method-IndivCtstm-sim_stateprobs}{\code{IndivCtstm$sim_stateprobs()}}
\item \href{#method-IndivCtstm-sim_qalys}{\code{IndivCtstm$sim_qalys()}}
\item \href{#method-IndivCtstm-sim_costs}{\code{IndivCtstm$sim_costs()}}
\item \href{#method-IndivCtstm-summarize}{\code{IndivCtstm$summarize()}}
\item \href{#method-IndivCtstm-clone}{\code{IndivCtstm$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-IndivCtstm-new"></a>}}
\if{latex}{\out{\hypertarget{method-IndivCtstm-new}{}}}
\subsection{Method \code{new()}}{
Create a new \code{IndivCtstm} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{IndivCtstm$new(trans_model = NULL, utility_model = NULL, cost_models = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{trans_model}}{The \code{trans_model} field.}

\item{\code{utility_model}}{The \code{utility_model} field.}

\item{\code{cost_models}}{The \code{cost_models} field.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{IndivCtstm} object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-IndivCtstm-sim_disease"></a>}}
\if{latex}{\out{\hypertarget{method-IndivCtstm-sim_disease}{}}}
\subsection{Method \code{sim_disease()}}{
Simulate disease progression (i.e., individual trajectories through a multi-state
model) using \code{IndivCtstmTrans$sim_disease()}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{IndivCtstm$sim_disease(max_t = 100, max_age = 100, progress = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{max_t}}{A scalar or vector denoting the length of time to simulate the model.
If a vector, must be equal to the number of simulated patients.}

\item{\code{max_age}}{A scalar or vector denoting the maximum age to simulate each patient until.
If a vector, must be equal to the number of simulated patients.}

\item{\code{progress}}{An integer, specifying the PSA iteration (i.e., sample) that should be
printed every \code{progress} PSA iterations. For example, if \code{progress = 2},
then every second PSA iteration is printed. Default is \code{NULL},
in which case no output is printed.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of self with simulated output stored in \code{disprog_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-IndivCtstm-sim_stateprobs"></a>}}
\if{latex}{\out{\hypertarget{method-IndivCtstm-sim_stateprobs}{}}}
\subsection{Method \code{sim_stateprobs()}}{
Simulate health state probabilities as a function of time using the
simulation output stored in \code{disprog}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{IndivCtstm$sim_stateprobs(t)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{t}}{A numeric vector of times.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of class \code{\link{stateprobs}}
stored in \code{stateprobs_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-IndivCtstm-sim_qalys"></a>}}
\if{latex}{\out{\hypertarget{method-IndivCtstm-sim_qalys}{}}}
\subsection{Method \code{sim_qalys()}}{
Simulate quality-adjusted life-years (QALYs) as a function of \code{disprog_} and
\code{utility_model}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{IndivCtstm$sim_qalys(
  dr = 0.03,
  type = c("predict", "random"),
  lys = TRUE,
  by_patient = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dr}}{Discount rate.}

\item{\code{type}}{\code{"predict"} for mean values or \code{"random"} for random samples
as in \verb{$sim()} in \code{\link{StateVals}}.}

\item{\code{lys}}{If \code{TRUE}, then life-years are simulated in addition to QALYs.}

\item{\code{by_patient}}{If \code{TRUE}, then QALYs and/or costs are computed at the patient level.
If \code{FALSE}, then they are averaged across patients by health state.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of
class \link{qalys} stored in \code{qalys_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-IndivCtstm-sim_costs"></a>}}
\if{latex}{\out{\hypertarget{method-IndivCtstm-sim_costs}{}}}
\subsection{Method \code{sim_costs()}}{
Simulate costs as a function of \code{disprog_} and \code{cost_models}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{IndivCtstm$sim_costs(
  dr = 0.03,
  type = c("predict", "random"),
  by_patient = FALSE,
  max_t = Inf
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dr}}{Discount rate.}

\item{\code{type}}{\code{"predict"} for mean values or \code{"random"} for random samples
as in \verb{$sim()} in \code{\link{StateVals}}.}

\item{\code{by_patient}}{If \code{TRUE}, then QALYs and/or costs are computed at the patient level.
If \code{FALSE}, then they are averaged across patients by health state.}

\item{\code{max_t}}{Maximum time duration to compute costs once a patient has
entered a (new) health state. By default, equal to \code{Inf},
so that costs are computed over the entire duration that a patient is in
a given health state. If time varies by each cost category, then time
can also be passed as a numeric vector of length equal to the number of
cost categories (e.g., \code{c(1, 2, Inf, 3)} for a model with four cost categories).}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An instance of \code{self} with simulated output of class \link{costs}
stored in \code{costs_}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-IndivCtstm-summarize"></a>}}
\if{latex}{\out{\hypertarget{method-IndivCtstm-summarize}{}}}
\subsection{Method \code{summarize()}}{
Summarize costs and QALYs so that cost-effectiveness analysis can be performed.
See \code{\link[=summarize_ce]{summarize_ce()}}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{IndivCtstm$summarize(by_grp = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{by_grp}}{If \code{TRUE}, then costs and QALYs are computed by subgroup. If
\code{FALSE}, then costs and QALYs are aggregated across all patients (and subgroups).}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-IndivCtstm-clone"></a>}}
\if{latex}{\out{\hypertarget{method-IndivCtstm-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{IndivCtstm$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
