# MIT License
#
# Copyright (c) 2025 Reed A. Cartwright <racartwright@gmail.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

the <- new.env(parent = emptyenv())

#' Ironseed: Improved Random Number Generator Seeding
#'
#' @description
#'
#' An ironseed is a 256-bit hash digest constructed from a variable-length
#' input sequence and can be used to generate a variable-length output sequence
#' of seeds, including initializing R's built-in random number generator.
#'
#' - `ironseed()` creates an ironseed from user supplied objects or
#'  automatically from multiple sources of entropy on the local system. It also
#'  initializes R's built-in random number generator from an ironseed.
#'
#' - `create_seqseq()` uses an ironseed to generate a sequence of 32-bit seeds.
#'
#' - `is_ironseed()` tests whether an object is an ironseed, and
#'   `is_ironseed_str()` tests if it is a string representing and ironseed.
#'
#' - `as_ironseed()` casts an object to an ironseed, and `parse_ironseed_str()`
#'   parses a string to an ironseed.
#'
#' @param ... objects
#' @param set_seed a logical indicating whether to initialize `.Random.seed`.
#' @param quiet a logical indicating whether to silence messages.
#' @param fe an ironseed
#' @param n a scalar integer specifying the number of seeds to generate
#' @param x a string, ironseed, or other object
#'
#' @returns An ironseed. If `.Random.seed` was initialized, the ironseed used
#'   will be returned invisibly.
#'
#' @details
#'
#' Ironseeds have a specific string representation, e.g.
#' "rBQSjhjYv1d-z8dfMATEicf-sw1NSWAvVDi-bQaKSKKQmz1", where each element is a
#' 64-bit number encoded in little-endian base58 format.
#'
#' Parameter `set_seed` defaults to `TRUE` if `.Random.seed` does not already
#' exist and `FALSE` otherwise.
#'
#' Ironseed behaves differently depending on the number of arguments passed as
#' `...`.
#'
#' - 0 arguments: If initialization is enabled, `ironseed()` generates an
#'   automatic ironseed. Otherwise, `ironseed()` returns the last ironseed used
#'   to initialize `.Random.seed`.
#'
#' - 1 argument: `ironseed(NULL)` generates an automatic ironseed. For
#'   `ironseed(x)`, if `x` is an ironseed object, it is used as is. If `x`, is
#'   a scalar character that matches an ironseed string, it is parsed to an
#'   ironseed. Otherwise, `x` hashed to create an ironseed.
#'
#' - 2+ arguments: `ironseed(x,y,...)` hashes the arguments to create an
#'   ironseed.
#'
#' @details
#'
#' An ironseed is a finite-entropy (or fixed-entropy) hash digest that can be
#' used to generate an unlimited sequence of seeds for initializing the state of
#' a random number generator. It is inspired by the work of M.E. O’Neill and
#' others.
#'
#' An ironseed is a 256-bit hash digest constructed from a variable-length
#' sequence of 32-bit inputs. Each ironseed consists of eight 32-bit
#' sub-digests. The sub-digests are 32-bit multilinear hashes that accumulate
#' entropy from the input sequence. Each input is included in every sub-digest.
#' The coefficients for the multilinear hashes are generated by a Weyl sequence.
#'
#' Multilinear hashes are also used to generate an output seed sequence from an
#' ironseed. Each 32-bit output value is generated by uniquely hashing the
#' sub-digests. The coefficients for the output are generated by a second
#' Weyl sequence.
#'
#' @seealso [.Random.seed]
#'
#' @references
#' - O’Neill (2015) Developing a seed_seq Alternative.
#'   <https://www.pcg-random.org/posts/developing-a-seed_seq-alternative.html>
#' - O’Neill (2015) Simple Portable C++ Seed Entropy.
#'   <https://www.pcg-random.org/posts/simple-portable-cpp-seed-entropy.html>
#' - O’Neill (2015) Random-Number Utilities.
#'   <https://gist.github.com/imneme/540829265469e673d045>
#' - Lemire and Kaser (2018) Strongly universal string hashing is fast.
#'   <http://arxiv.org/pdf/1202.4961>
#' - Weyl Sequence <https://en.wikipedia.org/wiki/Weyl_sequence>
#'
#' @export
#' @examples
#'
#' \dontshow{
#' oldseed <- ironseed::get_random_seed()
#' }
#'
#' # Generate an ironseed with user supplied data
#' ironseed::ironseed("Experiment", 20251031, 1)
#'
#' # Generate an ironseed automatically and initialize `.Random.seed` with it
#' ironseed::ironseed(set_seed = TRUE)
#'
#' \dontshow{
#' ironseed::set_random_seed(oldseed)
#' }
#'
ironseed <- function(..., set_seed = !has_random_seed(), quiet = FALSE) {
  x <- list(...)
  n <- length(x)

  # construct ironseed object based on arguments
  if (n == 0L && !isTRUE(set_seed)) {
    fe <- the$ironseed
  } else if (n == 0L || (n == 1L && is.null(x[[1]]))) {
    fe <- auto_ironseed()
  } else if (n == 1L && is_ironseed2(x[[1]])) {
    fe <- as_ironseed(x[[1]])
  } else {
    fe <- create_ironseed(x)
  }

  if (!isTRUE(set_seed)) {
    return(fe)
  }
  fill_random_seed(fe, quiet = quiet)
  the$ironseed <- fe
  invisible(fe)
}

#' Initialize .Random.seed
#'
#' @param x an ironseed.
#' @param seed a previous `.Random.seed`
#' @param quiet a logical indicating whether to silence messages.
#'
#' @returns `fill_random_seed()` returns the previous value of `.Random.seed` or
#' `NULL`.
#'
#' @export
#' @keywords internal
#' @importFrom stats runif
fill_random_seed <- function(x, quiet = FALSE) {
  stopifnot(is_ironseed(x))
  if (isFALSE(quiet)) {
    msg <- sprintf("** Ironseed : Seed %s", format(x))
    message(msg)
  }
  # save oldseed
  oldseed <- get_random_seed()
  # use set.seed to flush seed space and get again
  set.seed(1)
  seed <- get_random_seed()
  is_mt <- seed[2] == 624L

  # generate a seed sequence of the correct length
  seed[-1] <- create_seedseq(x, length(seed) - 1)
  # if seed[2] = 625, then MT will think it is uninitialized
  # set seed[2] to 624 to signal that it is initialized
  if (is_mt) {
    seed[2] <- 624L
  }
  # update .Random.seed with our own state
  assign(".Random.seed", seed, globalenv())
  # draw one value to trigger seed fixup
  runif(1)
  # return old seed
  invisible(oldseed)
}

#' @export
#' @keywords internal
#' @rdname fill_random_seed
has_random_seed <- function() {
  exists(".Random.seed", globalenv(), mode = "integer", inherits = FALSE)
}

#' @export
#' @keywords internal
#' @rdname fill_random_seed
get_random_seed <- function() {
  get0(
    ".Random.seed",
    globalenv(),
    mode = "integer",
    inherits = FALSE,
    ifnotfound = NULL
  )
}

#' @export
#' @keywords internal
#' @rdname fill_random_seed
set_random_seed <- function(seed) {
  assign(".Random.seed", seed, globalenv())
}

rm_random_seed <- function() {
  oldseed <- get_random_seed()
  rm(".Random.seed", envir = globalenv(), inherits = FALSE)
  invisible(oldseed)
}

create_ironseed <- function(x) {
  if (is.list(x)) {
    x <- lapply(x, unlist, use.names = FALSE)
  } else {
    x <- list(x)
  }
  .Call(R_create_ironseed, x)
}

auto_ironseed <- function() {
  .Call(R_auto_ironseed)
}

#' @export
#' @rdname ironseed
create_seedseq <- function(fe, n) {
  fe <- as_ironseed(fe)
  n <- as.integer(n)
  stopifnot(length(fe) == 8L)
  .Call(R_create_seedseq, fe, n)
}

ironseed_re <- paste0(
  "^[1-9A-HJ-NP-Za-km-z]{11}",
  "-[1-9A-HJ-NP-Za-km-z]{11}",
  "-[1-9A-HJ-NP-Za-km-z]{11}",
  "-[1-9A-HJ-NP-Za-km-z]{11}$"
)

#' @export
#' @rdname ironseed
is_ironseed <- function(x) {
  inherits(x, "ironseed_ironseed")
}

#' @export
#' @rdname ironseed
is_ironseed_str <- function(x) {
  is.character(x) && length(x) == 1L && grepl(ironseed_re, x)
}

is_ironseed2 <- function(x) {
  is_ironseed(x) || is_ironseed_str(x)
}

#' @export
#' @rdname ironseed
as_ironseed <- function(x) {
  if (is_ironseed(x)) {
    x
  } else if (is_ironseed_str(x)) {
    x <- parse_ironseed_str(x)
    structure(x, class = "ironseed_ironseed")
  } else if (is.numeric(x) && length(x) == 8L) {
    x <- as.integer(x)
    structure(x, class = "ironseed_ironseed")
  } else {
    stop("unable to convert `x` to ironseed")
  }
}

str_ironseed <- function(x) {
  stopifnot(length(x) == 8)
  x <- as.integer(x)

  # pack into 4 doubles
  x <- packBits(intToBits(x), "double")
  x <- .Call(R_base58_encode64, x)

  # Concatenate
  paste0(x, collapse = "-")
}

#' @export
as.character.ironseed_ironseed <- function(x, ...) {
  str_ironseed(x)
}

#' @export
format.ironseed_ironseed <- function(x, ...) {
  str_ironseed(x)
}

#' @export
#' @importFrom utils str
str.ironseed_ironseed <- function(object, ...) {
  cat(" ironseed ")
  str(format(object), give.head = FALSE)
}

#' @export
print.ironseed_ironseed <- function(x, ...) {
  s <- format(x, ...)
  cat("Ironseed: ")
  cat(s, sep = "\n")
  invisible(x)
}

#' @export
#' @rdname ironseed
parse_ironseed_str <- function(x) {
  stopifnot(is_ironseed_str(x))
  x <- strsplit(x, "-")[[1]]
  x <- .Call(R_base58_decode64, x)
  x <- numToInts(x)
  x
}
