% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elfunctions.R
\name{weightedEL}
\alias{weightedEL}
\title{Self-concordant multi-variate empirical likelihood with counts}
\source{
This original code was written for \insertCite{owen2013self}{smoothemplik}
and [published online](https://artowen.su.domains/empirical/) by Art B. Owen
(March 2015, February 2017). The present version was rewritten in \code{Rcpp} and
slightly reworked to contain fewer inner functions and loops.
}
\usage{
weightedEL(
  z,
  mu = NULL,
  ct = NULL,
  lambda.init = NULL,
  SEL = FALSE,
  return.weights = FALSE,
  lower = NULL,
  upper = NULL,
  order = 4L,
  weight.tolerance = NULL,
  thresh = 1e-16,
  itermax = 100L,
  verbose = FALSE,
  alpha = 0.3,
  beta = 0.8,
  backeps = 0
)
}
\arguments{
\item{z}{A numeric vector or a matrix with one data vector per column.}

\item{mu}{Hypothesised mean, default (0 ... 0) in R^ncol(z)}

\item{ct}{A numeric vector of non-negative counts.}

\item{lambda.init}{Starting lambda, default (0 ... 0)}

\item{SEL}{If \code{FALSE}, the default weight tolerance is \code{MachEps^(1/3)}, otherwise
it is \code{MachEps^(1/2)} of the maximum count.}

\item{return.weights}{Logical: if \code{TRUE}, returns the empirical probabilities. Default is memory-saving (\code{FALSE}).}

\item{lower}{Lower cut-off for [logTaylor()], default \code{1/nrow(z)}}

\item{upper}{Upper cutoff for [logTaylor()], default \code{Inf}}

\item{order}{Positive integer such that the Taylor approximation of this order to log(x) is self-concordant; usually 4 or higher. Passed to [logTaylor()].}

\item{weight.tolerance}{Weight tolerance for counts to improve numerical stability}

\item{thresh}{Convergence threshold for log-likelihood (the default is aggressive)}

\item{itermax}{Upper bound on number of Newton steps (seems ample)}

\item{verbose}{Logical: print output diagnostics?}

\item{alpha}{Backtracking line search parameter: acceptance of a decrease in function value by ALPHA*f of the prediction
based on the linear extrapolation.}

\item{beta}{Backtracking line search reduction factor. 0.1 corresponds to a very crude search, 0.8 corresponds
to a less crude search.}

\item{backeps}{Backtrack threshold: the search can miss by this much. Consider setting it to 1e-10
if backtracking seems to be failing due to round-off.}
}
\value{
A list with the following values:
\describe{
    \item{logelr}{Log of empirical likelihood ratio (equal to 0 if the hypothesised mean is equal to the sample mean)}
    \item{lam}{Vector of Lagrange multipliers}
    \item{wts}{Observation weights/probabilities (vector of length n)}
    \item{converged}{\code{TRUE} if algorithm converged. \code{FALSE} usually means that mu is not in the convex hull of the data. Then, a very small likelihood is returned (instead of zero).}
    \item{iter}{Number of iterations taken.}
    \item{ndec}{Newton decrement (see Boyd & Vandenberghe).}
    \item{gradnorm}{Norm of the gradient of log empirical likelihood.}
}
}
\description{
Implements the empirical-likelihood-ratio test for the mean of the coordinates of \code{z}
(with the hypothesised value \code{mu}). The counts need not be integer;
in the context of local likelihoods, they can be kernel observation weights.
}
\details{
Negative weights are not allowed. They could be useful in some applications, but they can destroy
convexity or even boundedness. They also make the Newton step fail to be of least squares type.

This function relies on the improved computational strategy for the empirical likelihood.
The search of the lambda multipliers is carried out via a dampened Newton method with guaranteed
convergence owing to the fact that the log-likelihood is replaced by its Taylor approximation
of any desired order (default: 4, the minimum value that ensures self-concordance).

Tweak \code{alpha} and \code{beta} with extreme caution. See \insertCite{boyd2004convex}{smoothemplik},
pp. 464--466 for details. It is necessary that \code{0 < alpha < 1/2} and \code{0 < beta < 1}.
\code{alpha = 0.3} seems better than 0.01 on some 2-dimensional test data (sometimes fewer iterations).

The argument names, except for \code{lambda.init}, are matching the original names in Art B. Owen's implementation.
The highly optimised one-dimensional counterpart, \code{weightedEL0}, is designed to return a faster
and a more accurate solution in the one-dimensional case.
}
\examples{
earth <- c(
  5.5, 5.61, 4.88, 5.07, 5.26, 5.55, 5.36, 5.29, 5.58, 5.65, 5.57, 5.53, 5.62, 5.29,
  5.44, 5.34, 5.79, 5.1, 5.27, 5.39, 5.42, 5.47, 5.63, 5.34, 5.46, 5.3, 5.75, 5.68, 5.85
)
weightedEL(earth, mu = 5.517, verbose = TRUE) # 5.517 is the modern accepted value

# Linear regression through empirical likelihood
coef.lm <- coef(lm(mpg ~ hp + am, data = mtcars))
xmat <- cbind(1, as.matrix(mtcars[, c("hp", "am")]))
yvec <- mtcars$mpg
foc.lm <- function(par, x, y) {  # The sample average of this
  resid <- y - drop(x \%*\% par)   # must be 0
  resid * x
}
minusEL <- function(par) -weightedEL(foc.lm(par, xmat, yvec), itermax = 10)$logelr
coef.el <- optim(c(mean(yvec), 0, 0), minusEL)$par
abs(coef.el - coef.lm) / coef.lm  # Relative difference

# Likelihood ratio testing without any variance estimation
# Define the profile empirical likelihood for the coefficient on am
minusPEL <- function(par.free, par.am)
  -weightedEL(foc.lm(c(par.free, par.am), xmat, yvec), itermax = 20)$logelr
# Constrained maximisation assuming that the coef on par.am is 3.14
coef.el.constr <- optim(coef.el[1:2], minusPEL, par.am = 3.14)$par
print(-2 * weightedEL(foc.lm(c(coef.el.constr, 3.14), xmat, yvec))$logelr)
# Exceeds the critical value qchisq(0.95, df = 1)
}
\references{
\insertAllCited{}
}
\seealso{
[logTaylor()], [weightedEL0()]
}
