#' Generate an example object from a JSON schema
#'
#' This function generates an example JSON object from a JSON schema.
#' This is used when enforcing a JSON schema through text-based handling
#' (requiring an example to be added to the prompt text).
#'
#' @param schema A list (R object) representing a JSON schema
#'
#' @return A list (R object) which matches the JSON schema definition
#'
#' @export
#'
#' @example inst/examples/answer_as_json.R
#' @family json
r_json_schema_to_example <- function(schema) {
  if ("schema" %in% names(schema)) {
    schema <- schema$schema
  }

  if (is.null(schema$type)) {
    return(NULL)
  }
  switch(
    schema$type,
    object = {
      # Generate an object with properties
      result <- list()
      if (!is.null(schema$properties)) {
        for (name in names(schema$properties)) {
          result[[name]] <- r_json_schema_to_example(schema$properties[[name]])
        }
      }
      return(result)
    },
    array = {
      # Generate an array with one example item
      example_item <- r_json_schema_to_example(schema$items)
      return(list(example_item))
    },
    string = {
      if (!is.null(schema$enum)) {
        return(schema$enum) # Include all possible values
      }
      return(ifelse(!is.null(schema$example), schema$example, "..."))
    },
    number = {
      return(ifelse(!is.null(schema$example), schema$example, 123.45))
    },
    integer = {
      return(ifelse(!is.null(schema$example), schema$example, 123))
    },
    boolean = {
      return(ifelse(!is.null(schema$example), schema$example, TRUE))
    },
    NULL
  ) # For unsupported or missing types, return NULL
}
