% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data_preprocessing.R
\name{prune_distance_network}
\alias{prune_distance_network}
\title{Prune Distance Data for Network Quality}
\usage{
prune_distance_network(
  data,
  virus_col,
  antibody_col,
  min_connections,
  iterations = 100
)
}
\arguments{
\item{data}{Data frame in long format containing:
- Column for viruses/antigens
- Column for antibodies/antisera
- Distance measurements (can contain NAs)
- Optional metadata columns}

\item{virus_col}{Character name of virus/antigen column}

\item{antibody_col}{Character name of antibody/antiserum column}

\item{min_connections}{Integer minimum required connections per point}

\item{iterations}{Integer maximum pruning iterations (default 100)}
}
\value{
A list containing two elements:
\item{pruned_data}{A \code{data.frame} containing only the measurements for the well-connected subset of points.}
\item{stats}{A \code{list} of pruning statistics including:
\itemize{
\item \code{original_points}: Number of unique antigens and sera before pruning.
\item \code{remaining_points}: Number of unique antigens and sera after pruning.
\item \code{iterations}: Number of pruning iterations performed.
\item \code{min_connections}: The minimum connection threshold used.
\item \code{is_connected}: A logical indicating if the final network is fully connected.
}
}
}
\description{
Iteratively removes viruses and antibodies with insufficient connections to create a
well-connected network subset. The pruning continues until all remaining points have
at least the specified minimum number of connections.
}
\examples{
# Create a sparse dataset with 12 viruses and 12 antibodies
viruses <- paste0("V", 1:12)
antibodies <- paste0("A", 1:12)
all_pairs <- expand.grid(Virus = viruses, Antibody = antibodies, stringsAsFactors = FALSE)

# Sample 70 pairs to create a sparse matrix
set.seed(42)
assay_data <- all_pairs[sample(nrow(all_pairs), 70), ]

# Ensure some viruses/antibodies are poorly connected for the example
assay_data <- assay_data[!(assay_data$Virus \%in\% c("V11", "V12")),]
assay_data <- assay_data[!(assay_data$Antibody \%in\% c("A11", "A12")),]

# Add back single connections for the poorly connected nodes
poor_connections <- data.frame(
  Virus = c("V11", "V1", "V12", "V2"),
  Antibody = c("A1", "A11", "A2", "A12")
)
assay_data <- rbind(assay_data, poor_connections)

# View connection counts before pruning
# Virus V11 and V12, and Antibody A11 and A12 have only 1 connection
table(assay_data$Virus)
table(assay_data$Antibody)

# Prune the network to keep only nodes with at least 2 connections
pruned_result <- prune_distance_network(
  data = assay_data,
  virus_col = "Virus",
  antibody_col = "Antibody",
  min_connections = 2
)
                                
# View connection counts after pruning
# The poorly connected nodes have been removed
table(pruned_result$pruned_data$Virus)
table(pruned_result$pruned_data$Antibody)

# Check the summary statistics
print(pruned_result$stats)

}
