# AlgeriAPIs - Access Algerian Data via Public APIs
# Version 0.1.0
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_algeria_gdp

library(testthat)

test_that("get_algeria_gdp() returns a tibble with the correct structure and content", {
  skip_on_cran()  # Skip test on CRAN if the API is unavailable

  result <- get_algeria_gdp()

  # Check that the result is not NULL
  expect_false(is.null(result))

  # Check that the result is a data.frame/tibble
  expect_s3_class(result, "data.frame")

  # Check that the column names are exactly as expected
  expect_named(result, c("indicator", "country", "year", "value", "value_label"))

  # Check data types of each column
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "double")
  expect_type(result$value_label, "character")

  # Check that the indicator column contains only the expected value
  expect_true(all(result$indicator == "GDP (current US$)"))

  # Check that the country column contains only "Algeria"
  expect_true(all(result$country == "Algeria"))

  # Check that the year range is correct
  expect_true(all(result$year >= 2010 & result$year <= 2022))

  # Check that the number of rows is 13 (2010–2022)
  expect_equal(nrow(result), 13)

  # Check that the number of columns is 5
  expect_equal(ncol(result), 5)

  # Check that there are no NA values in the value column
  expect_false(any(is.na(result$value)))

  # Check that all GDP values are positive numbers
  expect_true(all(result$value > 0))

  # Check that value_label is formatted with commas
  expect_true(all(grepl(",", result$value_label)))
})
