% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/count.R
\name{count}
\alias{count}
\alias{count_R}
\alias{count_IR}
\alias{count_I}
\alias{count_SI}
\alias{count_S}
\alias{count_all}
\alias{n_rsi}
\alias{count_df}
\title{Count isolates}
\source{
Wickham H. \strong{Tidy Data.} The Journal of Statistical Software, vol. 59, 2014. \url{http://vita.had.co.nz/papers/tidy-data.html}
}
\usage{
count_R(..., also_single_tested = FALSE)

count_IR(..., also_single_tested = FALSE)

count_I(..., also_single_tested = FALSE)

count_SI(..., also_single_tested = FALSE)

count_S(..., also_single_tested = FALSE)

count_all(...)

n_rsi(...)

count_df(data, translate_ab = "name", language = get_locale(),
  combine_SI = TRUE, combine_IR = FALSE)
}
\arguments{
\item{...}{one or more vectors (or columns) with antibiotic interpretations. They will be transformed internally with \code{\link{as.rsi}} if needed.}

\item{also_single_tested}{a logical to indicate whether (in combination therapies) also observations should be included where not all antibiotics were tested, but at least one of the tested antibiotics contains a target interpretation (e.g. S in case of \code{portion_S} and R in case of \code{portion_R}). \strong{This would lead to selection bias in almost all cases.}}

\item{data}{a \code{data.frame} containing columns with class \code{rsi} (see \code{\link{as.rsi}})}

\item{translate_ab}{a column name of the \code{\link{antibiotics}} data set to translate the antibiotic abbreviations to, using \code{\link{ab_property}}}

\item{language}{language of the returned text, defaults to system language (see \code{\link{get_locale}}) and can also be set with \code{\link{getOption}("AMR_locale")}. Use \code{language = NULL} or \code{language = ""} to prevent translation.}

\item{combine_SI}{a logical to indicate whether all values of S and I must be merged into one, so the output only consists of S+I vs. R (susceptible vs. resistant). This used to be the parameter \code{combine_IR}, but this now follows the redefinition by EUCAST about the interpretion of I (increased exposure) in 2019, see section 'Interpretation of S, I and R' below. Default is \code{TRUE}.}

\item{combine_IR}{a logical to indicate whether all values of I and R must be merged into one, so the output only consists of S vs. I+R (susceptible vs. non-susceptible). This is outdated, see parameter \code{combine_SI}.}
}
\value{
Integer
}
\description{
These functions can be used to count resistant/susceptible microbial isolates. All functions support quasiquotation with pipes, can be used in \code{dplyr}s \code{\link[dplyr]{summarise}} and support grouped variables, see \emph{Examples}.

\code{count_R} and \code{count_IR} can be used to count resistant isolates, \code{count_S} and \code{count_SI} can be used to count susceptible isolates.\cr
}
\details{
These functions are meant to count isolates. Use the \code{\link{portion}_*} functions to calculate microbial resistance.

The function \code{n_rsi} is an alias of \code{count_all}. They can be used to count all available isolates, i.e. where all input antibiotics have an available result (S, I or R). Their use is equal to \code{\link{n_distinct}}. Their function is equal to \code{count_S(...) + count_IR(...)}.

The function \code{count_df} takes any variable from \code{data} that has an \code{"rsi"} class (created with \code{\link{as.rsi}}) and counts the amounts of S, I and R. The resulting \emph{tidy data} (see Source) \code{data.frame} will have three rows (S/I/R) and a column for each variable with class \code{"rsi"}.

The function \code{rsi_df} works exactly like \code{count_df}, but adds the percentage of S, I and R.
}
\section{Interpretation of S, I and R}{

In 2019, EUCAST has decided to change the definitions of susceptibility testing categories S, I and R as shown below. Results of several consultations on the new definitions are available on the EUCAST website under "Consultations".

\itemize{
  \item{\strong{S} - }{Susceptible, standard dosing regimen: A microorganism is categorised as "Susceptible, standard dosing regimen", when there is a high likelihood of therapeutic success using a standard dosing regimen of the agent.}
  \item{\strong{I} - }{Susceptible, increased exposure: A microorganism is categorised as "Susceptible, Increased exposure" when there is a high likelihood of therapeutic success because exposure to the agent is increased by adjusting the dosing regimen or by its concentration at the site of infection.}
  \item{\strong{R} - }{Resistant: A microorganism is categorised as "Resistant" when there is a high likelihood of therapeutic failure even when there is increased exposure.}
}

Exposure is a function of how the mode of administration, dose, dosing interval, infusion time, as well as distribution and excretion of the antimicrobial agent will influence the infecting organism at the site of infection.

Source: \url{http://www.eucast.org/newsiandr/}.

\strong{This AMR package honours this new insight.}
}

\section{Read more on our website!}{

On our website \url{https://msberends.gitlab.io/AMR} you can find \href{https://msberends.gitlab.io/AMR/articles/AMR.html}{a tutorial} about how to conduct AMR analysis, the \href{https://msberends.gitlab.io/AMR/reference}{complete documentation of all functions} (which reads a lot easier than here in R) and \href{https://msberends.gitlab.io/AMR/articles/WHONET.html}{an example analysis using WHONET data}.
}

\examples{
# septic_patients is a data set available in the AMR package. It is true, genuine data.
?septic_patients

# Count resistant isolates
count_R(septic_patients$AMX)
count_IR(septic_patients$AMX)

# Or susceptible isolates
count_S(septic_patients$AMX)
count_SI(septic_patients$AMX)

# Count all available isolates
count_all(septic_patients$AMX)
n_rsi(septic_patients$AMX)

# Since n_rsi counts available isolates, you can
# calculate back to count e.g. non-susceptible isolates.
# This results in the same:
count_IR(septic_patients$AMX)
portion_IR(septic_patients$AMX) * n_rsi(septic_patients$AMX)

library(dplyr)
septic_patients \%>\%
  group_by(hospital_id) \%>\%
  summarise(R  = count_R(CIP),
            I  = count_I(CIP),
            S  = count_S(CIP),
            n1 = count_all(CIP),  # the actual total; sum of all three
            n2 = n_rsi(CIP),      # same - analogous to n_distinct
            total = n())          # NOT the number of tested isolates!

# Count co-resistance between amoxicillin/clav acid and gentamicin,
# so we can see that combination therapy does a lot more than mono therapy.
# Please mind that `portion_S` calculates percentages right away instead.
count_S(septic_patients$AMC)   # S = 1342 (71.4\%)
count_all(septic_patients$AMC) # n = 1879

count_S(septic_patients$GEN)   # S = 1372 (74.0\%)
count_all(septic_patients$GEN) # n = 1855

with(septic_patients,
     count_S(AMC, GEN))         # S = 1660 (92.3\%)
with(septic_patients,           # n = 1798
     n_rsi(AMC, GEN))

# Get portions S/I/R immediately of all rsi columns
septic_patients \%>\%
  select(AMX, CIP) \%>\%
  count_df(translate = FALSE)

# It also supports grouping variables
septic_patients \%>\%
  select(hospital_id, AMX, CIP) \%>\%
  group_by(hospital_id) \%>\%
  count_df(translate = FALSE)

}
\seealso{
\code{\link{portion}_*} to calculate microbial resistance and susceptibility.
}
\keyword{antibiotics}
\keyword{isolate}
\keyword{isolates}
\keyword{resistance}
\keyword{rsi}
\keyword{susceptibility}
