% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/G_predict.R
\name{G.predict}
\alias{G.predict}
\title{Generates the conditional predictions of random effects (BLUPs)}
\usage{
G.predict(G = NULL, gy = NULL, vcov.gy = NULL)
}
\arguments{
\item{G}{Input of the genomic relationship matrix \eqn{\boldsymbol{G}} in full form
(of dimension \eqn{(nx + ny) \times (nx + ny)}) (default = \code{NULL}).}

\item{gy}{Input of random effects (\emph{e.g.} breeding values) for individuals with known values.
Individual names should be assigned to \code{rownames} of this vector
and be found on the matrix \eqn{\boldsymbol{G}} (default = \code{NULL}).}

\item{vcov.gy}{The variance-covariance matrix associated with the random effects from the
individuals with known values (set \code{y}, of dimension \eqn{ny \times ny})
(default = \code{NULL}).}
}
\value{
A data frame with the predicted random effect values for individuals with
unobserved responses in the set \code{x}. If the variance-covariance matrix is provided,
standard errors are included in an additional column.
}
\description{
Predicts random effects values for individuals with unobserved responses (here called \code{x},
a vector of length \eqn{nx}) based on known random effect values for individuals with
observed responses (here called \code{y}, a vector of length \eqn{ny}). This is done using the
common genomic relationship matrix  \eqn{\boldsymbol{G}} for all
individuals (full matrix of dimension \eqn{(nx + ny) \times (nx + ny)}).

The prediction of unobserved responses will be performed through the
multivariante Normal conditional distribution. These predictions are identical to
what would be obtained if the entire set of individuals (\eqn{nx + ny}) were included into a
GBLUP animal model fit with individuals in the set \code{x} coded as missing.

The user needs to provide the matrix \eqn{\boldsymbol{G}} in full form.
Individual names (\eqn{nx + ny}) should be assigned to \code{rownames} and \code{colnames}, and these
can be in any order. If the variance-covariance matrix of the set \code{y} is provided,
standard errors of random effects in set \code{x} are calculated.
}
\examples{
\dontrun{
library(asreml) # Load asreml.

# Example: Apple data creating 100 missing observations.

# Prepare G (nx + ny).
G <- G.matrix(M = geno.apple, method = "VanRaden", sparseform = FALSE)$G
dim(G)

# Prepare subset of data.
# Select only 147 from 247 individuals from pheno.apple and geno.apple.
Gy <- G[1:147, 1:147]
phenoy <- pheno.apple[1:147, ]

# Obtain the BLUPs for the 147 individuals using ASReml-R.

# Blend Gy.
Gyb <- G.tuneup(G = Gy, blend = TRUE, pblend = 0.02)$Gb

# Get the Gy inverse
Gyinv  <- G.inverse(G = Gyb, sparseform = TRUE)$Ginv.sparse

# Fit a GBLUP model
phenoy$INDIV <- as.factor(phenoy$INDIV)
modelGBLUP <-
 asreml(
  fixed = JUI_MOT ~ 1,
  random = ~vm(INDIV, Gyinv),
  workspace = 128e06,
  data = phenoy)

# Obtain Predictions - BLUP (set y).
BLUP <- summary(modelGBLUP,coef = TRUE)$coef.random
head(BLUP)
gy <- as.matrix(BLUP[, 1])
rownames(gy) <- phenoy$INDIV

# Ready to make conditional predictions.
g.cond <- G.predict(G = G, gy = gy)
head(g.cond)
}

}
