% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/circ_cor.R
\name{circ_varcor_model}
\alias{circ_varcor_model}
\title{Analytic circular variances and correlations for bivariate angular models}
\usage{
circ_varcor_model(
  model = "vmsin",
  kappa1 = 1,
  kappa2 = 1,
  kappa3 = 0,
  mu1 = 0,
  mu2 = 0,
  nsim = 10000,
  ...
)
}
\arguments{
\item{model}{bivariate angular model. Must be one of \code{"vmsin"},
\code{"vmcos"}, or \code{"wnorm2"}.}

\item{kappa1, kappa2, kappa3}{concentration and covariance parameters.
Recycled to the same size. kappa3^2 must be < kappa1*kappa2 in the wnorm2 model
(see \link{rwnorm2} for a detailed parameterization of \code{wnorm2}).}

\item{mu1, mu2}{mean parameters. Ignored as they do not play any role in
the analytical formulas.}

\item{nsim}{Monte Carlo sample size. Ignored if all of \code{kappa1}, \code{kappa2}
and \code{abs(kappa3)} are < 150 or if model = \code{"wnorm2"}.}

\item{...}{additional model specific argment}
}
\value{
Returns a list with elements \code{var1}, \code{var2} (circular variances for the
first and second coordinates), \code{rho_fl} and \code{rho_js} (circular correlations).
See details.
}
\description{
Analytic circular variances and correlations for bivariate angular models
}
\details{
The function computes the analytic circular variances and correlations
(both Jammalamadaka-Sarma (JS) and Fisher-Lee (FL) forms) for von Mises sine,
von Mises cosine and bivariate wrapped normal distributions.

For \code{wnorm2}, expressions for the circular variances,
JS and FL correlation coefficients can be found in Mardia and Jupp (2009),
Jammalamadaka and Sarma (1988) and Fisher and Lee (1983) respectively.
For \code{vmsin} and \code{vmcos} these expressions are provided in Chakraborty and Wong (2018).

Because the analytic expressions in \code{vmsin} and \code{vmcos} models involve infinite sums
of product of Bessel functions,
if any of \code{kappa1}, \code{kappa2} and \code{abs(kappa3)} is larger
than or equal to 150, IID Monte Carlo with sample size \code{nsim} is used
to approximate \code{rho_js} for numerical stability.  From \code{rho_js},
\code{rho_fl} is computed using Corollary 2.2 in
Chakraborty and Wong (2018), which makes cost-complexity for
the \code{rho_fl} evaluation to be of order  O(\code{nsim}) for \code{vmsin}
and \code{vmcos} models. (In general,  \code{rho_fl} evaluation
is of order O(\code{nsim}^2)).

In addition, for the \code{vmcos} model, when \code{-150 < kappa3 < -1}
or \code{50 < max(kappa1, kappa2, abs(kappa3)) <= 150}, the analytic formulas
in Chakraborty and Wong (2018) are used; however, the reciprocal of the normalizing
constant and its partial derivatives are all calculated numerically via (quasi) Monte carlo method for
numerical stability. These (quasi) random numbers can be provided through the
argument \code{qrnd}, which must be a two column matrix, with each element being
a  (quasi) random number between 0 and 1. Alternatively, if \code{n_qrnd} is
provided (and \code{qrnd} is missing), a two dimensional sobol sequence of size \code{n_qrnd} is
generated via the function \link{sobol} from the R package \code{qrng}. If none of \code{qrnd}
or \code{n_qrnd} is available, a two dimensional sobol sequence of size 1e4 is used.
}
\examples{
circ_varcor_model("vmsin", kappa1= 1, kappa2 = 2, kappa3 = 3)

# Monte Carlo approximation
set.seed(1)
dat <- rvmsin(1000, 1, 2, 3)
# sample circular variance
circ_var <- function(x)
  1 - mean(cos(x - atan2(mean(sin(x)), mean(cos(x))) ))
circ_var(dat[, 1])
circ_var(dat[, 2])
circ_cor(dat, "fl")
circ_cor(dat, "js")

}
\references{
Fisher, N. I. and Lee, A. (1983). A correlation coefficient for circular data. Biometrika, 70(2):327-332.

Jammalamadaka, S. R. and Sarma, Y. (1988). A correlation coefficient for
angular variables. Statistical theory and data analysis II, pages 349-364.

Mardia, K. and Jupp, P. (2009). Directional Statistics. Wiley Series in Probability and Statistics. Wiley.

Chakraborty, S. and Wong, S, W.K. (2018). On the circular correlation coefficients
for bivariate von Mises distributions on a torus. arXiv e-print.
}
