% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{logreg_bvs}
\alias{logreg_bvs}
\title{Non-parallel version of Bayesian variable selector for logistic regression
data using nonlocal priors}
\usage{
logreg_bvs(exmat, chain1, nf, tau, r, a, b, in_cons, loopcnt, cplng, chain2)
}
\arguments{
\item{exmat}{An extended matrix where the first column is binary resonse
vector and the rest is the design matrix which has its first column all 1
to account for the intercept in the model and is the output of
\code{PreProcess} code where the fixed columns are moved to the beginning.}

\item{chain1}{The first chain or initial model where the MCMC algorithm
starts from. Note that the first \code{nf+1} elements are \code{1} where
\code{nf} is the number of fixed covariates that do not enter the selection
procedure and \code{1} is for the intercept.}

\item{nf}{The number of fixed covariates that do not enter the selection
procedure.}

\item{tau}{The paramter \code{tau} of the iMOM prior.}

\item{r}{The paramter \code{r} of the iMOM prior.}

\item{a}{The first parameter in beta distribution used as prior on model
size. This parameter is equal to 1 when uinform-binomial prior is used.}

\item{b}{The second paramter in beta distribution used as prior on model
size. This parameter is equal to 1 when uinform-binomial prior is used.}

\item{in_cons}{The average model size. This value under certain conditions
and when \code{p} is large, is equal to parameter \code{a} of the
beta-binomial prior.}

\item{loopcnt}{Number of iterations for MCMC procedure.}

\item{cplng}{A boolean variable indicating the coupling algorithm to be
performed or not.}

\item{chain2}{Second chain or model for starting the MCMC procedure. This
parameter is only used when \code{cplng=TRUE}. Thus, it could be simply
set to \code{chain1} when it is not used.}
}
\value{
It returns a list containing following objects:
\item{max_chain}{A \code{1} by \code{p+1} binary vector showing the selected
model with maximum probability. \code{1} means a specific variable is
selected. The first variable is always the intercept.}
\item{beta_hat}{The coefficient vector for the selected model. The first one
 is always for the intercept.}
\item{max_prop}{The unnormalized probability of the model with highest
posterior probability.}
 \item{num_iterations}{The number of MCMC iterations that are executed.
 This is used when \code{cplng=TRUE} to check whether the total designated
 MCMC iterations were used or two chains are coupled sooner than that.}
\item{cplng_flag}{This is used when \code{cplng=TRUE} and indicates whether
two chains are coupled or not.}
\item{num_vis_models}{Number of visited models in search for the highest
probability model. This contains redundant models too and is not the number
of unique models.}
\item{hash_key}{This is only used when \code{cplng = FALSE}. This is a
vector containing real numbers uniquely assigned to each model for
distinguishing them.}
\item{hash_prob}{This is only used when \code{cplng = FALSE}. This is a
vector of probabilities for each visited model.}
\item{vis_covs}{This is only used when \code{cplng = FALSE}. This is a
list where each element contains indices of covariates for each visited
model.}
}
\description{
This function performs Bayesian variable selection for
logistic regression data in a non-parallel fashion. It does not contain any
pre-processing step or variable initialization. Moreover it does not have
the feature to be run in parallel for performing the coupling algorithm.
Therefore in general, it is NOT recommended to be used unless the user
knows how to initialize all the parameters. However, this function is
called by \code{\link{bvs}} function, the recommended way to run Bayesian
variable selection for such datasets.
}
\examples{
### Initializing parameters
n <- 200
p <- 40
set.seed(123)
Sigma <- diag(p)
full <- matrix(c(rep(0.5, p*p)), ncol=p)
Sigma <- full + 0.5*Sigma
cholS <- chol(Sigma)
Beta <- c(-1.7,1.8,2.5)
X <- matrix(rnorm(n*p), ncol=p)
X <- X\%*\%cholS
colnames(X) <- paste("gene_",c(1:p),sep="")
beta <- numeric(p)
beta[c(1:length(Beta))] <- Beta
XB <- X\%*\%beta
probs <- as.vector(exp(XB)/(1+exp(XB)))
y <- rbinom(n,1,probs)
exmat <- cbind(y,X)
tau <- 0.5; r <- 1; a <- 3; b <- p-a; in_cons <- a;
loopcnt <- 100; cplng <- FALSE;
initProb <- in_cons/p

### Initializing Chains
schain <- p
while (schain > in_cons || schain == 0) {
chain1 <- rbinom(p, 1, initProb)
 schain <- sum(chain1)
}
chain1 <- as.numeric(c(1, chain1))
chain2 <- chain1
nf <- 0 ### No fixed columns

### Running the function
bvsout <- logreg_bvs(exmat,chain1,nf,tau,r,a,b,in_cons,loopcnt,cplng,chain2)

### Number of visited models for this specific run:
bvsout$num_vis_models

### The selected model:
which(bvsout$max_chain > 0)

### Estimated coefficients:
bvsout$beta_hat

### The unnormalized probability of the selected model:
bvsout$max_prob
}
\references{
Nikooienejad, A., Wang, W., and Johnson, V. E. (2016). Bayesian
variable selection for binary outcomes in high dimensional genomic studies
using nonlocal priors. Bioinformatics, 32(9), 1338-1345.\cr\cr
Nikooienejad, A., Wang, W., and Johnson, V. E. (2017). Bayesian Variable
Selection in High Dimensional Survival Time Cancer Genomic Datasets using
Nonlocal Priors. arXiv preprint, arXiv:1712.02964.\cr\cr
Johnson, V. E., and Rossell, D. (2010). On the use of non-local prior
densities in Bayesian hypothesis tests. Journal of the Royal Statistical
Society: Series B (Statistical Methodology), 72(2), 143-170.\cr\cr
Johnson, V. E. (1998). A coupling-regeneration scheme for
diagnosing convergence in Markov chain Monte Carlo algorithms. Journal of
the American Statistical Association, 93(441), 238-248.
}
\seealso{
\code{\link{bvs}}
}
\author{
Amir Nikooienejad
}
