/*
  Copyright (C) 2005 Steven L. Scott

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
*/

/*
 *  R : A Computer Language for Statistical Data Analysis
 *  Copyright (C) 1995, 1996  Robert Gentleman and Ross Ihaka
 *  Copyright (C) 2000 The R Development Core Team
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA.
 */

#include "nmath.hpp"
#include "dpq.hpp"
namespace Rmath{

double pt(double x, double n, int lower_tail, int log_p)
{
/* return  P[ T <= x ]  where
 * T ~ t_{n}  (t distrib. with n degrees of freedom).

 *      --> ./pnt.c for NON-central
 */
    double val;
#ifdef IEEE_754
    if (ISNAN(x) || ISNAN(n))
        return x + n;
#endif
    if (n <= 0.0) ML_ERR_return_NAN;

    if(!R_FINITE(x))
        return (x < 0) ? R_DT_0 : R_DT_1;
    if(!R_FINITE(n))
        return pnorm(x, 0.0, 1.0, lower_tail, log_p);
    if (n > 4e5) { /*-- Fixme(?): test should depend on `n' AND `x' ! */
        /* Approx. from  Abramowitz & Stegun 26.7.8 (p.949) */
        val = 1./(4.*n);
        return pnorm(x*(1. - val)/sqrt(1. + x*x*2.*val), 0.0, 1.0,
                     lower_tail, log_p);
    }

    val = pbeta(n / (n + x * x), n / 2.0, 0.5, /*lower_tail*/1, log_p);

    /* Use "1 - v"  if  lower_tail  and  x > 0 (but not both):*/
    if(x <= 0.)
        lower_tail = !lower_tail;

    if(log_p) {
        if(lower_tail) return log1p(-0.5*exp(val));
        else return val - M_LN2; /* = log(.5* pbeta(....)) */
    }
    else {
        val /= 2.;
        return R_D_Cval(val);
    }
}
}

