# Copyright (c) 2022, Adrian Dusa
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, in whole or in part, are permitted provided that the
# following conditions are met:
#     * Redistributions of source code must retain the above copyright
#       notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#     * The names of its contributors may NOT be used to endorse or promote products
#       derived from this software without specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL ADRIAN DUSA BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

`getMetadata` <- 
function(x, save = FALSE, declared = TRUE, OS = "Windows", encoding = "UTF-8", ...) {
    
    # TODO: detect DDI version or ask the version through a dedicated argument
    # http://www.ddialliance.org/Specification/DDI-Codebook/2.5/XMLSchema/field_level_documentation.html
    
    dots <- list(...)

    user_na <- TRUE # force reading the value labels
    if (
        is.element("user_na", names(dots)) && is.atomic(dots$user_na) && 
        length(dots$user_na) == 1 && is.logical(dots$user_na)
    ) {
        user_na <- dots$user_na
    }
    
    embed <- isTRUE(dots$embed)
    
    if (is.data.frame(x)) {
        error <- TRUE
        i <- 1
        while (i <= ncol(x) & error) {
            attrx <- attributes(x[[i]])
            if (any(is.element(c("label", "labels", "na_value", "na_range"), names(attrx)))) {
                error <- FALSE
            }
            i <- i + 1
        }

        if (error) {
            if (is.element("error_null", names(dots))) {
                return(NULL)
            }

            admisc::stopError("The input does not seem to contain any metadata.")
        }
        else {
            codeBook <- list()
            codeBook$dataDscr <- collectMetadata(x)
            if (embed) {
                codeBook$fileDscr <- list(
                    datafile = x
                )
            }
            return(codeBook)
        }
    }
    
    enter <- getEnter(OS)
    
    fromsetupfile <- isTRUE(dots$fromsetupfile)
    
    tp <- treatPath(x, type = "*")
    
    singlefile <- length(tp$files) == 1
    notes <- NA
    
    if (!fromsetupfile & !singlefile) {
        cat("Processing:\n")
    }

    data <- NULL

    result <- vector(mode = "list", length = length(tp$files))
    
    for (ff in seq(length(result))) {
        if (!fromsetupfile & !singlefile) {
            cat(tp$files[ff], "\n")
        }
        
        if (tp$fileext[ff] == "XML") {

            codeBook <- list()
            xml <- getXML(file.path(tp$completePath, tp$files[ff]))
            
            # lapply(xml_find_all(xml, "/d1:codeBook/d1:dataDscr/d1:var"), function(x) {
            #     list(label = admisc::trimstr(xml_text(xml_find_first(x, "d1:labl"))))
            # })
            
            xmlns <- xml2::xml_ns(xml)
            # d1  <-> ddi:codebook:2_5"
            # xsi <-> http://www.w3.org/2001/XMLSchema-instance
            # xsd <-> http://www.w3.org/2001/XMLSchema
            wns <- which(xmlns == "ddi:codebook:2_5")
            if (length(wns) == 0) {
                admisc::stopError("The XML document does not contain a DDI namespace.")
            }

            # <d>efault <n>ame <s>pace
            dns <- names(xmlns)[wns[1]]
            if (dns != "d1") {
                codeBook$xmlns <- dns
            }
            dns <- paste0(dns, ":")
            
            ### Unfortunately this does not work because some variables don't always have labels
            ### and we'll end up having a vector of labels that is shorter than the number of variables
            # xpath <- sprintf("/%scodeBook/%sdataDscr/%svar/%slabl", dns, dns, dns, dns)
            # varlab <- cleanup(xml2::xml_text(xml2::xml_find_all(xml, xpath)))
            ###

            xpath <- sprintf("/%scodeBook/%sdataDscr/%svar", dns, dns, dns)
            vars <- xml2::xml_find_all(xml, xpath)
            varlab <- cleanup(xml2::xml_text(xml2::xml_find_first(vars, sprintf("%slabl", dns))))

            xpath <- sprintf("/%scodeBook/%sfileDscr/%snotes", dns, dns, dns)
            notes <- xml2::xml_text(xml2::xml_find_first(xml, xpath))

            codeBook$dataDscr <- lapply(varlab, function(x) list(label = x))
            
            xpath <- sprintf("/%scodeBook/%sdataDscr/%svar/@name", dns, dns, dns)
            names(codeBook$dataDscr) <- admisc::trimstr(xml2::xml_text(xml2::xml_find_all(xml, xpath)))
            
            for (i in seq(length(codeBook$dataDscr))) {
                # nms <- xml_name(xml_contents(xml_find_all(xml, sprintf("/d1:codeBook/d1:dataDscr/d1:var[%s]", i))))
                
                # xpath <- sprintf("/%scodeBook/%sdataDscr/%svar[%s]", dns, dns, dns, i)
                # vars_i <- xml2::xml_find_first(xml, xpath)

                measurement <- xml2::xml_attr(vars[i], "nature")
                na_values <- NULL
                na_range <- NULL
                xpath <- sprintf("%sinvalrng/%srange", dns, dns)
                na_range[1] <- admisc::asNumeric(xml2::xml_attr(xml2::xml_find_first(vars[i], xpath), "min"))
                na_range[2] <- admisc::asNumeric(xml2::xml_attr(xml2::xml_find_first(vars[i], xpath), "max"))
                if (all(is.na(na_range))) {
                    na_range <- NULL
                }
                else {
                    if (is.na(na_range[1])) na_range[1] <- -Inf
                    if (is.na(na_range[2])) na_range[2] <- Inf
                }
                
                xpath <- sprintf("%scatgry/%scatValu", dns, dns)
                values <- cleanup(xml2::xml_text(xml2::xml_find_all(vars[i], xpath)))

                xpath <- sprintf("%svarFormat", dns)
                vformat <- xml2::xml_find_first(vars[i], xpath)
                type <- xml2::xml_attr(vformat, "type")
                varFormat <- xml2::xml_text(vformat)
                
                if (length(values) > 0) {
                    
                    catgry <- xml2::xml_find_all(vars[i], sprintf("%scatgry", dns))
                    
                    na_values <- c(na_values, values[unlist(lapply(catgry, function(x) {
                        grepl("Y", xml2::xml_attr(x, "missing"))
                    }))])
                    
                    labl <- unlist(lapply(catgry, function(x) {
                        xml2::xml_text(xml2::xml_find_first(x, sprintf("%slabl", dns)))
                    }))
                    
                    values <- values[!is.na(labl)]
                    labl <- cleanup(labl[!is.na(labl)])
                    
                    if (admisc::possibleNumeric(values)) {
                        values <- admisc::asNumeric(values)
                    }
                    
                    codeBook$dataDscr[[i]][["labels"]] <- values
                    names(codeBook$dataDscr[[i]][["labels"]]) <- labl
                }
                
                if (length(na_values) > 0) {
                    
                    if (admisc::possibleNumeric(na_values) & admisc::possibleNumeric(values)) {
                        na_values <- admisc::asNumeric(na_values)
                    }

                    na_values <- sort(unique(na_values))

                    if (!is.null(na_range) && is.numeric(na_values)) {
                        na_values <- na_values[na_values < na_range[1] | na_values > na_range[2]]
                    }
                    
                    if (length(na_values) > 0) {
                        codeBook$dataDscr[[i]]$na_values <- na_values
                    }
                }

                if (!is.null(na_range)) {
                    codeBook$dataDscr[[i]]$na_range <- na_range
                }

                if (is.na(measurement)) {
                    if (!is.na(type)) {
                        codeBook$dataDscr[[i]]$type <- "num" # default

                        if (type == "character") {
                            codeBook$dataDscr[[i]]$type <- "char"
                        }
                        else if (length(values) > 0) {
                            if (length(setdiff(values, na_values)) > 0) {
                                codeBook$dataDscr[[i]]$type <- "cat"
                            }
                        }
                    }
                }
                else {
                    if (is.element(measurement, c("nominal", "ordinal"))) {
                        codeBook$dataDscr[[i]]$type <- "cat"
                    }
                    else if (is.element(measurement, c("interval", "ratio"))) {
                        codeBook$dataDscr[[i]]$type <- "num"
                    }
                    else if (!is.na(type)) {
                        codeBook$dataDscr[[i]]$type <- type
                    }

                    codeBook$dataDscr[[i]]$measurement <- measurement
                }

                if (!is.na(vformat)) {
                    codeBook$dataDscr[[i]]$varFormat <- varFormat
                }

                if (identical(type, "character")) {
                    xpath <- sprintf("%stxt", dns)
                    txt <- cleanup(xml2::xml_text(xml2::xml_find_first(vars[i], xpath)))
                    if (!is.na(txt)) {
                        codeBook$dataDscr[[i]]$txt <- txt
                    }
                }
            }
        }
        else {
            if (tp$fileext[ff] == "SAV") {
                fargs <- names(formals(read_sav))
                arglist <- dots[is.element(names(dots), fargs)]
                arglist$file <- file.path(tp$completePath, tp$files[ff])
                arglist$user_na <- user_na
                arglist$encoding <- encoding
                data <- do.call(haven::read_sav, arglist)
            }
            else if (tp$fileext[ff] == "POR") {
                fargs <- names(formals(read_sav))
                arglist <- dots[is.element(names(dots), fargs)]
                arglist$file <- file.path(tp$completePath, tp$files[ff])
                arglist$user_na <- user_na
                data <- do.call(haven::read_por, arglist)
            }
            else if (tp$fileext[ff] == "DTA") {
                fargs <- names(formals(read_dta))
                arglist <- dots[is.element(names(dots), fargs)]
                arglist$file <- file.path(tp$completePath, tp$files[ff])
                arglist$encoding <- encoding
                data <- do.call(haven::read_dta, arglist)
            }
            else if (tp$fileext[ff] == "RDS") {
                data <- readRDS(file.path(tp$completePath, tp$files[ff]))
            }
            # not sure about SAS, as far as I understand the metadata is not embedded in the datafile
            # sometimes it might sit into a separate, catalog file or something (need to investigate)
            # else if (tp$fileext[ff] == "SAS7BDAT") {
            #     data <- haven::read_sas(file.path(tp$completePath, tp$files[ff]))
            # }
            
            codeBook <- list()
            codeBook$dataDscr <- collectMetadata(data)
        }
        
        codeBook$fileDscr$fileName <- tp$files[ff]

        filetypes <- c("SPSS", "SPSS", "Stata", "SAS", "R", "DDI", "Excel", "Excel")
        fileexts <- c("SAV", "POR", "DTA", "SAS7BDAT", "RDS", "XML", "XLS", "XLSX")

        codeBook$fileDscr$fileType <- filetypes[which(fileexts == tp$fileext[ff])]

        result[[ff]] <- codeBook
        
        if (save) {
            
            indent <- 4
            if (is.element("indent", names(dots))) {
                indent <- dots$indent
            }
            
            writeRlist(
                codeBook$dataDscr,
                OS = OS,
                indent = indent,
                dirpath = tp$completePath,
                filename = tp$filenames[ff]
            )
            
        }
    }

    names(result) <- tp$filenames
    
    if (singlefile) {
        if (!is.na(notes)) {
            if (grepl("# start data #", notes)) {
                # this can only be possible from an XML, DDI Codebook
                # therefore the varFormat should always be of an SPSS type
                notes <- unlist(strsplit(notes, split = "\\n"))
                data <- notes[
                    seq(
                        which(grepl("# start data #", notes)) + 1,
                        which(grepl("# end data #", notes)) - 1
                    )
                ]

                data <- read.csv(text = paste(data, collapse = "\n"), as.is = TRUE)
                # return(list(data, codeBook$dataDscr, declared = declared, spss = spss))

                # make_labelled is always and only about SPSS type of variables
                data <- make_labelled(
                    data,
                    codeBook$dataDscr,
                    declared = declared
                )

                embed <- TRUE
            }
        }
        
        if (embed & !is.null(data)) {
            codeBook$fileDscr$datafile <- data
        }
        
        return(codeBook)
    }
    else {
        return(result)
    }
}
