% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mutate_node_attrs.R
\name{mutate_node_attrs}
\alias{mutate_node_attrs}
\title{Mutate a set of node attribute values}
\usage{
mutate_node_attrs(graph, node_attr_from, expressions, node_attr_to = NULL)
}
\arguments{
\item{graph}{a graph object of class
\code{dgr_graph}.}

\item{node_attr_from}{the name of the node attribute
column from which values will be mutated. An
\code{NA} value can be provided here if node
attribute names are used in \code{expressions}
statements. Note that if \code{NA} is used, a value
must be provided for \code{node_attr_to}.}

\item{expressions}{one or more expressions for
mutation of all node attribute values specified by
\code{node_attr_from}. To reference the node
attribute given in \code{node_attr_from}, use the
\code{~} character. Otherwise, all node attributes
can be referenced by using the names of those node
attributes directly in the expressions. Expressions
are evaluated in the order provided.}

\item{node_attr_to}{an optionally supplied name of
a new node attribute to which the mutated values
will be applied. This will retain the original node
attribute(s) and its values. If \code{NA} is used
with \code{node_attr_from}, a value must be provided
here (since mutated values must be placed
somewhere).}
}
\value{
a graph object of class \code{dgr_graph}.
}
\description{
Within a graph's internal node data
frame (ndf), mutate numeric node attribute values
using one or more expressions. Optionally, one can
specify a different node attribute name and create
a new node attribute while retaining the original
node attribute and its values.
}
\examples{
# Create a graph with 3 nodes
graph <-
  create_graph() \%>\%
  add_path(3) \%>\%
  set_node_attrs(
    "width", c(3.4, 2.3, 7.2))

# Get the graph's internal ndf to show which
# node attributes are available
get_node_df(graph)
#>   id type label width
#> 1  1 <NA>     1   3.4
#> 2  2 <NA>     2   2.3
#> 3  3 <NA>     3   7.2

# Mutate the `width` node attribute, dividing
# each value by 2
graph <-
  graph \%>\%
  mutate_node_attrs("width", "~ / 2")

# Get the graph's internal ndf to show that the
# node attribute `width` had its values changed
get_node_df(graph)
#>   id type label width
#> 1  1 <NA>     1  1.70
#> 2  2 <NA>     2  1.15
#> 3  3 <NA>     3  3.60

# Create a new node attribute, called `length`,
# that is the log of values in `width` plus 2
# (and round all values to 2 decimal places)
graph <-
  graph \%>\%
  mutate_node_attrs(
    "width", "round(log(~) + 2, 2)", "length")

# Get the graph's internal ndf to show that
# the node attribute values had been mutated
# and used as the new node attribute `length`
get_node_df(graph)
#>   id type label width length
#> 1  1 <NA>     1  1.70   2.53
#> 2  2 <NA>     2  1.15   2.14
#> 3  3 <NA>     3  3.60   3.28

# Create a new node attribute called `area`,
# which is the product of the `width` and
# `length` attributes; note that we can provide
# NA to `node_attr_from` since we are naming
# at least one of the node attributes in the
# `expressions` vector (and providing a new
# node attribute name: `area`)
graph <-
  graph \%>\%
  mutate_node_attrs(
    NA, "width * length", "area")

# Get the graph's internal ndf to show that
# the node attribute values had been multiplied
# together, creating a new node attribute `area`
get_node_df(graph)
#>   id type label width length   area
#> 1  1 <NA>     1  1.70   2.53  4.301
#> 2  2 <NA>     2  1.15   2.14  2.461
#> 3  3 <NA>     3  3.60   3.28 11.808
}

