% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mutate_node_attrs_ws.R
\name{mutate_node_attrs_ws}
\alias{mutate_node_attrs_ws}
\title{Mutate node attribute values for a selection of nodes}
\usage{
mutate_node_attrs_ws(graph, ...)
}
\arguments{
\item{graph}{A graph object of class \code{dgr_graph}.}

\item{...}{Expressions used for the mutation of node attributes. LHS of each
expression is either an existing or new node attribute name. The RHS can
consist of any valid R code that uses node attributes as variables.
Expressions are evaluated in the order provided, so, node attributes
created or modified are ready to use in subsequent expressions.}
}
\value{
A graph object of class \code{dgr_graph}.
}
\description{
Within a graph's internal node data frame (ndf), mutate node attribute values
only for nodes in a selection by using one or more expressions.
}
\details{
This function makes use of an active selection of nodes (and the function
ending with \verb{_ws} hints at this).

Selections of nodes can be performed using the following node selection
(\verb{select_*()}) functions: \code{\link[=select_nodes]{select_nodes()}}, \code{\link[=select_last_nodes_created]{select_last_nodes_created()}},
\code{\link[=select_nodes_by_degree]{select_nodes_by_degree()}}, \code{\link[=select_nodes_by_id]{select_nodes_by_id()}}, or
\code{\link[=select_nodes_in_neighborhood]{select_nodes_in_neighborhood()}}.

Selections of nodes can also be performed using the following traversal
(\verb{trav_*()}) functions: \code{\link[=trav_out]{trav_out()}}, \code{\link[=trav_in]{trav_in()}}, \code{\link[=trav_both]{trav_both()}},
\code{\link[=trav_out_node]{trav_out_node()}}, \code{\link[=trav_in_node]{trav_in_node()}}, \code{\link[=trav_out_until]{trav_out_until()}}, or
\code{\link[=trav_in_until]{trav_in_until()}}.
}
\examples{
# Create a graph with 3 nodes
# and then select node `1`
graph <-
  create_graph() \%>\%
  add_path(n = 3) \%>\%
  set_node_attrs(
    node_attr = width,
    values = c(1.4, 0.3, 1.1)) \%>\%
  select_nodes(nodes = 1)

# Get the graph's internal ndf
# to show which node attributes
# are available
graph \%>\% get_node_df()

# Mutate the `width` node
# attribute for the nodes
# only in the active selection
# of nodes (node `1`); here,
# we divide each value in the
# selection by 2
graph <-
  graph \%>\%
  mutate_node_attrs_ws(
    width = width / 2)

# Get the graph's internal
# ndf to show that the node
# attribute `width` was
# mutated only for node `1`
graph \%>\% get_node_df()

# Create a new node attribute,
# called `length`, that is the
# log of values in `width` plus
# 2 (and, also, round all values
# to 2 decimal places)
graph <-
  graph \%>\%
  clear_selection() \%>\%
  select_nodes(nodes = 2:3) \%>\%
  mutate_node_attrs_ws(
    length = (log(width) + 2) \%>\%
               round(2))

# Get the graph's internal ndf
# to show that the node attribute
# values had been mutated only
# for nodes `2` and `3` (since
# node `1` is excluded, an NA
# value is applied)
graph \%>\% get_node_df()

# Create a new node attribute
# called `area`, which is the
# product of the `width` and
# `length` attributes
graph <-
  graph \%>\%
  mutate_node_attrs_ws(
    area = width * length)

# Get the graph's internal ndf
# to show that the node attribute
# values had been multiplied
# together (with new attr `area`)
# for nodes `2` and `3`
graph \%>\% get_node_df()

# We can invert the selection
# and mutate node `1` several
# times to get an `area` value
# for that node
graph <-
  graph \%>\%
  invert_selection() \%>\%
  mutate_node_attrs_ws(
    length = (log(width) + 5) \%>\%
               round(2),
    area = width * length)

# Get the graph's internal ndf
# to show that the 2 mutations
# occurred for node `1`, yielding
# non-NA values for its node
# attributes without changing
# those of the other nodes
graph \%>\% get_node_df()

}
