\name{fAssocmatrix}
\alias{fAssocmatrix}
\title{
Extract expected associations over multiple permutations
}
\description{
This function combines the functions \code{fRanXY}, \code{dnearneigh} and \code{fAssoctable}, to enable the user to extract repeated expected associations over multiple permutations. Unlike \code{fAssoctable}, this function operates in a single step where the user is only required to input the home range of interest, the association distance threshold \code{Gprox} and those individuals to be used in the analysis. The user is offered the flexibility to include all individuals in the analysis, or a series of possible combinations of individuals based on mean group size and probability of occurance.
The output table generated by this function creates a series of 'expected' associations according to what would be expected if individuals within a population distributed themselves at random within their home range. The output table is a data frame with a permutation number, a group identifier (i.e. those individuals occurring within a defined distance threshold) and the individual id. Each line in a single permutation corresponds to a unique individual. 
Note, the running time of this finction is positively related to the  number of associations in the system. Running time can be decreased by reducing the number of permutations \code{iPerm}, limiting numbers of individuals contained in the study \code{iID} or by decreasing the maximum association distance threshold \code{Gprox}.
}
\usage{
fAssocmatrix(sPerm, Gprox, iextract, iID)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{sPerm}{
a sequence of numbers representing the number of permutations to be run. If 100 permutations are to be run, \code{iPerm=1:100}
}
  \item{Gprox}{
a numeric field representing the distance threshold (max) for an association to occur
}
  \item{iextract}{
either a \code{SpatialPolygonsDateFrame} or a \code{SpatialGridDataFrame} object. This determines whether points are sampled from within a bounding area or
according to a probability surface
}
  \item{iID}{
either a sequence of numbers \code{(=IDs)} or a matrix   
}
}

\value{
\item{Permutation }{a numeric vector containing the permutation number}
\item{Group }{a numeric vector representing the group id. Individuals which occur within the distance threshold (i.e. between 0 and \code{Gprox}) have the same group number}
\item{IDs }{a character vector containing the identity of the animal within each group. \code{IDs} will be unique within each permutation}

}

\author{
Ross Dwyer \email{ross.dwyer@uq.edu.au}
}

\seealso{
\code{\link{fRanXY}}, \code{\link{fAssoctable}}, \code{\link{fAssocmatrix}}, \code{\link{fAssocplot}}, \code{\link{dnearneigh}}
}
\examples{

## Load required packages
library(adehabitatHR)

## Load data
data(StudyArea)
data(Roos)

## Convert df into a spatial points df object
coordinates(Roos) <- ~x+y

## Estimation of UD and KUD 95% for the 10 animals
ud <- kernelUD(Roos,h="href",same4all=TRUE)
ver95 <- getverticeshr(ud,95,unin = c("m"),unout=c("ha"))
udsgdf <- as(estUDm2spixdf(ud),"SpatialGridDataFrame")

## Run 1 permutation using only the first 5 individuals using random points taken 
## from within each individuals 95% KUD.
## Associations are determined if individuals are within 50 m from one another
fAssocmatrix(sPerm=1,
             Gprox=50,
             iextract=ver95,
             iID=1:5)

## Run 20 permutations using individuals accoring to sighting probability (ave no. roos seen on survey  = 6) 
## Probability-weighted random points are extracted according to the UD
## Associations are determined if individuals are within 50 m from one another

## Extract sighting probability

ID_counts <- tapply(rep(1,length(Roos[[1]])),Roos[[1]],sum)
No_SurveyDays <- 400

pObs <- as.vector(ID_counts)/No_SurveyDays # proportion of time individuals found on survey 
nameObs <- names(ID_counts) 
TotalAnimals <- 10 # Total number of animals in study
samplesize <- 6 # Number of IDs to include in simulations (with replacement)
bootlength <- 20 # Number of permutations

ExpProb <- data.frame(Kangaroo=nameObs,Probability=pObs) # Table of Expected Probabilities

## Generate ID combinations based on observation probability (with replacements)  
pID <- sapply(1:bootlength,function(i) sample(x=TotalAnimals,
                                              size=samplesize,
                                              replace=FALSE,
                                              prob=pObs))

## What individuals are contained in the first permutation?
nameObs[pID[,1]] 
## What individuals are contained in the last permutation?
nameObs[pID[,bootlength]]

## Run association matrix function to generate multiple permutations of 'expected' associations
AMud_50m <- fAssocmatrix(sPerm=1:bootlength,
             	         Gprox=50,
             	         iextract=udsgdf,
             	         iID=pID)
}

