% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ADBH_fun.R
\name{kernel.ADBH}
\alias{kernel.ADBH}
\alias{kernel.ADBH.crit}
\alias{kernel.ADBH.fast}
\title{Kernel functions for ADBH}
\usage{
kernel.ADBH.crit(msg = "", stepf, pv.numer, pv.denom, alpha, sorted.pv,
  bigMem = FALSE, verbose = TRUE)

kernel.ADBH.fast(msg = "", stepf, pv.numer, pv.denom, bigMem = FALSE,
  verbose = TRUE)
}
\arguments{
\item{msg}{a character string to be displayed if \code{verbose=TRUE}.}

\item{stepf}{a list of the CDFs under the null hypothesis of each p-value.}

\item{pv.numer}{a numeric vector. Contains all values of the p-values supports if we search for the critical constants. If not, contains only the observed p-values. Must be in increasing order.}

\item{pv.denom}{a numeric vector. Identical to \code{pv.numer} for a step-down procedure. Equals \code{c.m} for a step-up procedure.}

\item{alpha}{the target FDR level, a number strictly between 0 and 1.}

\item{sorted.pv}{a vector of observed p-values, in increasing order.}

\item{bigMem}{a boolean. If \code{TRUE}, the code uses matrixes and functions of the apply family as far as possible (faster for small number of hypotheses and support size, but slower otherwise due to memory management overhead). If \code{FALSE}, computations are done with \code{for} loops and chunks to conserve memory.}

\item{verbose}{a boolean indicating if \code{msg} must be printed. Used when \code{bigMem=TRUE}, to print messages informing if in-memory computation was successful or if loops and chunks were used as fallback.}
}
\value{
For \code{kernel.ADBH.crit}, 
a list which elements are:
\item{crit.consts}{a vector of critical constants}
\item{pval.transf}{a vector of transformed p-values (only for step-down direction).}

For \code{kernel.ADBH.fast},
a vector of transformed p-values.
}
\description{
Kernel functions that transform 
observed p-values or their support according to [AHSU] and [AHSD].
The output is used by \code{\link{ADBH}}.
Additionally, \code{kernel.ADBH.crit} computes and returns
the critical constants.
The end user should not use them.
}
\details{
When computing critical constants under step-down,
that is, when using \code{kernel.ADBH.crit} with
\code{pv.numer=pv.denom}, we still need
to get transformed p-values to compute
the adjusted p-values. Also, note that here the critical constants are computed by
the kernel function and not by the principal
function \code{\link{ADBH}}, contrary to what happens
with \code{\link{DBH}}. This is why \code{sorted.pv} is needed.

This version: 2018-02-20.
}
\examples{
X1 <- c(4, 2, 2, 14, 6, 9, 4, 0, 1)
X2 <- c(0, 0, 1, 3, 2, 1, 2, 2, 2)
N1 <- rep(148, 9)
N2 <- rep(132, 9)
Y1 <- N1-X1
Y2 <- N2-X2
df <- data.frame(X1,Y1,X2,Y2)
df
#Construction of the p-values and their support
df.formatted <- fisher.pvalues.support(counts = df, input = "noassoc")
raw.pvalues <- df.formatted$raw
pCDFlist <- df.formatted$support

m <- length(raw.pvalues)
alpha <- 0.05

#Compute the step functions from the supports
stepf <- build.stepfuns(pCDFlist)

#We stay in a step-down context, where pv.numer=pv.denom,
#for the sake of simplicity

#If not searching for critical constants, we use only the observed p-values
sorted.pvals <- sort(raw.pvalues)
y <- kernel.ADBH.fast("", stepf, sorted.pvals, sorted.pvals)

#If searching for critical constants, we use (almost) the complete support
pv.list.all <- unique(sort(as.numeric(unlist(pCDFlist))))
# apply the shortcut drawn from Lemma 4, that is
# c.1 >= the effective critical value associated to alpha/(m + alpha)
pv.list<-short.eff(pv.list.all, alpha/(m + alpha) )
# then re-add the observed p-values (needed to compute the adjusted p-values),
# because we may have removed some of them the shortcut
pv.list <- sort(unique(c(pv.list, sorted.pvals)))
# compute transformed support
y <- kernel.ADBH.crit("", stepf, pv.list, pv.list, alpha, sorted.pvals)
crit.constants <- y$crit.consts
#The following exists only for step-down direction
transformed.pvalues <- y$pval.transf

}
\seealso{
\code{\link{ADBH}}, \code{\link{DiscreteFDR}},
\code{\link{kernel.DBH}}, \code{\link{DBH}}
}
