#' Draw contour for EEM data
#' 
#' This function is a wrapper function for \code{\link{filled.contour}} to draw contour for EEM data.
#' 
#' @param x a list of EEM data generated by \code{\link{readEEM}} function or 
#' EEMweight object generated by \code{\link{extract}}-related functions.
#' @param n sample number. The number should not exceed \code{length(EEM)}
#' @param ncomp number of components
#' @param xlab (optional) x-axis label
#' @param ylab (optional) y-axis label
#' @param color.palette (optional) contour color palette. See \code{\link[grDevices]{palette}} for more details
#' @param nlevels (optional) number of levels used to separate range of intensity value
#' @param title (optional) plot title
#' @param ... (optional) further arguments passed to other methods of \code{\link[graphics]{filled.contour}} 
#' 
#' @return A figure is returned on the graphic device
#' 
#' @examples
#' # method for class "EEM"
#' data(applejuice)
#' drawEEM(applejuice, 1) # draw contour of the first sample
#' 
#' # method for class "EEMweight"
#' applejuice_uf <- unfold(applejuice) # unfold list into matrix
#' result <- prcomp(applejuice_uf) 
#' drawEEM(getLoading(result), 1) # plot loading of the first PC
#' 
#' @export
#' 
#' @importFrom graphics filled.contour
#' @importFrom colorRamps matlab.like
#' @importFrom reshape2 acast
drawEEM <- function(x, ...) UseMethod("drawEEM", x)

#' @describeIn drawEEM draw contour of EEM data created by \code{\link{readEEM}} function
#' @export
drawEEM.EEM <-
  function(x, n, xlab = "Excitation wavelength [nm]", ylab = "Emission wavelength [nm]", 
           color.palette = matlab.like, nlevels = 50, title = NULL, ...){
    
    # check number of argument
    if (nargs() < 2) stop("Not enough inputs. Aborted")
    
    # get information from EEM
    data <- x[[n]]
    
    # if title is not provided
    if (is.null(title)) {
        title <- names(x)[n] # if title is not provided, call it   
    }
    
    # draw contour
    x <- as.numeric(colnames(data)) #EX
    y <- as.numeric(rownames(data)) #EM
    z <- t(as.matrix(data))
    filled.contour(x, y, z, xlab = xlab, ylab = ylab, 
                   color.palette = color.palette, 
                   main = title, nlevels = nlevels, ...) 
  }

#' @describeIn drawEEM draw contours of the output from \code{\link[EEM]{getLoading}} and 
#' \code{\link[EEM]{getReg}}. 
#' @export
drawEEM.EEMweight <- function(x, ncomp, 
                       xlab = "Excitation wavelength [nm]", ylab = "Emission wavelength [nm]", 
                       color.palette = matlab.like, nlevels = 50, title = NULL,
                       ...){
    
#     # transpose if not in correct form
#     if (!isTRUE(grepl("EX...EM...", rownames(x$value)[1]))) {
#         x$value <- t(x$value)
#     }
    
    # check inputs such that ncomp cannot exceed totalcomp
    totalcomp <- dim(x$value)[2]
    if (ncomp > totalcomp) stop("ncomp cannot exceed totalcomp.")
    
    # extract data from x
    value <- x$value[,ncomp]
    id <- rownames(x$value)
    
    # get EX and EM
    EX <- getEX(id)
    EM <- getEM(id)

    data <- data.frame(ex = as.numeric(EX), em = as.numeric(EM), value = value)
    
    # cast data
    castedData <- acast(data, em~ex, value.var = "value")
    
    # prepare data for plotting
    ex <- as.numeric(colnames(castedData)) #EX
    em <- as.numeric(rownames(castedData)) #EM
    value <- t(as.matrix(castedData))
    
    # title
    if (is.null(title)) {
        title <- x$title #if title is not provided, call it
        title <- paste(title, ncomp)
        if (x$title %in% "Regression coefficient") {
            title <- paste0(x$title, " (", ncomp, " LV)")
            if (ncomp > 1) title <- sub("LV", "LVs", title)
        }
    } 
    
    # plotting
    filled.contour(ex, em, value, xlab = xlab, ylab = ylab, 
                   color.palette = color.palette, 
                   nlevels = nlevels, main = title, ...    
    ) 
}

#' @describeIn drawEEM draw contour of a matrix with columns being 
#' excitation wavelength and rows being emission wavelength
#' @export
drawEEM.matrix <-
    function(x, xlab = "Excitation wavelength [nm]", ylab = "Emission wavelength [nm]", 
             color.palette = matlab.like, nlevels = 50, title = NULL, ...){
            
        # get information from EEM
        data <- x
        
        # draw contour
        x <- as.numeric(colnames(data)) #EX
        y <- as.numeric(rownames(data)) #EM
        z <- t(as.matrix(data))
        filled.contour(x, y, z, xlab = xlab, ylab = ylab, 
                       color.palette = color.palette, 
                       main = title, nlevels = nlevels, ...) 
    }

#' @describeIn drawEEM draw contour of a data frame with columns being 
#' excitation wavelength and rows being emission wavelength
#' @export
drawEEM.data.frame <-
    function(x, xlab = "Excitation wavelength [nm]", ylab = "Emission wavelength [nm]", 
             color.palette = matlab.like, nlevels = 50, title = NULL, ...){
        x <- as.matrix(x)
        drawEEM.matrix(x, xlab = xlab, ylab = ylab, color.palette = color.palette,
                       nlevels = nlevels, title = title, ...)
    }

#' @describeIn drawEEM draw contour of a vector of numeric values which have names in 
#' the format of EX...EM...
#' @export
drawEEM.numeric <-
    function(x, xlab = "Excitation wavelength [nm]", ylab = "Emission wavelength [nm]", 
             color.palette = matlab.like, nlevels = 50, title = NULL, ...){
        
        # convert data to matrix form
        name <- names(x)
        EX <- getEX(name)
        EM <- getEM(name)
        data <- data.frame(ex = as.numeric(EX), em = as.numeric(EM), value = x)
        data.casted <- acast(data, em ~ ex, value.var = "value")
        
        drawEEM.matrix(data.casted, xlab = xlab, ylab = ylab, color.palette = color.palette,
                       nlevels = nlevels, title = title, ...)
    }