% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EGA.fit.R
\name{EGA.fit}
\alias{EGA.fit}
\title{\code{\link[EGAnet]{EGA}} Optimal Model Fit using the Total Entropy Fit Index  (\code{\link[EGAnet]{tefi}})}
\usage{
EGA.fit(
  data,
  model = c("glasso", "TMFG"),
  steps = c(3, 4, 5, 6, 7, 8),
  n = NULL
)
}
\arguments{
\item{data}{Matrix or data frame.
Dataset or correlation matrix}

\item{model}{Character.
A string indicating the method to use.
Defaults to \code{"glasso"}

Current options are:

\itemize{

\item{\strong{\code{"glasso"}}}
{Estimates the Gaussian graphical model using graphical LASSO with
extended Bayesian information criterion to select optimal regularization parameter.
See \code{\link[EGAnet]{EBICglasso.qgraph}}}

\item{\strong{\code{"TMFG"}}}
{Estimates a Triangulated Maximally Filtered Graph.
See \code{\link[NetworkToolbox]{TMFG}}}

}}

\item{steps}{Numeric vector.
Range of steps to be used in the model selection.
Defaults from 3 to 8 steps (based on Pons & Latapy, 2006)}

\item{n}{Integer.
Sample size (if the data provided is a correlation matrix)}
}
\value{
Returns a list containing:

\item{EGA}{The \code{\link[EGAnet]{EGA}} output for the best fitting model}

\item{steps}{The number of steps used in the best fitting model from
the \code{\link[igraph]{cluster_walktrap}} algorithm}

\item{EntropyFit}{The \code{\link[EGAnet]{tefi}} Index for the unique solutions given the range of steps
(vector names represent the number of steps)}

\item{Lowest.EntropyFit}{The lowest value for the \code{\link[EGAnet]{tefi}} Index}
}
\description{
Estimates the best fitting model using \code{\link[EGAnet]{EGA}}.
The number of steps in the \code{\link[igraph]{cluster_walktrap}} detection
algorithm is varied and unique community solutions are compared using
\code{\link[EGAnet]{tefi}}.
}
\examples{

# Load data
wmt <- wmt2[,7:24]

\donttest{# Estimate EGA
## plot.type = "qqraph" used for CRAN checks
## plot.type = "GGally" is the default
ega.wmt <- EGA(data = wmt, plot.type = "qgraph")

# Estimate optimal EGA
fit.wmt <- EGA.fit(data = wmt)

# Plot optimal fit
plot(fit.wmt$EGA, plot.type = "qgraph")

# Compare with CFA
cfa.ega <- CFA(ega.wmt, estimator = "WLSMV", data = wmt)
cfa.fit <- CFA(fit.wmt$EGA, estimator = "WLSMV", data = wmt)

lavaan::lavTestLRT(cfa.ega$fit, cfa.fit$fit, method = "satorra.bentler.2001")
}

}
\references{
# Entropy fit measures \cr
Golino, H., Moulder, R. G., Shi, D., Christensen, A. P., Garrido, L. E., Neito, M. D., Nesselroade, J., Sadana, R., Thiyagarajan, J. A., & Boker, S. M. (in press).
Entropy fit indices: New fit measures for assessing the structure and dimensionality of multiple latent variables.
\emph{Multivariate Behavioral Research}.
\doi{10.31234/osf.io/mtka2}

# Original implementation of EGA.fit \cr
Golino, H., Thiyagarajan, J. A., Sadana, M., Teles, M., Christensen, A. P., & Boker, S. M. (under review).
Investigating the broad domains of intrinsic capacity, functional ability, and environment:
An exploratory graph analysis approach for improving analytical methodologies for measuring healthy aging.
\emph{PsyArXiv}.
\doi{10.31234/osf.io/hj5mc}

# Walktrap algorithm \cr
Pons, P., & Latapy, M. (2006).
Computing communities in large networks using random walks.
\emph{Journal of Graph Algorithms and Applications}, \emph{10}, 191-218.
\doi{10.7155/jgaa.00185}
}
\seealso{
\code{\link[EGAnet]{bootEGA}} to investigate the stability of EGA's estimation via bootstrap,
\code{\link[EGAnet]{EGA}} to estimate the number of dimensions of an instrument using EGA,
and \code{\link[EGAnet]{CFA}} to verify the fit of the structure suggested by EGA using confirmatory factor analysis.
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
