% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootEGA.R
\name{bootEGA}
\alias{bootEGA}
\title{Dimension Stability Analysis of \code{\link[EGAnet]{EGA}}}
\usage{
bootEGA(
  data,
  uni = TRUE,
  iter,
  type = c("parametric", "resampling"),
  corr = c("cor_auto", "pearson", "spearman"),
  model = c("glasso", "TMFG"),
  model.args = list(),
  algorithm = c("walktrap", "louvain"),
  algorithm.args = list(),
  typicalStructure = TRUE,
  plot.typicalStructure = TRUE,
  plot.type = c("GGally", "qgraph"),
  plot.args = list(),
  ncores,
  ...
)
}
\arguments{
\item{data}{Matrix or data frame.
Includes the variables to be used in the \code{bootEGA} analysis}

\item{uni}{Boolean.
Should unidimensionality be checked?
Defaults to \code{TRUE}.
Set to \code{FALSE} to check for multidimensionality only.
If \code{TRUE}, then the same number of variables as the original
data (i.e., from argument \code{data}) are generated from a factor
model with one factor and loadings of .70. These data are then
appended to the original data and dimensionality is checked.
If the number of dimensions is one or two, then the original
data are unidimensional; otherwise, the data are multidimensional
(see Golino, Shi, et al., 2020 for more details)}

\item{iter}{Numeric integer.
Number of replica samples to generate from the bootstrap analysis.
At least \code{500} is recommended}

\item{type}{Character.
A string indicating the type of bootstrap to use.

Current options are:

\itemize{

\item{\strong{\code{"parametric"}}}
{Generates \code{n} new datasets (multivariate normal random distributions) based on the
original dataset, via the \code{\link[MASS]{mvrnorm}} function}

\item{\strong{\code{"resampling"}}}
{Generates n random subsamples of the original data}

}}

\item{corr}{Type of correlation matrix to compute. The default uses \code{\link[qgraph]{cor_auto}}.
Current options are:

\itemize{

\item{\strong{\code{cor_auto}}}
{Computes the correlation matrix using the \code{\link[qgraph]{cor_auto}} function from
\code{\link[qgraph]{qgraph}}}.

\item{\strong{\code{pearson}}}
{Computes Pearson's correlation coefficient using the pairwise complete observations via
the \code{\link[stats]{cor}}} function.

\item{\strong{\code{spearman}}}
{Computes Spearman's correlation coefficient using the pairwise complete observations via
the \code{\link[stats]{cor}}} function.
}}

\item{model}{Character.
A string indicating the method to use.

Current options are:

\itemize{

\item{\strong{\code{glasso}}}
{Estimates the Gaussian graphical model using graphical LASSO with
extended Bayesian information criterion to select optimal regularization parameter.
This is the default method}

\item{\strong{\code{TMFG}}}
{Estimates a Triangulated Maximally Filtered Graph}

}}

\item{model.args}{List.
A list of additional arguments for \code{\link[EGAnet]{EBICglasso.qgraph}}
or \code{\link[NetworkToolbox]{TMFG}}}

\item{algorithm}{A string indicating the algorithm to use or a function from \code{\link{igraph}}
Current options are:

\itemize{

\item{\strong{\code{walktrap}}}
{Computes the Walktrap algorithm using \code{\link[igraph]{cluster_walktrap}}}

\item{\strong{\code{louvain}}}
{Computes the Walktrap algorithm using \code{\link[igraph]{cluster_louvain}}}

}}

\item{algorithm.args}{List.
A list of additional arguments for \code{\link[igraph]{cluster_walktrap}}, \code{\link[igraph]{cluster_louvain}},
or some other community detection algorithm function (see examples)}

\item{typicalStructure}{Boolean.
If \code{TRUE}, returns the typical network of partial correlations
(estimated via graphical lasso or via TMFG) and estimates its dimensions.
The "typical network" is the median of all pairwise correlations over the \emph{n} bootstraps.
Defaults to \code{TRUE}}

\item{plot.typicalStructure}{Boolean.
If \code{TRUE}, returns a plot of the typical network (partial correlations),
which is the median of all pairwise correlations over the \emph{n} bootstraps,
and its estimated dimensions.
Defaults to \code{TRUE}}

\item{plot.type}{Character.
Plot system to use.
Current options are \code{\link[qgraph]{qgraph}} and \code{\link{GGally}}.
Defaults to \code{"GGally"}.}

\item{plot.args}{List.
A list of additional arguments for the network plot.
For \code{plot.type = "qgraph"}:

\itemize{

\item{\strong{\code{vsize}}}
{Size of the nodes. Defaults to 6.}

}
For \code{plot.type = "GGally"} (see \code{\link[GGally]{ggnet2}} for
full list of arguments):

\itemize{

\item{\strong{\code{vsize}}}
{Size of the nodes. Defaults to 6.}

\item{\strong{\code{label.size}}}
{Size of the labels. Defaults to 5.}

\item{\strong{\code{alpha}}}
{The level of transparency of the nodes, which might be a single value or a vector of values. Defaults to 0.7.}

\item{\strong{\code{edge.alpha}}}
{The level of transparency of the edges, which might be a single value or a vector of values. Defaults to 0.4.}

 \item{\strong{\code{legend.names}}}
{A vector with names for each dimension}

\item{\strong{\code{color.palette}}}
{The color palette for the nodes. For custom colors,
enter HEX codes for each dimension in a vector.
See \code{\link[EGAnet]{color_palette_EGA}} for 
more details and examples}

}}

\item{ncores}{Numeric.
Number of cores to use in computing results.
Defaults to \code{parallel::detectCores() / 2} or half of your
computer's processing power.
Set to \code{1} to not use parallel computing

If you're unsure how many cores your computer has,
then use the following code: \code{parallel::detectCores()}}

\item{...}{Additional arguments.
Used for deprecated arguments from previous versions of \code{\link{EGA}}}
}
\value{
Returns a list containing:

\item{iter}{Number of replica samples in bootstrap}

\item{boot.ndim}{Number of dimensions identified in each replica sample}

\item{boot.wc}{Item allocation for each replica sample}

\item{bootGraphs}{Networks of each replica sample}

\item{summary.table}{Summary table containing number of replica samples, median,
standard deviation, standard error, 95\% confidence intervals, and quantiles (lower = 2.5\% and upper = 97.5\%)}

\item{frequency}{Proportion of times the number of dimensions was identified
(e.g., .85 of 1,000 = 850 times that specific number of dimensions was found)}

\item{EGA}{Output of the original \code{\link[EGAnet]{EGA}} results}

\item{typicalGraph}{A list containing:

\itemize{

\item{\strong{\code{graph}}}
{Network matrix of the median network structure}

\item{\strong{\code{typical.dim.variables}}}
{An ordered matrix of item allocation}

\item{\strong{\code{wc}}}
{Item allocation of the median network}
   }
}
}
\description{
\code{bootEGA} Estimates the number of dimensions of \emph{n} bootstraps
using the empirical (partial) correlation matrix (parametric) or resampling from
the empirical dataset (non-parametric). It also estimates a typical
median network structure, which is formed by the median or mean pairwise (partial)
correlations over the \emph{n} bootstraps.
}
\examples{
# Load data
wmt <- wmt2[,7:24]

\donttest{# bootEGA glasso example
## plot.type = "qqraph" used for CRAN checks
## plot.type = "GGally" is the default
boot.wmt <- bootEGA(data = wmt, iter = 500, plot.type = "qgraph",
type = "parametric", ncores = 2)

# bootEGA TMFG example
boot.wmt <- bootEGA(data = wmt, iter = 500, model = "TMFG",
plot.type = "qgraph", type = "parametric", ncores = 2)

# bootEGA Louvain example
boot.wmt <- bootEGA(data = wmt, iter = 500, algorithm = "louvain",
plot.type = "qgraph", type = "parametric", ncores = 2)

# bootEGA Spinglass example
boot.wmt <- bootEGA(data = wmt, iter = 500, model = "TMFG", plot.type = "qgraph",
algorithm = igraph::cluster_spinglass, type = "parametric", ncores = 2)
}

# Load data
intwl <- intelligenceBattery[,8:66]

\donttest{# Another bootEGA example
boot.intwl <- bootEGA(data = intwl, iter = 500,
plot.type = "qgraph", type = "parametric", ncores = 2)
}

}
\references{
# Original implementation of bootEGA \cr
Christensen, A. P., & Golino, H. (2019).
Estimating the stability of the number of factors via Bootstrap Exploratory Graph Analysis: A tutorial.
\emph{PsyArXiv}.
\doi{10.31234/osf.io/9deay}

# Structural consistency (see \code{\link[EGAnet]{dimStability}}) \cr
Christensen, A. P., Golino, H., & Silvia, P. J. (in press).
A psychometric network perspective on the validity and validation of personality trait questionnaires.
\emph{European Journal of Personality}.
\doi{10.1002/per.2265}
}
\seealso{
\code{\link[EGAnet]{EGA}} to estimate the number of dimensions of an instrument using EGA
and \code{\link[EGAnet]{CFA}} to verify the fit of the structure suggested by EGA using confirmatory factor analysis.
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
