% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EGA.estimate.R
\name{EGA.estimate}
\alias{EGA.estimate}
\title{A Sub-routine Function for \code{EGA}}
\usage{
EGA.estimate(
  data,
  n = NULL,
  corr = c("cor_auto", "pearson", "spearman"),
  model = c("glasso", "TMFG"),
  model.args = list(),
  algorithm = c("walktrap", "leiden", "louvain"),
  algorithm.args = list(),
  consensus.method = c("highest_modularity", "most_common", "iterative", "lowest_tefi"),
  consensus.iter = 100,
  ...
)
}
\arguments{
\item{data}{Matrix or data frame.
Variables (down columns) or correlation matrix.
If the input is a correlation matrix,
then argument \code{n} (number of cases) is \strong{required}}

\item{n}{Integer.
Sample size if \code{data} provided is a correlation matrix}

\item{corr}{Type of correlation matrix to compute. The default uses \code{\link[qgraph]{cor_auto}}.
Current options are:

\itemize{

\item{\strong{\code{cor_auto}}}
{Computes the correlation matrix using the \code{\link[qgraph]{cor_auto}} function from
\code{\link[qgraph]{qgraph}}}.

\item{\strong{\code{pearson}}}
{Computes Pearson's correlation coefficient using the pairwise complete observations via
the \code{\link[stats]{cor}}} function.

\item{\strong{\code{spearman}}}
{Computes Spearman's correlation coefficient using the pairwise complete observations via
the \code{\link[stats]{cor}}} function.
}}

\item{model}{Character.
A string indicating the method to use.

Current options are:

\itemize{

\item{\strong{\code{glasso}}}
{Estimates the Gaussian graphical model using graphical LASSO with
extended Bayesian information criterion to select optimal regularization parameter.
This is the default method}

\item{\strong{\code{TMFG}}}
{Estimates a Triangulated Maximally Filtered Graph}

}}

\item{model.args}{List.
A list of additional arguments for \code{\link[EGAnet]{EBICglasso.qgraph}}
or \code{\link[EGAnet]{TMFG}}}

\item{algorithm}{A string indicating the algorithm to use or a function from \code{\link{igraph}}
Current options are:

\itemize{

\item{\strong{\code{walktrap}}}
{Computes the Walktrap algorithm using \code{\link[igraph]{cluster_walktrap}}}

\item{\strong{\code{leiden}}}
{Computes the Leiden algorithm using \code{\link[igraph]{cluster_leiden}}}

\item{\strong{\code{louvain}}}
{Computes the Louvain algorithm using \code{\link[igraph]{cluster_louvain}}}

}}

\item{algorithm.args}{List.
A list of additional arguments for \code{\link[igraph]{cluster_walktrap}}, \code{\link[igraph]{cluster_louvain}},
or some other community detection algorithm function (see examples)}

\item{consensus.method}{Character.
What consensus clustering method should be used? 
Defaults to \code{"highest_modularity"}.
Current options are:

\itemize{

\item{\strong{\code{highest_modularity}}}
{Uses the community solution that achieves the highest modularity
across iterations}

\item{\strong{\code{most_common}}}
{Uses the community solution that is found the most
across iterations}

\item{\strong{\code{iterative}}}
{Identifies the most common community solutions across iterations
and determines how often nodes appear in the same community together.
A threshold of 0.30 is used to set low proportions to zero.
This process repeats iteratively until all nodes have a proportion of
1 in the community solution.
}

\item{\code{lowest_tefi}}
{Uses the community solution that achieves the lowest \code{\link[EGAnet]{tefi}}
across iterations}

}}

\item{consensus.iter}{Numeric.
Number of iterations to perform in consensus clustering for the Louvain algorithm
(see Lancichinetti & Fortunato, 2012).
Defaults to \code{100}}

\item{...}{Additional arguments.
Used for deprecated arguments from previous versions of \code{\link{EGA}}}
}
\value{
Returns a list containing:

\item{estimated.network}{A symmetric network estimated using either the
\code{\link{EBICglasso.qgraph}} or \code{\link[EGAnet]{TMFG}}}

\item{wc}{A vector representing the community (dimension) membership
of each node in the network. \code{NA} values mean that the node
was disconnected from the network}

\item{n.dim}{A scalar of how many total dimensions were identified in the network}

\item{cor.data}{The zero-order correlation matrix}
}
\description{
Estimates the number of dimensions of a given dataset or correlation matrix
using the graphical lasso (\code{\link{EBICglasso.qgraph}}) or the
Triangulated Maximally Filtered Graph (\code{\link[EGAnet]{TMFG}})
network estimation methods.
}
\details{
Two community detection algorithms, Walktrap (Pons & Latapy, 2006) and
Louvain (Blondel et al., 2008), are pre-programmed because of their
superior performance in simulation studies on psychological
data generated from factor models (Christensen & Golino; 2020; Golino et al., 2020).
Notably, any community detection algorithm from the \code{\link{igraph}}
can be used to estimate the number of communities (see examples).
}
\examples{
# Obtain data
wmt <- wmt2[,7:24]

\donttest{# Estimate EGA
ega.wmt <- EGA.estimate(data = wmt)

# Estimate EGAtmfg
ega.wmt.tmfg <- EGA.estimate(data = wmt, model = "TMFG")

# Estimate EGA with Louvain algorithm
ega.wmt.louvain <- EGA.estimate(data = wmt, algorithm = "louvain")

# Estimate EGA with Spinglass algorithm
ega.wmt.spinglass <- EGA.estimate(
  data = wmt,
  algorithm = igraph::cluster_spinglass # any {igraph} algorithm
)}

}
\references{
# Louvain algorithm \cr
Blondel, V. D., Guillaume, J.-L., Lambiotte, R., & Lefebvre, E. (2008).
Fast unfolding of communities in large networks.
\emph{Journal of Statistical Mechanics: Theory and Experiment}, \emph{2008}, P10008.

# Compared all \emph{igraph} community detections algorithms, introduced Louvain algorithm, simulation with continuous and polytomous data \cr
Christensen, A. P., & Golino, H. (under review).
Estimating factors with psychometric networks: A Monte Carlo simulation comparing community detection algorithms.
\emph{PsyArXiv}.

# Original simulation and implementation of EGA \cr
Golino, H. F., & Epskamp, S. (2017).
Exploratory graph analysis: A new approach for estimating the number of dimensions in psychological research.
\emph{PLoS ONE}, \emph{12}, e0174035.

Golino, H. F., & Demetriou, A. (2017).
Estimating the dimensionality of intelligence like data using Exploratory Graph Analysis.
\emph{Intelligence}, \emph{62}, 54-70.

# Current implementation of EGA, introduced unidimensional checks, continuous and dichotomous data \cr
Golino, H., Shi, D., Christensen, A. P., Garrido, L. E., Nieto, M. D., Sadana, R., & Thiyagarajan, J. A. (2020).
Investigating the performance of Exploratory Graph Analysis and traditional techniques to identify the number of latent factors: A simulation and tutorial.
\emph{Psychological Methods}, \emph{25}, 292-320.

# Walktrap algorithm \cr
Pons, P., & Latapy, M. (2006).
Computing communities in large networks using random walks.
\emph{Journal of Graph Algorithms and Applications}, \emph{10}, 191-218.
}
\seealso{
\code{\link{bootEGA}} to investigate the stability of EGA's estimation via bootstrap
and \code{\link{CFA}} to verify the fit of the structure suggested by EGA using confirmatory factor analysis.
}
\author{
Alexander P. Christensen <alexpaulchristensen at gmail.com> and Hudson Golino <hfg9s at virginia.edu>
}
