% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/community.detection.R
\name{community.detection}
\alias{community.detection}
\title{Apply a Community Detection Algorithm}
\usage{
community.detection(
  network,
  algorithm = c("edge_betweenness", "fast_greedy", "fluid", "infomap", "label_prop",
    "leading_eigen", "leiden", "louvain", "optimal", "spinglass", "walktrap"),
  allow.singleton = FALSE,
  membership.only = TRUE,
  ...
)
}
\arguments{
\item{network}{Matrix or \code{\link{igraph}} network object}

\item{algorithm}{Character or \code{\link{igraph}} \code{cluster_*} function
(length = 1).
Available options:

\itemize{

\item \code{"edge_betweenness"} --- See \code{\link[igraph]{cluster_edge_betweenness}} for more details

\item \code{"fast_greedy"} --- See \code{\link[igraph]{cluster_fast_greedy}} for more details

\item \code{"fluid"} --- See \code{\link[igraph]{cluster_fluid_communities}} for more details

\item \code{"infomap"} --- See \code{\link[igraph]{cluster_infomap}} for more details

\item \code{"label_prop"} --- See \code{\link[igraph]{cluster_label_prop}} for more details

\item \code{"leading_eigen"} --- See \code{\link[igraph]{cluster_leading_eigen}} for more details

\item \code{"leiden"} --- See \code{\link[igraph]{cluster_leiden}} for more details.
\emph{Note}: The Leiden algorithm will default to the
modularity objective function (\code{objective_function = "modularity"}). 
Set \code{objective_function = "CPM"} to use the 
Constant Potts Model instead (see examples)

\item \code{"louvain"} --- See \code{\link[igraph]{cluster_louvain}} for more details

\item \code{"optimal"} --- See \code{\link[igraph]{cluster_optimal}} for more details

\item \code{"spinglass"} --- See \code{\link[igraph]{cluster_spinglass}} for more details

\item \code{"walktrap"} --- See \code{\link[igraph]{cluster_walktrap}} for more details

}}

\item{allow.singleton}{Boolean (length = 1).
Whether singleton or single node communities should be allowed.
Defaults to \code{FALSE}.
When \code{FALSE}, singleton communities will be set to
missing (\code{NA}); otherwise, when \code{TRUE}, singleton
communities will be allowed}

\item{membership.only}{Boolean (length = 1).
Whether the memberships only should be output.
Defaults to \code{TRUE}.
Set to \code{FALSE} to obtain all output for the
community detection algorithm}

\item{...}{Additional arguments to be passed on to
\code{\link{igraph}}'s community detection functions
(see \code{algorithm} for link to arguments of each algorithm)}
}
\value{
Returns memberships from a community detection algorithm
}
\description{
General function to apply community detection algorithms available in
\code{\link{igraph}}. Follows the \code{\link{EGAnet}} approach of setting
singleton and disconnected nodes to missing (\code{NA})
}
\examples{
# Load data
wmt <- wmt2[,7:24]

# Estimate network
network <- EBICglasso.qgraph(data = wmt)

# Compute Edge Betweenness
community.detection(network, algorithm = "edge_betweenness")

# Compute Fast Greedy
community.detection(network, algorithm = "fast_greedy")

# Compute Fluid
community.detection(
  network, algorithm = "fluid",
  no.of.communities = 2 # needs to be set
)

# Compute Infomap
community.detection(network, algorithm = "infomap")

# Compute Label Propagation
community.detection(network, algorithm = "label_prop")

# Compute Leading Eigenvector
community.detection(network, algorithm = "leading_eigen")

# Compute Leiden (with modularity)
community.detection(
  network, algorithm = "leiden",
  objective_function = "modularity"
)

# Compute Leiden (with CPM)
community.detection(
  network, algorithm = "leiden",
  objective_function = "CPM",
  resolution_parameter = 0.05 # "edge density"
)

# Compute Louvain
community.detection(network, algorithm = "louvain")

# Compute Optimal (identifies maximum modularity solution)
community.detection(network, algorithm = "optimal")

# Compute Spinglass
community.detection(network, algorithm = "spinglass")

# Compute Walktrap
community.detection(network, algorithm = "walktrap")

# Example with {igraph} network
community.detection(
  convert2igraph(network), algorithm = "walktrap"
)

}
\references{
Csardi, G., & Nepusz, T. (2006). 
The igraph software package for complex network research.
\emph{InterJournal, Complex Systems}, 1695.
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
