test.lw <-
structure(function # Function to test maximum valid lw value.
### This function performs the weight transformation of the data matrix
### after Klovan & Imbrie (1971) with different weight limits and returns the 
### maximum value for which the transformation remains stable.
(X,
### Numeric matrix with m samples (rows) and n variables (columns).
lw
### Numeric vector specifying the weight transformation limit, i.e. 
### quantile; default is 0.
){
  ## check/set default value
  if(missing(lw) == TRUE) {lw = 0}
  
  ## loop through all elements of vector lw
  for(i in 1:length(lw)) {
    ## rescale X constant sum
    X  <- X / apply(X, 1, sum)
    ## define quantile extraction function
    qts <- function(X, lw) quantile(X, c(lw, 1-lw), type = 5)
    ## apply quantile function column-wise
    ls <- t(apply(X, 2, qts, lw = lw[i]))
    ## perform weight-transformation
    W <- t((t(X) - ls[,1]) / (ls[,2] - ls[,1]))
    ## optional break when transformation is erroneous
    if (is.na(mean(W))) {
      i = i - 1
      break}
  }
 
  ## assign last valid step number and lw-value
  step  <- i
  lw.max  <- lw[i]
  
  ##value<< A list with objects
  list(step = step, ##<< Numeric scalar with position of last valid value.
       lw.max = lw.max) ##<< Numeric scalar with last valid value of lw.
  ##end<<
  
  ##references<<
  ## Dietze E, Hartmann K, Diekmann B, IJmker J, Lehmkuhl F, Opitz S, 
  ## Stauch G, Wuennemann B, Borchers A. 2012. An end-member algorithm for 
  ## deciphering modern detrital processes from lake sediments of Lake Donggi 
  ## Cona, NE Tibetan Plateau, China. Sedimentary Geology 243-244: 169-180. \cr
  ## Klovan JE, Imbrie J. 1971. An Algorithm and FORTRAN-IV Program for 
  ## Large-Scale Q-Mode Factor Analysis and Calculation of Factor Scores. 
  ## Mathematical Geology 3: 61-77.
    
  ##seealso<<
  ## \code{\link{EMMA}}, \code{\link{check.data}}, 
  ## \code{\link{test.parameters}}
  
  ##keyword<<
  ## EMMA
}, ex = function(){
  ## load example data set
  data(X.artificial, envir = environment())
  
  ## create weight transformation limits vector
  lw <- seq(0, 0.6, by = 0.02)
  
  ## test the vector
  test.lw(X = X.artificial, lw = lw)
})
