% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EMSS.R
\name{EMSS}
\alias{EMSS}
\title{EM type Estimation Methods for the Heckman's Sample Selection Model}
\usage{
EMSS(
  response,
  selection,
  data,
  method = "ECM",
  initial.param = NULL,
  eps = 10^(-10)
)
}
\arguments{
\item{response}{a formula for the response equation.}

\item{selection}{a formula for the selection equation.}

\item{data}{a data frame and data has to be included with the form of \code{data.frame}.}

\item{method}{a character indicating which method to be used. \code{ECM} stands for Expectation Conditional Maximization, and
\code{ECMnr} stands for Expectation Conditioncal Maximization with Newton-Raphson, and \code{ECME} for Expectation Conditional Maximization Either.}

\item{initial.param}{a vector, initial parameter values for the estimation. The length of the initial parameters has to
be same as the length of parameters, which are to be estimated.}

\item{eps}{a numerical error value for the end of the loop. A minimum value that can be arbitrarily set
to terminate the iteration of the function, in order to find the optimal parameter estimation.}
}
\value{
\code{ECM} returns an object of class \code{"ECM"}.
The object class \code{"ECM"} is a list
containing the following components.

\item{call}{a matched call.}
\item{estimate_response}{estimated regression coefficients for the response formula.}
\item{estimate_selection}{estimated regression coefficients for the sample selection formula.}
\item{estimate_sigma}{an estimated scale paramter for the bivariate normal distribution.}
\item{estimate_rho}{an estimated correlation coefficient for the bivariate normal distribution.}
\item{hessian_mat}{hessian matrix for parameters.}
\item{resp_leng}{the numbers of coefficients for the response formula}
\item{select_leng}{the numbers of coefficients for the selection formula}
\item{Q_value}{the vallue of the Q function for EM type algorithms}
\item{names_response}{names of regression coefficients for the reponse formula.}
\item{names_selection}{names of regression coefficients for the selection formula.}
}
\description{
Some algorithms: \code{ECM}, \code{ECMnr} and \code{ECME} can be used to estimate parameters
in Heckman selection model and contain the advantages of the \code{EM} algorithm: easy
implementation and numerical stability. \code{"ECMnr"} stands for Expectation/Conditioncal
Maximization with Newton-Raphson, and \code{"ECME"} for Expectation/Conditional Maximization Either.
}
\details{
The dependent variable of the selection equation (specified by argument selection) must have exactly
two levels (e.g., 'FALSE' and 'TRUE', or '0' and '1'). The default argument method is "ECM" and the
default start values ("NULL") are obtained by two-step estimation of this model through the commend
\code{selection} from the package \code{sampleSelection}. NA's are allowed in the data. These are
ignored if the corresponding outcome is unobserved, otherwise observations which contain NA
(either in selection or outcome) are changed to 0.
}
\section{Background}{

Heckman selection model is classic to deal with the data where the outcome is partially observed and
the missing part is not at random. Heckman (1979) developed \code{2-step} and maximum likelihood
estimation (\code{MLE}) to do the estimation for this selection model. And these two method are
described in R package \code{sampleSelection} by Toomet and Henningsen (2008). Zhelonkin et al. (2016)
developed robust 2-stage method which performs more robustly than the 2-step method to deal with the
data where outlying observations exist and \code{ssmrob} package is available. Zhao et al. (2020) extended
EM algorithm to more general cases resulting in three algorithms: ECM, ECM(NR), and ECME. They also own
EM algorithm's main advantages, namely, stability and ease of implementation.
}

\examples{
data(Smoke, package = "EMSS")
ex1 <- EMSS(response = cigs_intervals ~ educ,
           selection = smoker ~ educ + age,
           data = Smoke)
print(ex1)

data(Smoke, package = "EMSS")
ex2 <- EMSS(response = cigs_intervals ~ educ,
           selection =  smoker ~ educ + age,
           data = Smoke, method="ECMnr")
print(ex2)

## example using random numbers with exclusion restriction

N <- 1000
errps <- mvtnorm::rmvnorm(N,c(0,0),matrix(c(1,0.5,0.5,1),2,2) )
xs <- runif(N)
ys <- xs+errps[,1]>0
xo <- runif(N)
yo <- (xo+errps[,2])*(ys>0)
ex3 <- EMSS(response = yo ~ xo,
           selection = ys ~ xs,
           initial.param = c(rep(0,4), 0.3, 0.6), method="ECMnr")
print(ex3)

}
\references{
Heckman, J. (1979) Sample selection bias as a specication error. \emph{Econometrica}, 47, 153-161.

Toomet, O. and Henningsen, A. (2008) Sample selection models in R:Package sampleSelection. \emph{Journal of Statistical Software}, 27, 1-23.

Zhao,J., Kim, H.-J. and Kim, H.-M. (2020) New EM-type algorithms for the Heckman selection model. \emph{Computational Statistics and Data Analysis}, 146, https://doi.org/10.1016/j.csda.2020.106930.

Zhelonkin, M., Genton, M.G. and Ronchetti, E. (2016) Robust inference in sample selection models. \emph{Journal of the Royal Statistical Society Series B}, 78, 805-827.
}
