\name{rates}
\alias{rates}
\alias{probs}
\title{Declustering Probabilities, Background Seismicity Rate and
Clustering Coefficient}
\description{
  Functions to estimate the declustring probabilities, background seismicity
  rate and clustering (triggering) coefficient for a fitted ETAS model.
}
\usage{
  probs(fit)
  rates(fit, dimyx=NULL, plot.it=TRUE)
}
\arguments{
  \item{fit}{A fitted ETAS model. An object of class \code{"etas"}.}
  \item{dimyx}{Dimensions of the rectangular discretization
           grid for the geographical study region.
           A numeric vector of length 2.}
  \item{plot.it}{Logical flag indicating whether to plot the rates or
           return them as pixel images.}
}
\value{
  If \code{plot.it=TRUE}, the function produces plots of the
  background seismicity rate and clustering coefficient.

  If \code{plot.it=FALSE}, it returns a list of length 3,
  with the total spatial intensity, background seismicity rate
  and clustering as components (objects of
  \code{im} class in the package \code{spatstat}).
}
\details{
  The function \code{probs} returns estimates of the declustring probabilities
  \deqn{p_j = 1 - \frac{\mu(x_j, y_j)}{lambda(t_j, x_j, y_j|H_{t_j})}}{p[j] = 1 - mu(x[j], y[j])/lambda(t[j], x[j], y[j]|H_t[j])}
  where \eqn{1-p_j}{1-p[j]} is the probability that event \eqn{j}{j}
  is a background event.

  The function \code{rates} returns kernel estimate of the background
  seismicity rate \eqn{\mu(x,y)}{mu(x,y)} and the clustering (triggering)
  coefficient
  \deqn{\omega(x,y)=1-\frac{\mu(x,y)}{\Lambda(x,y)}}{omega(x,y)=1-mu(x,y)/Lambda(x,y)}
  where \eqn{\Lambda(x,y)}{Lambda(x,y)} is the total spatial intensity
  function.

  The argument \code{dimyx} determines the rectangular discretization
  grid dimensions. If it is given, then it must be a numeric vector
  of length 2 where the first component \code{dimyx[1]} is the
  number of subdivisions in the y-direction (latitude) and the
  second component \code{dimyx[2]} is the number of subdivisions
  in the x-direction (longitude). %The default (\code{NULL}) sets
  %it to be \code{dimyx=c(128, 128)}.
}
\references{
  Zhuang J, Ogata Y, Vere-Jones D (2002).
  Stochastic Declustering of Space-Time Earthquake Occurrences.
  \emph{Journal of the American Statistical Association},
  \bold{97}(458), 369--380.
  \href{http://dx.doi.org/10.1198/016214502760046925}{doi:10.1198/016214502760046925}.

  Zhuang J, Ogata Y, Vere-Jones D (2006).
  Diagnostic Analysis of Space-Time Branching Processes for Earthquakes.
  In \emph{Case Studies in Spatial Point Process Modeling},
  pp. 275--292. Springer Nature.
  \href{http://dx.doi.org/10.1007/0-387-31144-0_15}{doi:10.1007/0-387-31144-0_15}.

  Zhuang J (2011).
  Next-day Earthquake Forecasts for the Japan Region Generated by
  the ETAS Model.
  \emph{Earth, Planets and Space},
  \bold{63}(3), 207--216.
  \href{http://dx.doi.org/10.5047/eps.2010.12.010}{doi:10.5047/eps.2010.12.010}.
}

\seealso{
  \code{\link{etas}}
}
\examples{

  # preparing the catalog
  iran.cat <- catalog(iran.quakes, time.begin="1973/01/01",
     study.start="1996/01/01", study.end="2016/01/01",
     lat.range=c(25, 42), long.range=c(42, 63), mag.threshold=4.5)

  print(iran.cat)
  \dontrun{
  plot(iran.cat)}

  # initial parameters values
  param01 <- c(0.46, 0.23, 0.022, 2.8, 1.12, 0.012, 2.4, 0.35)

  # fitting the model and
  \dontrun{
  iran.fit <- etas(iran.cat, param0=param01)}

  # estimating the declustering probailities
  \dontrun{
  pr <- probs(iran.fit)
  plot(iran.cat$longlat.coord[,1:2], cex=2 * (1 - pr$prob))}

  # estimating the  background seismicity rate and clustering coefficient
  \dontrun{
  rates(iran.fit, dimyx=c(100, 125))
  iran.rates <- rates(iran.fit, dimyx=c(200, 250), plot.it=FALSE)
  summary(iran.rates$background)}
}
\author{Abdollah Jalilian
  \email{jalilian@razi.ac.ir}
}
\keyword{spatial}
\keyword{math}
\keyword{earthquake modeling}
