% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/UserFunctions.R
\name{setEase}
\alias{setEase}
\title{Setting the parameters for simulating a model}
\usage{
setEase(
  N,
  threshold,
  dioecy,
  mutMatrixObj,
  genomeObj,
  selectionObj,
  stopCondition = list(),
  initGenoFreq = matrix(),
  selfRate = NA_real_
)
}
\arguments{
\item{N}{the population size}

\item{threshold}{the maximum number of generations}

\item{dioecy}{logical indicating whether the simulated population is
dioecious or hermaphroditic}

\item{mutMatrixObj}{a \code{MutationMatrix} object}

\item{genomeObj}{a \code{Genome} object}

\item{selectionObj}{a \code{Selection} object}

\item{stopCondition}{list of vectors that each describe the alleles that
must be fixed to define a stop condition. Each of these stop conditions
will therefore be associated with a stop condition}

\item{initGenoFreq}{A vector of the size of the genotype number
describing the initial allele frequencies common to all simulations}

\item{selfRate}{the selfing rate}
}
\value{
an \code{Ease} object
}
\description{
Last step before the simulation of the model, the creation of a \code{Ease}
class object makes it possible to gather all the ingredients for a complete
parameterization of a model and then to simulate it.
}
\details{
The \code{Ease} class is used to manage the simulations by handling the
objects needed to build the model. Thus to build an object of class
\code{Ease}, it is necessary to have defined an object \code{Genome},
as well as an object \code{MutationMatrix} and an object \code{Selection}
(even if it is neutral, see \link[Ease]{setSelectNeutral}).

Once simulated with the method \code{simulate}, an Ease object contains
the results of the simulations and the records of these last ones if the
parameter \code{recording} of \code{simulate} was fixed at \code{TRUE}.
To obtain these results and records, it is necessary to use the functions
\link[Ease]{getResults} and \link[Ease]{getRecords}.
}
\examples{

library(tidyr)
library(ggplot2)

### Simple example of a succession of allele replacements to each other
# in a deterministic way and simply by mutations (selection is neutral) ###

# Let's put a single diploid locus with 5 alleles:
LD <- list(dl1 = as.factor(c("a1", "a2", "a3", "a4", "a5", "a6")))
HL <- list(hl = as.factor("noHL"))
genomeObj <- setGenome(listHapLoci = HL, listDipLoci = LD)

# The only possible mutations are mutations from the a1 to a2 allele,
# from a2 to a3, etc.:
mutMatrixObj <- setMutationMatrixByRates(
  genomeObj = genomeObj,
  forwardMut = 1e-2
)
# The matrix thus constructed looks like this:
mutMatrixObj

# The selection is neutral:
selectionObjNeutral <- setSelectNeutral(genomeObj = genomeObj)

# We can thus define an Ease object with a population size of 100
# sex-separated individuals and a threshold of 700 generations:
mod <- setEase(
  N = 100, threshold = 700, dioecy = TRUE,
  mutMatrixObj = mutMatrixObj,
  genomeObj = genomeObj,
  selectionObj = selectionObjNeutral
)

# For the simulation we shut down the drift to have a deterministic
# evolution of allelic frequencies:
mod <- simulate(mod, recording = TRUE, verbose = TRUE, drift = FALSE)

# We recover the \code{data.frame} of the simulation record and we modify
# a little the organization of the data:
records <- getRecords(mod)[[1]]
records <- gather(records, "allele", "freqAllele", 45:50)

# Then we display the evolution of the allelic frequencies of each of the
# alleles of the locus:
ggplot(records, aes(x = gen, y = freqAllele, color = allele)) +
  geom_line() +
  ylim(0, 1)


### Example of simulation of cyto-nuclear Bateson-Dobzhansky-Muller
# incompatibilities (BDMI) ###

# Two loci: a haploid locus and a diploid locus. Each has two alleles,
# an ancestral allele in upper case and a derived allele in lower case:
LD <- list(dl = as.factor(c("A", "a")))
HL <- list(hl = as.factor(c("B", "b")))
genomeObj <- setGenome(listHapLoci = HL, listDipLoci = LD)

# The mutation rate to derived alleles is set at 1e-2:
mutMatrixObj <- setMutationMatrixByRates(genomeObj = genomeObj, forwardMut = 1e-2)

# The two derived alleles a and b are incompatible and therefore impose
# a fitness cost on their carrier:
selectionObj <- setSelectOnInds(
  genomeObj = genomeObj,
  indFit = c(1, 1, 1, 1, 0.5, 0)
)

# We can then define the Ease object by specifying the population size (100),
# the maximum generation threshold (1e6), that we want the individuals to be
# hermaphroditic (dioecy = FALSE) and that they reproduce at 50\% by
# selfing. The simulation stops if one of the two derived alleles is fixed:
mod <- setEase(
  N = 100, threshold = 1e6, dioecy = FALSE, selfRate = 0.5,
  stopCondition = list("a", "b"),
  mutMatrixObj = mutMatrixObj,
  genomeObj = genomeObj,
  selectionObj = selectionObj
)

# The model is simulated:
mod <- simulate(mod, nsim = 10, verbose = TRUE)

# And the results plotted:
plot(mod)

}
\author{
Ehouarn Le Faou
}
