% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elo.seq.R
\name{elo.seq}
\alias{elo.seq}
\alias{fastelo}
\title{calculate Elo ratings}
\usage{
elo.seq(winner, loser, Date, draw = NULL, presence = NULL, startvalue = 1000,
               k = 100, normprob = TRUE, init = "average", intensity = NULL,
               iterate = 0, runcheck = TRUE, progressbar = FALSE)
fastelo(WINNER, LOSER, ALLIDS, KVALS, STARTVALUES, NORMPROB = TRUE, ROUND = TRUE)
}
\arguments{
\item{winner}{either a factor or character vector with winner IDs of dyadic dominance interactions}

\item{loser}{either a factor or character vector with loser IDs of dyadic dominance interactions}

\item{Date}{character vector of form "YYYY-MM-DD" with the date of the respective interaction}

\item{draw}{logical, which interactions ended undecided (i.e. drawn or tied)? By default all \code{FALSE}, i.e. no undecided interactions occurred. Note that in this case, \code{winner}/\code{loser} values can be interchanged}

\item{presence}{optional data.frame, to supply data about presence and absence of individuals for part of the time the data collection covered. see details}

\item{startvalue}{the value of Elo ratings of the two individuals that are involved in the first interaction of the overall sequence prior to this interaction. By default set to 1000. See also \code{init}}

\item{k}{factor \emph{k} that determines the maximum change in ratings. By default \code{k=100}}

\item{normprob}{logical (by default \code{TRUE}). Should a normal curve be assumed for calculating the winning/losing probablities, or a logistic curve. See \code{\link{winprob}} for details}

\item{init}{character, what Elo rating does an individual have prior to its first interaction. Three options are available:
\code{average}: individuals always start with the value specified in \code{startvalue}. Given stable composition of the group, this also reflects the average Elo rating on each day in that group, \cr
\code{bottom}: subjects entering at the current lowest Elo value if the lowest value getting lower its getting lower for all subjects which had this lowest values before, it is reflecting that in some species new subjects entering a group at the bottom level "bottom entry"\cr
\code{bottom_low}: same as \code{bottom} but additionally the start values getting after the first interaction lower for all non-interacting subjects and, reflecting that we have at start no knowledge about the subjects this option offers for "bottom entry" species the possibility to consider that in a way that those subjects which are not interacting getting lower from start on}

\item{intensity}{a character vector or factor describing intensity of interaction, to be matched with custom k values if specified}

\item{iterate}{not yet implemented}

\item{runcheck}{logical, should several checks regarding data integrety be performed, by default \code{TRUE}. See \code{\link{seqcheck}}}

\item{progressbar}{logical, should progress bars be displayed, by default \code{progressbar=TRUE}}

\item{WINNER}{same as \code{winner} for use in \code{fastelo()}}

\item{LOSER}{same as \code{loser} for use in \code{fastelo()}}

\item{ALLIDS}{character vector, contains all the indivuals IDS}

\item{KVALS}{numeric vector of the same length \code{WINNER}, i.e. one k value for each interaction}

\item{STARTVALUES}{numeric vector of the same length as \code{ALLIDS}, i.e. one start value for each individual}

\item{NORMPROB}{logical, by default \code{TRUE}: same as \code{normprob} for use in \code{fastelo()}}

\item{ROUND}{logical, by default \code{TRUE}: should ratings be rounded to integers. For use in \code{fastelo()}}
}
\value{
An object of class \code{elo}, which is list with 10 items that serves as basis to extract relevant information:
\item{mat}{a date by ID-\code{matrix} with raw Elo ratings}
\item{lmat}{a date by ID-\code{matrix} with raw Elo ratings}
\item{cmat}{a date by ID-\code{matrix} with raw Elo ratings}
\item{pmat}{a date by ID-\code{matrix} with with presence data}
\item{nmat}{a date by ID-\code{matrix} containing the number of interactions a given ID was involved in on a given day}
\item{logtable}{details on each single interaction}
\item{stability}{a \code{data.frame} containing information about stability (see \code{\link{stab_elo}}) }
\item{truedates}{vector of class \code{Date} covering the ranges of dates in the dataset}
\item{misc}{various}
\item{allids}{a (sorted) character vector with all IDs that occur in the dataset}

\code{fastelo()} returns a list with ten items:
\item{\code{$ratings}}{numeric vector of the final ratings in the same order as \code{ALLIDS}}
\item{\code{$winprobs}}{numeric vector with winning probabilities in the same order as the interactions were supplied}
\item{\code{$rtype}}{character of length 1, as a marker that the result comes from \code{fastelo()}}
\item{\code{$startvalues}}{numeric vector with start values}
\item{\code{$kvalues}}{numeric vector with k values}
\item{\code{$winner}}{character vector with winners}
\item{\code{$loser}}{character vector with losers}
\item{\code{$allids}}{character vector with all IDs that occur in the sequence}
\item{\code{$normprob}}{logical, was normal probability used for winning expectations}
\item{\code{$round}}{logical, was rounding to integers used during the calculation of ratings}
}
\description{
calculate Elo ratings from a sequence of dominance interactions
}
\details{
The presence 'matrix' is actually an object of class \code{data.frame} containing information about wether an individual was present on a given day or not. The first column represents the dates, running at least from the date of the earliest interaction until at least the date of the last interaction with one line per day (regardless of whether there were actually interactions observed on each day). Further, each individual is represented as a column in which "1" indicates an individual was present on the row-date and a "0" indicates the individuals absence on this date. \code{NA}s are not allowed. See \code{\link{advpres}} for an example.

The function \code{fastelo()} is a stripped-down version of \code{elo.seq()}, which performs only the most basic calculations while ignoring anything that is date and presence related. Neither does it perform data checks. In other words, it just calculates ratings based on the sequence. It's most useful in simulations, for example when estimating optimal k parameters. Its main advantage is its speed, which is substantially faster than \code{elo.seq()}. Note that currently there is no support for tied interactions. The main difference to note is that both, start values and k values have to be supplied as vectors with one value for each individual and interaction respectively.
}
\examples{
data(adv)
res <- elo.seq(winner = adv$winner, loser = adv$loser, Date = adv$Date)
summary(res)

# with custom k
data(adv2)
table(adv2$intensity)

myks <- list(displace = 20, fight = 200)
res <- elo.seq(winner = adv2$winner, loser = adv2$loser, Date = adv2$Date,
               k = myks, intensity = adv2$intensity)
extract_elo(res)
summary(res)

# with custom start values
# if we know prior ranks:
myranks <- 1:7
names(myranks) <- letters[1:7]
mypriors <- createstartvalues(myranks, shape = 0.3)
res <- elo.seq(winner = adv2$winner, loser = adv2$loser, Date = adv2$Date,
               k = myks, intensity = adv2$intensity, startvalue = mypriors$res)
extract_elo(res)

# compare elo.seq and fastelo
xdata <- randomsequence(10, 500)
allids <- colnames(xdata$pres)[2:ncol(xdata$pres)]
winner <- xdata$seqdat$winner
loser <- xdata$seqdat$loser
Date <- xdata$seqdat$Date
k <- rep(100, length(winner))
svals <- rep(1000, length(allids))

res1 <- fastelo(WINNER = winner, LOSER = loser, ALLIDS = allids, KVALS = k,
                STARTVALUES = svals, NORMPROB = TRUE)$ratings
names(res1) <- allids
res1 <- sort(res1, decreasing = TRUE)
res2 <- extract_elo(elo.seq(winner = winner, loser = loser, Date = Date,
                            startvalue = 1000, k = 100, normprob = TRUE,
                            runcheck = FALSE))
res1
res2
}
\references{
\insertRef{elo1978}{EloRating}

\insertRef{albers2001}{EloRating}

\insertRef{neumann2011}{EloRating}

\insertRef{newton-fisher2017a}{EloRating}
}
\author{
Christof Neumann and Lars Kulik
}
