# creation of input server.R file for UI of the EmiStatR model
# author: Arturo Torres, Kai Klepiszewski
# organization: LIST
# date1: 06.02.2015 - 16.02.2016
# date2: 05.09.2017 - 07.09.2017

# 1.    wastewater
# qs  : individual water consumption of households [l/(PE d)]
# CODs: sewage pollution - COD concentration [g/(PE d)]
# NH4s: sewage pollution - NH4 concentration [g/(PE d)]
# 
# 2.    infiltration water
# qf  : infiltration water inflow [l/(s ha)]
# CODf: infiltration water pollution - COD concentration [g/(PE d)]
# NH4f: infiltration water pollution - NH4 concentration [g/(PE d)]
# 
# 3.    rainwater
# CODr: rainwater pollution - COD concentration [mg/l]
# NH4r: rainwater pollution - NH4 concentration [mg/l]
# stat: name of the rain measurement station
# peri: period of analysis of rainfall 
# dura: duration of the period of analysis of rainfall [year]
# depth: mean height of precipitation [mm]
# pDur: mean duration of the rain [min]
# iMin: minimum intensity [mm/(delta.t min)]
# delta.t  : rain time interval [min]
# 
# 
# P1: data.frame containing columns tt (date and time), P (rain time series), and i (intensity)

#===================================================================================================
library(shiny)
library(shinyFiles) # shinyDirChoose()

# Define server logic 
shinyServer(function(input, output, session) {
  volumes <<- getVolumes()
  # Reactive expression to compose a data frame containing all of
  # the values
  #-----------------------------------------------------------------------------------------
  sliderValores1 <- reactive({
    # Compose data frame
    data.frame(
      Variable = c("Water consumption, qs [l/(PE d)]", 
               "Pollution COD [g/(PE d)]",
               "Pollution NH4 [g/(PE d)]"),
      Value = as.character(c(input$qs, 
                             input$CODs,
                             input$NH4s)), 
      stringsAsFactors=FALSE)   
  }) 
  
  #-----------------------------------------------------------------------------------------
  sliderValores2 <- reactive({
    # Compose data frame
    data.frame(
      Variable = c("Inflow, qf [l/(s ha)]:",
               "Pollution COD [g/(PE d)]:",
               "Pollution NH4 [g/(PE d)]:"),
      Value = as.character(c( 
        input$qf,
        input$CODf,
        input$NH4f)), 
      stringsAsFactors=FALSE)
  }) 
  
  #-----------------------------------------------------------------------------------------
  sliderValores3 <- reactive({
    #     # duration of the period calculation
    #     dura = input$peri[2]-input$peri[1]+1                              #  <--------------------
    
    # save variables
    CODr  <<- isolate(input$CODr)
    NH4r  <<- isolate(input$NH4r)
    stat  <<- isolate(input$stat)
    # peri  <- isolate(input$peri)                #  <--------------------
    
    
    # definition of stat
    if(input$radio.data == ""){
      stat <<- ""
    }
    
    if(input$radio.data == "1"){
      stat <<- "Dahl"
    }
    
    if(input$radio.data == "2"){
      stat <<- "Esch-sur-Sure"
    }
    
    if(input$radio.data == "3"){
      if (is.null(input$file1)){
        stat <<- ""
      }else stat <<- input$stat
    }
    
    # Compose data frame    
    data.frame(
      Variable = c("Pollution COD [mg/l]:",
               "Pollution NH4 [mg/l]:",
               "Rain measurement station:" #,    #  <--------------------
               #"Period:",    #  <--------------------
               #"Duration of the period [year]:"   #  <--------------------
      ),
      Value = as.character(c( 
        input$CODr,
        input$NH4r,
        stat #,
        #paste(input$peri, collapse=' - '),   #  <--------------------
        #dura   #  <--------------------
      )), 
      stringsAsFactors=FALSE)
  }) 
  #-----------------------------------------------------------------------------------------
  output$contents <- renderTable({
    
    # input$file1 will be NULL initially. After the user selects
    # and uploads a file, it will be a data frame with 'name',
    # 'size', 'type', and 'datapath' columns. The 'datapath'
    # column will contain the local filenames where the data can
    # be found.
    
    inFile <- input$file1
    
    # if (is.null(inFile))
    #   return(NULL)
    
    if(input$radio.data == "3"){
      read.csv(inFile$datapath, header=TRUE)  
    }
    
  })
  #-----------------------------------------------------------------------------------------
  readValores4 <- reactive({
    inFile <- input$file1
    
    if(input$radio.data == ""){
      output$textInput <- renderUI({
        NULL
      })
      output$fileInput <- renderUI({
        NULL
      })
      output$actionButton.save <- renderUI({
        NULL
      }) 
    }
    
    if(input$radio.data == "1"){
      data(P1)
      data <- P1
      
      output$textInput <- renderUI({
        NULL
      })
      
      output$fileInput <- renderUI({
        NULL
      })
    }
    
    if(input$radio.data == "2"){
      data(Esch_Sure2010)
      data <- Esch_Sure2010
      
      output$textInput <- renderUI({
        NULL
      })
      
      output$fileInput <- renderUI({
        NULL
      })
    }
    
    if(input$radio.data == "3"){
      if (is.null(inFile)){
        output$fileInput <- renderUI({
          fileInput('file1', 'Choose rain time series as CSV File...',
                    accept=c('text/csv', 
                             'text/comma-separated-values,text/plain', 
                             '.csv'))
        })
        
        output$textInput <- renderUI({
          textInput("stat", label = h5("Rain measurement station:"), 
                    value = "Enter name station...")
        }) 
        
        return(NULL)
      }
      
      output$fileInput <- renderUI({
        fileInput('file1', 'Choose rain time series as CSV File...',
                  accept=c('text/csv', 
                           'text/comma-separated-values,text/plain', 
                           '.csv'))
      })
      
      output$textInput <- renderUI({
        textInput("stat", label = h5("Rain measurement station:"), 
                  value = "Enter name station...")
      }) 
      
      data <- read.csv(inFile$datapath, header = TRUE)
    }
    
    
    if(input$radio.data != "" & length(parseDirPath(volumes, input$directory)) > 0){
      output$actionButton.save <- renderUI({
        actionButton("save", label = "Save & close")
      }) 
      
      
      ## convert date-time to POSIXct POSIXt
      # tt <- as.data.frame(strptime(data[,1], format='%d/%m/%Y %H:%M')) # original time series
      tt  <- strptime(data[,1], format='%Y-%m-%d %H:%M:%S') # final time series
      
      # calculate total height of precipitation
      a  <- as.data.frame(data)
      depth <<- sum(a[,2])
      
      # calculate time step
      dtt <- difftime(tt[2], tt[1], units="min")
      delta.t  <<- as.numeric(dtt)
      
      # calculate intensity
      # f2 <- function(x) ifelse(x==0,1,x)
      # f2 <- function(x) {x/delta.t}                  # <-------------------
      # i  <- as.data.frame(sapply(a[,2], f2))   # <-------------------
      # colnames(i) <- c("i")    # <-------------------
      # ii <- summary(i)         # <-------------------
      
      # calculate duration of the rain
      f1 <- function(x) ifelse(x>0,1,0)
      c <- sapply(a[,2], f1)                         # <=================
      pDur <<- sum(c)*delta.t                              # <=================   
      
      # # # save variables
      # # setwd(nd_tmp)
      # # load("rainwater.RData")
      # rw <<- list(depth, pDur, CODr, NH4r,                         # <-------------------
      #             stat, delta.t)     #  <--------------------
      # rw <<- setNames(rw, c("depth", "pDur", "CODr", "NH4r",            # <-------------------
      #                       "stat", "delta.t"))    #  <--------------------
      # # save(rw, file="rainwater.RData")
      # # setwd(cd)
      
      # Compose data frame
      data.frame(
        Variable = c("Mean height [mm]:",
                 "Mean duration [min]:",
                 # "Minimum intensity [mm/min]",     # <-------------------
                 # "Mean intensity [mm/min]",        # <-------------------
                 # "Maximum intensity [mm/min]",     # <-------------------
                 "Time interval [min]"),
        Value = as.character(c( 
          depth,
          pDur, 
          #ii[1],                # <-------------------
          #ii[4],                # <-------------------
          #ii[6],                # <-------------------
          delta.t
        )), 
        stringsAsFactors=FALSE)
    }
  })
  #-----------------------------------------------------------------------------------------
  output$plot1 <- renderPlot({
    inFile <- input$file1
 
    if(input$radio.data == "1"){
      data(P1)
      data <- P1
    }
    
    if(input$radio.data == "2"){
      data(Esch_Sure2010)
      data <- Esch_Sure2010
    }
    
    if(input$radio.data == "3"){
      if (is.null(inFile))
        return(NULL)
      
      data <- read.csv(inFile$datapath, header = TRUE)
    }
    
    if(input$radio.data != ""){
      tt <- strptime(data[,1], format='%Y-%m-%d %H:%M:%S') # final time series
      
      # precipitation data
      data0 <- as.data.frame(tt)
      data0 <- cbind.data.frame(tt, as.data.frame(data[,2]))

      P1 <<- cbind.data.frame(data0) # include intensity  # <------------------ 
      colnames(P1) <- c("Time [y-m-d h:m:s]", "P [mm]")

      # creating plot
      daterange <- c(min(tt), max(tt))
      plot(tt, data[,2], type ="l", xaxt="n", col = "blue")
      axis.POSIXct(1, at=seq(daterange[1], daterange[2], by="day"), 
                   format="%b-%Y") #label the x axis by months
    }
  })
  #-----------------------------------------------------------------------------------------
  # Show the values using an HTML table
  output$valores1 <- renderTable({
    sliderValores1()
  })
  #-----------------------------------------------------------------------------------------
  output$valores2 <- renderTable({
    sliderValores2()
  })
  #-----------------------------------------------------------------------------------------
  output$valores3 <- renderTable({
    sliderValores3()
  })
  #-----------------------------------------------------------------------------------------
  output$valores4 <- renderTable({
    readValores4()
  })
  #-----------------------------------------------------------------------------------------
  observe({
    # dir
    # if(length(parseDirPath(volumes, input$directory)) > 0){
    # volumes <<- getVolumes()}
    volumes <<- getVolumes()
    
    shinyDirChoose(input, 'directory', roots=volumes)
    directory <- reactive(input$directory)
    output$directory <- renderPrint(directory())
    
    # path
    path <- reactive({
      home <- normalizePath("~")
      file.path(home, paste(unlist(dir()$path[-1]), collapse = .Platform$file.sep))
    })
    
    # files
    output$files <- renderPrint(list.files(path()))
    
    if(length(parseDirPath(volumes, input$directory)) < 1){
      showNotification("Please, choose directory.")
    }else{
      output$actionButton.save <- renderUI({
        actionButton("save", label = "Save & close")
      }) 
    }
  })
  #-----------------------------------------------------------------------------------------
  output$valores5 <- renderPrint(
    {parseDirPath(volumes, input$directory)}
    )
  #-----------------------------------------------------------------------------------------
  # take an action, save, when button is clicked
  observe({
    
    if(input$radio.data == ""){
      showNotification("Please, select precipitation data.")
    } 
    
    # if(length(parseDirPath(volumes, input$directory) < 1)){
    #   showNotification("Please, choose directory.")
    # }
    
    ifelse (input$save == 0,
            return(),
            {qs   <- isolate(input$qs)
            CODs <- isolate(input$CODs)
            NH4s <- isolate(input$NH4s)
            
            qf   <- isolate(input$qf)
            CODf <- isolate(input$CODf)
            NH4f <- isolate(input$NH4f)
            
            # loading variables for saving
            # setwd(nd_tmp)
            # load("rainwater.RData")
            # load("P1.RData")
            # #file.remove(list.files()) # deleting all temporal files
            
            # saving variables
            # setwd(nd)
            
            cdir <<- parseDirPath(volumes, input$directory)
            setwd(cdir)
            
            ww <- list(qs=qs,CODs=CODs,NH4s=NH4s)
            save(ww,file=paste("wastewater.RData", sep=""))
            
            inf <- list(qf=qf,CODf=CODf,NH4f=NH4f)
            save(inf,file=paste("infiltration.RData", sep=""))
            
            rw <- list(depth=depth, pDur=pDur, CODr=CODr, NH4r=NH4r, stat=stat, delta.t=delta.t)
            save(rw,file=paste("rainwater.RData", sep=""))
            
            save(P1, file="P1.RData")
            write.csv(P1, file = "P1.csv")
            
            shiny::stopApp
            stopApp(returnValue = "closed and saved")
            
            setwd(cdir)
            })
    
  })
  #-----------------------------------------------------------------------------------------
  # take an action, close, when button is clicked
  observe({
    ifelse (input$close == 0,
            return(),
            {shiny::stopApp
              stopApp(returnValue = "closed and not saved")
            }
            
    )
  })
  #-----------------------------------------------------------------------------------------
})