% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/net.utils.R
\name{dissolution_coefs}
\alias{dissolution_coefs}
\title{Dissolution Coefficients for Stochastic Network Models}
\usage{
dissolution_coefs(dissolution, duration, d.rate = 0)
}
\arguments{
\item{dissolution}{Right-hand sided STERGM dissolution formula
(see \code{\link{netest}}). See below for list of supported
dissolution models.}

\item{duration}{A vector of mean edge durations in arbitrary time units.}

\item{d.rate}{Departure or exit rate from the population, as a single
homogeneous rate that applies to the entire population.}
}
\value{
A list of class \code{disscoef} with the following elements:
\itemize{
 \item \strong{dissolution:} right-hand sided STERGM dissolution formula
        passed in the function call.
 \item \strong{duration:} mean edge durations passed into the function.
 \item \strong{coef.crude:} mean durations transformed into logit
       coefficients.
 \item \strong{coef.adj:} crude coefficients adjusted for the risk of
       departure on edge persistence, if the \code{d.rate} argument is
       supplied.
 \item \strong{coef.form.corr:} corrections to be subtracted from formation
       coefficients.
 \item \strong{d.rate:} the departure rate.
 \item \strong{diss.model.type:} the form of the dissolution model; options
       include \code{edgesonly}, \code{nodematch}, \code{nodemix}, and
       \code{nodefactor}.
}
}
\description{
Calculates dissolution coefficients, given a dissolution model
             and average edge duration, to pass as offsets to an ERGM/TERGM
             model fit in \code{netest}.
}
\details{
This function performs two calculations for dissolution coefficients
used in a network model estimated with \code{\link{netest}}:
\enumerate{
 \item \strong{Transformation:} the mean durations of edges in a network are
       mathematically transformed to logit coefficients.
 \item \strong{Adjustment:} in a dynamic network simulation in an open
       population (in which there are departures), it is further necessary to
       adjust these coefficients; this upward adjustment accounts for
       departure as a competing risk to edge dissolution.
}

The current dissolution models supported by this function and in network
model estimation in \code{\link{netest}} are as follows:
\itemize{
 \item \code{~offset(edges)}: a homogeneous dissolution model in which the
        edge duration is the same for all partnerships. This requires
        specifying one duration value.
 \item \code{~offset(edges) + offset(nodematch("<attr>"))}: a heterogeneous
        model in which the edge duration varies by whether the nodes in the
        dyad have similar values of a specified attribute. The duration
        vector should now contain two values: the first is the mean edge
        duration of non-matched dyads, and the second is the duration of the
        matched dyads.
 \item \code{~offset(edges) + offset(nodemix("<attr>"))}: a heterogeneous
        model that extends the nodematch model to include non-binary
        attributes for homophily. The duration vector should first contain
        the base value, then the values for every other possible combination
        in the term.
 \item \code{~offset(edges) + offset(nodefactor("<attr>"))}: a heterogeneous
        model in which the edge duration varies by a specified attribute. The
        duration vector should first contain the base value, then the values
        for every other value of that attribute in the term. This option is
        deprecated.
}
}
\examples{
## Homogeneous dissolution model with no departures
dissolution_coefs(dissolution = ~offset(edges), duration = 25)

## Homogeneous dissolution model with departures
dissolution_coefs(dissolution = ~offset(edges), duration = 25,
                  d.rate = 0.001)

## Heterogeneous dissolution model in which same-race edges have
## shorter duration compared to mixed-race edges, with no departures
dissolution_coefs(dissolution = ~offset(edges) + offset(nodematch("race")),
                  duration = c(20, 10))

## Heterogeneous dissolution model in which same-race edges have
## shorter duration compared to mixed-race edges, with departures
dissolution_coefs(dissolution = ~offset(edges) + offset(nodematch("race")),
                  duration = c(20, 10), d.rate = 0.001)

\dontrun{
## Extended example for differential homophily by age group
# Set up the network with nodes categorized into 5 age groups
nw <- network_initialize(n = 1000)
age.grp <- sample(1:5, 1000, TRUE)
nw <- set_vertex_attribute(nw, "age.grp", age.grp)

# durations = non-matched, age.grp1 & age.grp1, age.grp2 & age.grp2, ...
# TERGM will include differential homophily by age group with nodematch term
# Target stats for the formation model are overall edges, and then the number
# matched within age.grp 1, age.grp 2, ..., age.grp 5
form <- ~edges + nodematch("age.grp", diff = TRUE)
target.stats <- c(450, 100, 125, 40, 80, 100)

# Target stats for the dissolution model are duration of non-matched edges,
# then duration of edges matched within age.grp 1, age.grp 2, ..., age.grp 5
durs <- c(60, 30, 80, 100, 125, 160)
diss <- dissolution_coefs(~offset(edges) +
                            offset(nodematch("age.grp", diff = TRUE)),
                          duration = durs)

# Fit the TERGM
fit <- netest(nw, form, target.stats, diss)

# Full diagnostics to evaluate model fit
dx <- netdx(fit, nsims = 10, ncores = 4, nsteps = 300)
print(dx)

# Simulate one long time series to examine timed edgelist
dx <- netdx(fit, nsims = 1, nsteps = 5000, keep.tedgelist = TRUE)

# Extract timed-edgelist
te <- as.data.frame(dx)
head(te)

# Limit to non-censored edges
te <- te[which(te$onset.censored == FALSE & te$terminus.censored == FALSE),
         c("head", "tail", "duration")]
head(te)

# Look up the age group of head and tail nodes
te$ag.head <- age.grp[te$head]
te$ag.tail <- age.grp[te$tail]
head(te)

# Recover average edge durations for age-group pairing
mean(te$duration[te$ag.head != te$ag.tail])
mean(te$duration[te$ag.head == 1 & te$ag.tail == 1])
mean(te$duration[te$ag.head == 2 & te$ag.tail == 2])
mean(te$duration[te$ag.head == 3 & te$ag.tail == 3])
mean(te$duration[te$ag.head == 4 & te$ag.tail == 4])
mean(te$duration[te$ag.head == 5 & te$ag.tail == 5])
durs
}

}
\keyword{netUtils}
