% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FRK_wrapper.R, R/SRE.R, R/SREfit.R,
%   R/SREpredict.R, R/SREutils.R
\name{FRK}
\alias{FRK}
\alias{SRE}
\alias{SRE.fit}
\alias{predict,SRE-method}
\alias{coef,SRE-method}
\title{Construct SRE object, fit and predict}
\usage{
FRK(
  f,
  data,
  basis = NULL,
  BAUs = NULL,
  est_error = TRUE,
  average_in_BAU = TRUE,
  sum_variables = NULL,
  normalise_wts = TRUE,
  fs_model = "ind",
  vgm_model = NULL,
  K_type = c("block-exponential", "precision", "unstructured"),
  n_EM = 100,
  tol = 0.01,
  method = c("EM", "TMB"),
  lambda = 0,
  print_lik = FALSE,
  response = c("gaussian", "poisson", "gamma", "inverse-gaussian", "negative-binomial",
    "binomial"),
  link = c("identity", "log", "square-root", "logit", "probit", "cloglog", "inverse",
    "inverse-squared"),
  optimiser = nlminb,
  fs_by_spatial_BAU = FALSE,
  known_sigma2fs = NULL,
  taper = NULL,
  ...
)

SRE(
  f,
  data,
  basis,
  BAUs,
  est_error = TRUE,
  average_in_BAU = TRUE,
  sum_variables = NULL,
  normalise_wts = TRUE,
  fs_model = "ind",
  vgm_model = NULL,
  K_type = c("block-exponential", "precision", "unstructured"),
  normalise_basis = TRUE,
  response = c("gaussian", "poisson", "gamma", "inverse-gaussian", "negative-binomial",
    "binomial"),
  link = c("identity", "log", "square-root", "logit", "probit", "cloglog", "inverse",
    "inverse-squared"),
  include_fs = TRUE,
  fs_by_spatial_BAU = FALSE,
  ...
)

SRE.fit(
  object,
  n_EM = 100L,
  tol = 0.01,
  method = c("EM", "TMB"),
  lambda = 0,
  print_lik = FALSE,
  optimiser = nlminb,
  known_sigma2fs = NULL,
  taper = NULL,
  ...
)

\S4method{predict}{SRE}(
  object,
  newdata = NULL,
  obs_fs = FALSE,
  pred_time = NULL,
  covariances = FALSE,
  n_MC = 400,
  type = "mean",
  k = NULL,
  percentiles = c(5, 95),
  kriging = "simple"
)

\S4method{coef}{SRE}(object, ...)
}
\arguments{
\item{f}{\code{R} formula relating the dependent variable (or transformations thereof) to covariates}

\item{data}{list of objects of class \code{SpatialPointsDataFrame}, \code{SpatialPolygonsDataFrame}, \code{STIDF}, or  \code{STFDF}. If using space-time objects, the data frame must have another field, \code{t}, containing the time index of the data point. If the assumed response distribution is "binomial" or "negative-binomial", the data frame must have another field, \code{k}, containing the known constant parameter \eqn{k} for each observation.}

\item{basis}{object of class \code{Basis} (or \code{TensorP_Basis})}

\item{BAUs}{object of class \code{SpatialPolygonsDataFrame}, \code{SpatialPixelsDataFrame}, \code{STIDF}, or \code{STFDF}. The object's data frame must contain covariate information as well as a field \code{fs} describing the fine-scale variation up to a constant of proportionality. If the function \code{FRK()} is used directly, then BAUs are created automatically, but only coordinates can then be used as covariates}

\item{est_error}{(applicable only if \code{response} = "gaussian") flag indicating whether the measurement-error variance should be estimated from variogram techniques. If this is set to 0, then \code{data} must contain a field \code{std}. Measurement-error estimation is currently not implemented for spatio-temporal datasets}

\item{average_in_BAU}{if \code{TRUE}, then multiple data points falling in the same BAU are averaged; the measurement error of the averaged data point is taken as the average of the individual measurement errors}

\item{sum_variables}{if \code{average_in_BAU == TRUE}, the string \code{sum_variables} indicates which data variables (can be observations or covariates) are to be summed rather than averaged}

\item{normalise_wts}{if \code{TRUE}, the rows of the incidence matrices \eqn{C_Z} and \eqn{C_P} are normalised to sum to 1, so that the mapping represents a weighted average; if false, no normalisation of the weights occurs (i.e., the mapping corresponds to a weighted sum)}

\item{fs_model}{if "ind" then the fine-scale variation is independent at the BAU level. Only the independent model is allowed for now, future implementation will include CAR/ICAR (in development)}

\item{vgm_model}{(applicable only if \code{response} = "gaussian") an object of class \code{variogramModel} from the package \code{gstat} constructed using the function \code{vgm}. This object contains the variogram model that will be fit to the data. The nugget is taken as the measurement error when \code{est_error = TRUE}. If unspecified, the variogram used is \code{gstat::vgm(1, "Lin", d, 1)}, where \code{d} is approximately one third of the maximum distance between any two data points}

\item{K_type}{the parameterisation used for the basis-function covariance matrix, \code{K}. If \code{method} = "EM", \code{K_type} can be "unstructured" or "block-exponential". If \code{method} = "TMB", \code{K_type} can be "precision" or "block-exponential". The default is "block-exponential", however if \code{FRK()} is used and \code{method} = "TMB", for computational reasons \code{K_type} is set to "precision"}

\item{n_EM}{(applicable only if \code{method} = "EM") maximum number of iterations for the EM algorithm}

\item{tol}{(applicable only if \code{method} = "EM") convergence tolerance for the EM algorithm}

\item{method}{parameter estimation method to employ. Currently "EM" and "TMB" are supported}

\item{lambda}{(applicable only if \code{K_type} = "unstructured") ridge-regression regularisation parameter (0 by default). Can be a single number, or a vector (one parameter for each resolution)}

\item{print_lik}{(applicable only if \code{method} = "EM") flag indicating whether to plot log-likelihood vs. iteration after convergence of the EM estimation algorithm}

\item{response}{string indicating the assumed distribution of the response variable. It can be "gaussian", "poisson", "negative-binomial", "binomial", "gamma", or "inverse-gaussian". If \code{method} = "EM", only "gaussian" can be used}

\item{link}{string indicating the desired link function. Can be "log", "identity", "logit", "probit", "cloglog", "reciprocal", or "reciprocal-squared". Note that only sensible link-function and response-distribution combinations are permitted. If \code{method} = "EM", only "identity" can be used}

\item{optimiser}{(applicable only if \code{method} = "TMB") the optimising function used for model fitting when \code{method} = "TMB" (default is \code{nlminb}). Users may pass in a function object or a string corresponding to a named function. Optional parameters may be passed to \code{optimiser} via \code{...}. The only requirement of \code{optimiser} is that the first three arguments correspond to the initial parameters, the objective function, and the gradient, respectively (this may be achieved by simply constructing a wrapper function)}

\item{fs_by_spatial_BAU}{(applicable only in a spatio-temporal setting and if \code{method} = "TMB") if \code{TRUE}, then each spatial BAU is associated with its own fine-scale variance parameter; otherwise, a single fine-scale variance parameter is used}

\item{known_sigma2fs}{known value of the fine-scale variance parameter. If \code{NULL} (the default), the fine-scale variance parameter is estimated as usual. If \code{known_sigma2fs} is not \code{NULL}, the fine-scale variance is fixed to the supplied value; this may be a scalar, or vector of length equal to the number of spatial BAUs (if \code{fs_by_spatial_BAU = TRUE})}

\item{taper}{positive numeric indicating the strength of the covariance/partial-correlation tapering. Only applicable if \code{K_type} = "block-exponential", or if \code{K_type} = "precision" and the the basis-functions are irregular or the manifold is not the plane. If \code{taper} is \code{NULL} (default) and \code{method} = "EM", no tapering is applied; if \code{method} = "TMB", tapering must be applied (for computational reasons), and we set it to 3 if it is unspecified}

\item{...}{other parameters passed on to \code{auto_basis()} and \code{auto_BAUs()} when calling \code{FRK()}, or the user specified function \code{optimiser()} when calling \code{FRK()} or \code{SRE.fit()}}

\item{normalise_basis}{flag indicating whether to normalise the basis functions so that they reproduce a stochastic process with approximately constant variance spatially}

\item{include_fs}{(applicable only if \code{method} = "TMB") flag indicating whether the fine-scale variation should be included in the model}

\item{object}{object of class \code{SRE} returned from the constructor \code{SRE()} containing all the parameters and information on the SRE model. Note that prior to v2.x, \code{loglik()} and \code{SRE.fit()} took the now-defunct argument \code{SRE_model} instead of \code{object}}

\item{newdata}{object of class \code{SpatialPoylgons}, \code{SpatialPoints}, or \code{STI}, indicating the regions or points over which prediction will be carried out. The BAUs are used if this option is not specified.}

\item{obs_fs}{flag indicating whether the fine-scale variation sits in the observation model (systematic error; indicated by \code{obs_fs = TRUE}) or in the process model (process fine-scale variation; indicated by \code{obs_fs = FALSE}, default). For non-Gaussian data models, and/or non-identity link functions, if \code{obs_fs = TRUE}, then the fine-scale variation is removed from the latent process \eqn{Y}; however, they are re-introduced for computation of the conditonal mean \eqn{\mu} and response variable \eqn{Z}}

\item{pred_time}{vector of time indices at which prediction will be carried out. All time points are used if this option is not specified}

\item{covariances}{(applicable only for \code{method} = "EM") logical variable indicating whether prediction covariances should be returned or not. If set to \code{TRUE}, a maximum of 4000 prediction locations or polygons are allowed}

\item{n_MC}{(applicable only if \code{method} = "TMB") a positive integer indicating the number of MC samples at each location}

\item{type}{(applicable only if \code{method} = "TMB") vector of strings indicating the quantities for which inference is desired. If "link" is in \code{type}, inference on the latent Gaussian process \eqn{Y(\cdot)} is included; if "mean" is in \code{type}, inference on the mean process \eqn{\mu(\cdot)} is included (and the probability process, \eqn{\pi(\cdot)},  if applicable); if "response" is in \code{type}, inference on the noisy data process \eqn{Z} is included. Only applicable if \code{method} = "TMB"}

\item{k}{(applicable only if \code{response} is "binomial" or "negative-binomial") vector of size parameters at each BAU}

\item{percentiles}{(applicable only if \code{method} = "TMB") a vector of scalars in (0, 100) specifying the desired percentiles of the posterior predictive distribution; if \code{NULL}, no percentiles are computed}

\item{kriging}{(applicable only if \code{method} = "TMB") string indicating the kind of kriging: "simple" ignores uncertainty due to estimation of the fixed effects, while "universal" accounts for this source of uncertainty}
}
\description{
The Spatial Random Effects (SRE) model is the central object in FRK. The function \code{FRK} provides a wrapper for the construction and estimation of the SRE object from data, using the functions \code{SRE} (the object constructor) and \code{SRE.fit} (for fitting it to the data). Please see \code{\link{SRE-class}} for more details on the SRE object's properties and methods.
}
\details{
\strong{Model description}

The hierarchical model implemented in \code{FRK} is a spatial generalised 
linear mixed model (GLMM), which may be summarised as
\deqn{Z_j \mid \mu_{Z,j}, \psi \sim EF(\mu_{Z, j}, \psi)}
\deqn{\mu_Z = C\mu}
\deqn{g(\mu) = Y}
\deqn{Y = T\alpha + S\eta + \xi}
\deqn{\alpha \mid \theta \sim N(0, K)}
\deqn{\xi \mid \sigma^2_\xi \sim N(0, \sigma^2_\xi V),}
where \eqn{Z_j} denotes a datum, \eqn{EF(\cdot)} denotes an exponential 
family member with mean parameter \eqn{\mu_{Z, j}}, \eqn{\mu} is the mean 
process evaluated over the BAUs, \eqn{g(\cdot)} is a link function that links
the mean process \eqn{\mu(\cdot)} to the latent Gaussian process \eqn{Y(\cdot)},  
\eqn{Y} is the latent Gaussian process evaluated over the BAUs, \eqn{T} are 
regression covariates at the BAU level associated with regression parameters
\eqn{\alpha}, \eqn{S} is a matrix of basis function evaluations over the BAUs, 
\eqn{\eta} are the random coefficients associated with the basis functions, and \eqn{\xi} is 
a vector containing fine-scale variation at the BAU level. The prior 
distribution of the basis-function coefficients, \eqn{\eta}, are formulated 
using either a covariance or precision matrix, depending on the argument 
\code{K_type}; the parameters of these matrices are estimated during model 
fitting. The covariance matrix of \eqn{\xi} is diagonal, with its 
diagonal elements proportional to the field `fs' in the 
BAUs (typically set to one). The constant of proportionality is estimated 
during model fitting. 

When the data is Gaussian, and an identity link function is used, the preceding 
model simplifies considerably: specifically,
\deqn{Z = CY + C\delta + e,} 
where \eqn{Z} is the data vector, \eqn{\delta} is systematic error at the 
BAU level, and \eqn{e} represents independent measurement error.  

\strong{Set-up}

\code{SRE()} 
constructs a spatial random effects model from the user-defined formula, data object (a list 
of spatially-referenced data), basis functions and a set of Basic Areal Units (BAUs). 
It first takes each object in the list \code{data} and maps it to the BAUs -- this 
entails binning point-referenced data into the BAUs (and averaging within the 
BAU if \code{average_in_BAU = TRUE}), and finding which BAUs are associated 
with observations. Following this, the incidence matrix, \eqn{C}, is 
constructed. 
All required matrices (\eqn{S}, \eqn{T}, \eqn{C}, etc.) 
are constructed within \code{SRE()} and returned as part of the \code{SRE} object. 
\code{SRE()} also intitialises the parameters and random effects using 
sensible defaults. Please see 
\code{\link{SRE-class}} for more details. 
The functions \code{observed_BAUs()} and \code{unobserved_BAUs()} return the 
indices of the observed and unobserved BAUs, respectively. 

\strong{Model fitting}

\code{SRE.fit()} takes an object of class \code{SRE} and estimates all unknown
parameters, namely the covariance matrix \eqn{K}, the fine scale variance
(\eqn{\sigma^2_{\xi}} or \eqn{\sigma^2_{\delta}}, depending on whether Case 1
or Case 2 is chosen; see the vignette) and the regression parameters \eqn{\alpha}.
There are two methods of model fitting currently implemented, both of which 
implement maximum likelihood estimation (MLE).
\itemize{
 \item{MLE via the expectation maximisation
 (EM) algorithm. }{This method is implemented only
 for Gaussian data and an identity link function.
 The log-likelihood (given in Section 2.2 of the vignette) is evaluated at each
iteration at the current parameter estimate. Optimation continues until
convergence is reached (when the log-likelihood stops changing by more than
\code{tol}), or when the number of EM iterations reaches \code{n_EM}.
The actual computations for the E-step and M-step are relatively straightforward.
The E-step contains an inverse of an \eqn{r \times r} matrix, where \code{r}
is the number of basis functions which should not exceed 2000. The M-step
first updates the matrix \eqn{K}, which only depends on the sufficient
statistics of the basis-function coefficients \eqn{\eta}. Then, the regression
parameters \eqn{\alpha} are updated and a simple optimisation routine
(a line search) is used to update the fine-scale variance
\eqn{\sigma^2_{\delta}} or \eqn{\sigma^2_{\xi}}. If the fine-scale errors and
measurement random errors are homoscedastic, then a closed-form solution is
available for the update of \eqn{\sigma^2_{\xi}} or \eqn{\sigma^2_{\delta}}.
Irrespectively, since the updates of \eqn{\alpha}, and \eqn{\sigma^2_{\delta}}
or \eqn{\sigma^2_{\xi}}, are dependent, these two updates are iterated until
the change in \eqn{\sigma^2_{\cdot}} is no more than 0.1\%.}
 \item{MLE via \code{TMB}. }{This method is implemented for
 all available data models and link functions offered by \code{FRK}. Furthermore,
 this method faciliates the inclusion of many more basis function than possible
 with the EM algorithm (in excess of 10,000). \code{TMB} applies
 the Laplace approximation to integrate out the latent random effects from the
 complete-data likelihood. The resulting approximation of the marginal
 log-likelihood, and its derivatives with respect to the parameters, are then
 called from within \code{R} using the optimising function \code{optimiser}
 (default \code{nlminb()}).}
}

\code{info_fit()} extracts information on the fitting (convergence, etc.), 
\code{coef()} extracts the estimated regression regression coefficients, and 
\code{loglik()} returns the final log-likelihood. 

\emph{Wrapper for set-up and model fitting}

The function \code{FRK()} acts as a wrapper for the functions \code{SRE()} and 
\code{SRE.fit()}. An added advantage of using \code{FRK()} directly is that it 
automatically generates BAUs and basis functions based on the data. Hence 
\code{FRK()} can be called using only a list of data objects and an \code{R} 
formula, although the \code{R} formula can only contain space or time as 
covariates when BAUs are not explicitly supplied with the covariate data.


\strong{Prediction}

Once the parameters are estimated, the \code{SRE} object is passed onto the 
function \code{predict()} in order to carry out optimal predictions over the 
same BAUs used to construct the SRE model with \code{SRE()}. The first part 
of the prediction process is to construct the matrix \eqn{S} over the 
prediction polygons. This is made computationally efficient by treating the 
prediction over polygons as that of the prediction over a combination of BAUs. 
This will yield valid results only if the BAUs are relatively small. Once the 
matrix \eqn{S} is found, a standard Gaussian inversion (through conditioning) 
using the estimated parameters is used for prediction.

\code{predict} returns the BAUs (or an object specified in \code{newdata}), 
which are of class \code{SpatialPixelsDataFrame}, \code{SpatialPolygonsDataFrame}, 
or \code{STFDF}, with predictions and 
uncertainty quantification added. 
If \code{method} = "TMB", the returned object is a list, containing the 
previously described predictions, and a list of Monte Carlo samples. 
The predictions and uncertainties can be easily plotted using \code{\link{plot}}
or \code{spplot} from the package \code{sp}.
}
\examples{
library("FRK")
library("sp")
## Generate process and data
m <- 250                                                   # Sample size
zdf <- data.frame(x = runif(m), y= runif(m))               # Generate random locs
zdf$Y <- 3 + sin(7 * zdf$x) + cos(9 * zdf$y)               # Latent process
zdf$z <- rnorm(m, mean = zdf$Y)                            # Simulate data
coordinates(zdf) = ~x+y                                    # Turn into sp object

## Construct BAUs and basis functions 
BAUs <- auto_BAUs(manifold = plane(), data = zdf, 
                  nonconvex_hull = FALSE, cellsize = c(0.03, 0.03), type="grid") 
BAUs$fs <- 1 # scalar fine-scale covariance matrix
basis <- auto_basis(manifold =  plane(), data = zdf, nres = 2)

## Fit the SRE model
S <- SRE(f = z ~ 1, list(zdf), basis = basis, BAUs = BAUs)

## Compute observed and unobserved BAUs    
observed_BAUs(S)
unobserved_BAUs(S)   

## Fit with 5 EM iterations so as not to take too much time
S <- SRE.fit(S,n_EM = 5, tol = 0.01, print_lik = TRUE)

## Check fit info, final log-likelihood, and estimated regression coefficients
info_fit(S)
loglik(S)
coef(S)

## Predict over BAUs
pred <- predict(S)

## Plot
\dontrun{
plotlist <- plot(S, pred)
ggpubr::ggarrange(plotlist = plotlist, nrow = 1, align = "hv", legend = "top")}
}
\references{
Zammit-Mangion, A. and Cressie, N. (2017). FRK: An R package for spatial and spatio-temporal prediction with large datasets. Journal of Statistical Software, 98(4), 1-48. doi:10.18637/jss.v098.i04.
}
\seealso{
\code{\link{SRE-class}} for details on the SRE object internals, 
\code{\link{auto_basis}} for automatically constructing basis functions, and
\code{\link{auto_BAUs}} for automatically constructing BAUs.
}
