% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/mvspectrum.R, R/univariate_spectrum-functions.R
\name{mvspectrum}
\alias{check_mvspectrum_normalized}
\alias{get_spectrum_from_mvspectrum}
\alias{mvpgram}
\alias{mvspectrum}
\alias{normalize_mvspectrum}
\alias{spectrum_of_linear_combination}
\title{Estimates spectrum of multivariate time series}
\usage{
mvspectrum(series, method = c("pgram", "multitaper", "direct", "lag window",
  "wosa", "mvspec", "ar"), normalize = FALSE, smoothing = FALSE, ...)

normalize_mvspectrum(mvspectrum.output)

check_mvspectrum_normalized(f.U)

mvpgram(series)

get_spectrum_from_mvspectrum(mvspectrum.output,
  which = seq_len(dim(mvspectrum.output)[2]))

spectrum_of_linear_combination(mvspectrum.output, beta)
}
\arguments{
\item{series}{a \eqn{T \times K} array with \code{T} observations from the
\eqn{K}-dimensional time series \eqn{\mathbf{X}_t}. Can be a \code{matrix}, \code{data.frame},
or a multivariate \code{ts} object.}

\item{method}{string; method for spectrum estimation; see \code{method} argument in
\code{\link[sapa]{SDF}} (in the \pkg{sapa} package); use
\code{"mvspec"} to use \code{\link[astsa]{mvspec}} (\pkg{astsa} package); or
use \code{"pgram"} to use \code{\link[stats]{spec.pgram}}.}

\item{normalize}{logical; if \code{TRUE} the spectrum will be normalized (see
Value below for details).}

\item{smoothing}{logical; if \code{TRUE} the spectrum will be
smoothed with a nonparametric estimate using \code{\link[mgcv]{gam}}
and an exponential family (with \code{link = log}). Only works
for univariate spectrum. The smoothing
parameter is chosen automatically using generalized cross-validation
(see \code{\link[mgcv]{gam}} for details). Default: \code{FALSE}.}

\item{mvspectrum.output}{an object of class \code{"mvspectrum"} representing
the multivariate spectrum of \eqn{\mathbf{X}_t} (not necessarily \code{normalize}d).}

\item{f.U}{multivariate spectrum of class \code{'mvspectrum'} with
\code{normalize = TRUE}.}

\item{which}{integer(s); the spectrum of which series whould be extracted. By default,
it returns all univariate spectra as a matrix (frequencies in rows).}

\item{beta}{numeric; vector \eqn{\boldsymbol \beta} that defines the linear
combination.}

\item{\dots}{additional arguments passed to \code{\link[sapa]{SDF}} or
\code{\link[astsa]{mvspec}} (e.g., \code{taper})}
}
\value{
\code{mvspectrum} returns a 3D array of dimension \eqn{num.freqs \times K \times K}, where
\itemize{
 \item num.freqs is the number of frequencies
 \item K is the number of series (columns in \code{series}).
}

\code{normalize_mvspectrum} returns a normalized spectrum over
positive frequencies, which:
\describe{
  \item{univariate:}{adds up to \eqn{0.5},}
  \item{multivariate:}{adds up to Hermitian \eqn{K \times K} matrix
  with 0.5 in the diagonal and purely imaginary elements in the off-diagonal.}
}

\code{check_mvspectrum_normalized} throws an error if spectrum is not
normalized correctly.

\code{get_spectrum_from_mvspectrum} returns either a matrix of all univariate spectra,
or one single column (if \code{which} is specified.)

\code{spectrum_of_linear_combination} returns a vector with length equal to
the number of rows of \code{mvspectrum.output}.
}
\description{
The spectrum of a multivariate time series is a matrix-valued function of the
frequency \eqn{\lambda \in [-\pi, \pi]}, which is symmetric/Hermitian around
\eqn{\lambda = 0}.

\code{mvspectrum} estimates it and returns a 3D array of dimension
\eqn{num.freqs \times K \times K}.  Since the spectrum is symmetric/Hermitian around
 \eqn{\lambda = 0} it is sufficient to store only positive frequencies.
In the implementation in this package we thus usually
consider only positive frequencies (omitting \eqn{0}); \code{num.freqs} refers
to the number of positive frequencies only.

\code{normalize_mvspectrum} normalizes the spectrum such that
it adds up to \eqn{0.5} over all positive frequencies (by symmetry it will
add up to 1 over the whole range -- thus the name \emph{normalize}).

For a \eqn{K}-dimensional time series it adds
up to a Hermitian \eqn{K \times K} matrix with 0.5 in the diagonal and
imaginary elements (real parts equal to \eqn{0}) in the off-diagonal.
Since it is Hermitian the mvspectrum will add up to the identity matrix
over the whole range of frequencies, since the off-diagonal elements
are purely imaginary (real part equals 0) and thus add up to 0.

\code{check_mvspectrum_normalized} checks if the spectrum is normalized
(see \code{\link{normalize_mvspectrum}} for the requirements).

\code{mvpgram} computes the multivariate periodogram estimate using
bare-bone multivariate fft (\code{\link[stats]{mvfft}}). Please use
\code{mvspectrum(..., method = 'pgram')} instead of \code{mvpgram} directly.

This function is merely included to have one method that does not
require the \pkg{astsa} nor the \pkg{sapa} R packages.  However,
it is strongly encouraged to install either one of them to get (much)
better estimates.  See Details.

\code{get_spectrum_from_mvspectrum} extracts the spectrum of one time series from an
\code{"mvspectrum"} object by taking the i-th diagonal entry for each frequency.

\code{spectrum_of_linear_combination} computes the spectrum of the linear
combination  \eqn{\mathbf{y}_t = \mathbf{X}_t \boldsymbol \beta} of \eqn{K}
time series \eqn{\mathbf{X}_t}.  This can be efficiently computed by the
quadratic form
\deqn{
  f_{y}(\lambda) = \boldsymbol \beta' f_{\mathbf{X}}(\lambda) \boldsymbol \beta \geq 0,
}
for each \eqn{\lambda}. This holds for any \eqn{\boldsymbol \beta}
(even \eqn{\boldsymbol \beta = \boldsymbol 0} -- not only for
\eqn{||\boldsymbol \beta ||_2 = 1}.
For \eqn{\boldsymbol \beta = \boldsymbol e_i} (the i-th basis vector)
this is equivalent to \code{get_spectrum_from_mvspectrum(..., which = i)}.
}
\details{
For an orthonormal time series \eqn{\mathbf{U}_t} the raw periodogram adds up
to \eqn{I_K}
over all (negative and positive) frequencies.  Since we only consider
positive frequencies, the normalized multivariate spectrum should add up to
\eqn{0.5 \cdot I_K} plus a Hermitian imaginary matrix (which will add up to zero
when combined with its symmetric counterpart.)
As we often use non-parametric smoothing for less variance, the spectrum estimates
do not satisfy this identity exactly. \code{normalize_mvspectrum} thus adjust the
estimates so they satisfy it again exactly.

\code{mvpgram} has no options for improving spectrum estimation whatsoever.
It thus yields very noisy (in fact, inconsistent) estimates of the
multivariate spectrum \eqn{f_{\mathbf{X}}(\lambda)}.
If you want to obtain better estimates then please use other \code{method}s in
\code{\link{mvspectrum}} (this is highly recommended to obtain more
reasonable/stable estimates).
}
\examples{
set.seed(1)
XX <- cbind(rnorm(100), arima.sim(n = 100, list(ar = 0.9)))
ss3d <- mvspectrum(XX)
dim(ss3d)

ss3d[2,,] # at omega_1; in general complex-valued, but Hermitian
identical(ss3d[2,,], Conj(t(ss3d[2,,]))) # is Hermitian

ss <- mvspectrum(XX[, 1], smoothing = TRUE)

\dontrun{
  mvspectrum(XX, normalize = TRUE)
}
ss <- mvspectrum(whiten(XX)$U, normalize = TRUE)

xx <- scale(rnorm(100), center = TRUE, scale = FALSE)
var(xx)
sum(mvspectrum(xx, normalize = FALSE, method = "direct")) * 2
sum(mvspectrum(xx, normalize = FALSE, method = "wosa")) * 2
xx <- scale(rnorm(100), center = TRUE, scale = FALSE)
ss <- mvspectrum(xx)
ss.n <- normalize_mvspectrum(ss)
sum(ss.n)
# multivariate
UU <- whiten(matrix(rnorm(40), ncol = 2))$U
S.U <- mvspectrum(UU, method = "wosa")
mvspectrum2wcov(normalize_mvspectrum(S.U))
XX <- matrix(rnorm(1000), ncol = 2)
SS <- mvspectrum(XX, "direct")
ss1 <- mvspectrum(XX[, 1], "direct")

SS.1 <- get_spectrum_from_mvspectrum(SS, 1)
plot(ss1, SS.1)
XX <- matrix(arima.sim(n = 1000, list(ar = 0.9)), ncol = 4)
beta.tmp <- rbind(1, -1, 2, 0)
yy <- XX \%*\% beta.tmp

SS <- mvspectrum(XX, "wosa")
ss.yy.comb <- spectrum_of_linear_combination(SS, beta.tmp)
ss.yy <- mvspectrum(yy, "wosa")

plot(ss.yy, log = "y")
lines(ss.yy.comb, col = "red", lty = 1, lwd = 2)
}
\references{
See References in \code{\link[stats]{spectrum}}, \code{\link[sapa]{SDF}},
\code{\link[astsa]{mvspec}}.
}
\keyword{manip}
\keyword{ts}

