

#' Dominance `(n,k)` rule for magnitude tables
#' 
#' Supports application of multiple values for `n` and `k`. The function works on magnitude tables containing negative cell values by calculating contribution based on absolute values.
#' 
#' This methodn only supports suppressing a single numeric variable.
#' @param data the dataset
#' @param x ModelMatrix generated by parent function
#' @param crossTable crossTable generated by parent function
#' @param numVar vector containing numeric values in the data set
#' @param n parameter `n` in dominance rule. 
#' @param k parameter `k` in dominance rule.
#' @param protectZeros parameter determining whether cells with value 0 should be suppressed.
#' @param charVar Variable in data holding grouping information. Dominance will be calculated after aggregation within these groups.
#' @param ... unused parameters
#' @return logical vector that is `TRUE` in positions corresponding to cells breaching the dominance rules.
#' @export
#'
#' @author Daniel Lupp 
#' 
DominanceRule <- function(data, x, crossTable, numVar, n, k,
                          protectZeros = FALSE, charVar, ...) {
  if (length(n) != length(k))
    stop("You must provide an equal number of inputs for n and k.")
  if (is.null(numVar))
    stop("You must provide a numeric variable numVar to use the dominance rule.")
  if (length(numVar) > 1){
    warning("Multiple numVar were supplied, only the first is suppressed.")
    numVar <- numVar[1]
  }
  if (!protectZeros)
    cat("Zeroes are not primary suppressed.\n")
  else
    cat("Zeroes are primary suppressed.\n")
  abs_num <- as.data.frame(as.matrix(crossprod(x, as.matrix(abs(data[, numVar, drop = FALSE])))))
  abs_cellvals <- abs(data[[numVar]])
  
  if (length(charVar)) {
    if (length(charVar) == 1) {
      charVar_groups <- data[[charVar]]
    } else {
      stop("Only single charVar implemented")
    }
  } else {
    charVar_groups <- NULL
  }
  
  primary <- mapply(function (a,b) FindDominantCells(x, abs_cellvals, abs_num, a,b, charVar_groups = charVar_groups),
                    n,k)

  dominant <- apply(primary, 1, function (x) Reduce(`|`, x))
  if (!protectZeros) 
    return(dominant)
  dominant | (abs_num == 0)
}

FindDominantCells <- function(x, cellvals, num, n, k, charVar_groups) {
  max_cont <- MaxContribution(x, cellvals, n = n, groups = charVar_groups)
  max_cont[is.na(max_cont)] <- 0
  as.vector(num > 0 & rowSums(max_cont) > num*k/100)
}
