\name{ldBands}
\alias{ldBands}
\alias{summary.ldBands}
\alias{print.ldBands}
\alias{plot.ldBands}
\alias{print.summary.ldBands}
\title{Group Sequential Boundaries using the Lan-DeMets Approach}
\description{
This function computes and plots group sequential stopping boundaries
from the Lan-DeMets method with a variety of \eqn{\alpha}-spending
functions using the \code{ld98} program from the Department of
Biostatistics, University of Wisconsin written by DM Reboussin, DL
DeMets, KM Kim, and KKG Lan.  Such stopping boundaries are
useful for early termination of clinical trials for safety problems or
for efficacy.  Simple \code{plot} and \code{print} methods are
implemented.  Simple sample size and minimally detectable effect sizes
given sample sizes may be obtained with a \code{summary} method if
\code{power} was specified to \code{ldBands}.  Alternatively,
\code{summary} computes, for each look, the difference in means that
must be achieved to cross a boundary if \code{n} and \code{sd} are
specified, or the minimum difference in proportions and the odds ratios
that must be achieved to cross a boundary if \code{n} and \code{pbar}
are specified.
}
\usage{
ldBands(n = length(times), times = NULL, alpha = 0.05, sided = 2,
        alphaLower=alpha/2, alphaUpper=alpha/2,
        information = NULL,
        spending=c('OBrien-Fleming','Pocock','alpha*t^phi',
                   'Hwang-Shih-DeCani'),
        phi=1,
        spending2=c('OBrien-Fleming','Pocock','alpha*t^phi',
                    'Hwang-Shih-DeCani'),
        phi2=1,
        truncate = Inf, power = NULL, pr = TRUE)
\method{print}{ldBands}(x, \dots)
\method{plot}{ldBands}(x, xlab='Time', ylab='Z', actual=NULL, type='b', \dots)

\method{summary}{ldBands}(object, stdiff=NULL, n=NULL,
        p1=NULL, p2=NULL,  hr=NULL, events=NULL,
        pbar=NULL, sd=NULL, \dots)
\method{print}{summary.ldBands}(x, \dots)
}
\arguments{
  \item{n}{number of interim analyses.  If \code{times} is given, is
	automatically taken as the length of \code{times}.
  For \code{summary.ldBands}, \code{n} is the sample size, to obtain
  detectable standardized difference.}
  \item{times}{times at which interim analyses are done}
  \item{alpha}{overall \eqn{\alpha} level for the multiple tests.
	Default is \code{0.05}.  If \code{sided=3} is the \eqn{\alpha}-level
  for the lower bounds, otherwise is the total \eqn{\alpha}.}
  \item{sided}{set to \code{1} to use a one-sided test, \code{3} for
	asymmetric two-sided bounds}
  \item{alphaLower}{\eqn{\alpha}-level for lower bound if
	\code{sided=3}.  Defaults to \eqn{\alpha/2}.  When \code{sided=3}
	\code{alpha} is recalculated from \code{alphaLower+alphaUpper}.}
  \item{alphaUpper}{\eqn{\alpha}-level for upper bound if
	\code{sided=3}.  Defaults to \eqn{\alpha/2}.}
  \item{information}{a vector of information times if different from
	\code{times}.  Used for computing covariances of test statistics.}
  \item{spending}{an \eqn{\alpha} spending function.  Default is
	O'Brien-Fleming function.  If \code{sided=3} is the spending
	function for the lower bounds.}
  \item{phi}{parameter for the third and fourth types of spending
	functions (exponent of time for the third, default is \code{1})}
  \item{spending2}{spending function for the upper bounds if
	\code{sided=3}.  Defaults to same spending function for lower
	bounds.}
  \item{phi2}{parameter for third and fourth spending functions if
	\code{sided=3}.  Default is \code{phi}.}
  \item{truncate}{a value at which to truncate \eqn{Z} critical values
	so that early looks will have some power without really affecting
	the overall \eqn{\alpha} level.  Default is no truncation.}
  \item{power}{A power for which to compute a drift parameter; useful in
  sample size calculations}
  \item{pr}{set to \code{FALSE} to supress the actual output of the
	\code{ld98} program}
  \item{x}{an object created by \code{ldBands} or \code{summary.ldBands}}
  \item{xlab}{\eqn{x}-axis label}
  \item{ylab}{\eqn{y}-axis label}
  \item{actual}{an optional list with two components: times and \eqn{z}
	values to add as points to the first plot}
  \item{type}{default is \code{"b"} causing both points and lines to be
	drawn for boundaries.  Use \code{type="l"} for example to suppress
	drawing circles at points.}
  \item{object}{an object created by \code{ldBands}}
  \item{stdiff}{standardized difference to detect}
  \item{p1}{probability of event in group 1}
  \item{p2}{probability of event in group 2, to compare with \code{p1}
	in order to obtain sample size for two-sample binomial}
  \item{hr}{hazard ratio to detect, to obtain needed number of events at
	end of study for either treatment arm using the logrank test}
  \item{events}{number of events per treatment arm at end of study, to
	obtain detectable hazard ratio}
  \item{pbar}{mean of actual probabilities of events in two treatment
	arms}
  \item{sd}{standard deviation of an observation}
  \item{\dots}{unused}
}
\details{
This function assumes that you have stored the \code{ld98} executable
in a subdirectory that is in your system path.  Obtain \code{ld98}
program from the URL given in the reference below.

The \code{plot} method makes two plots if \code{power} is specified, the
second containing exit and cumulative exit probabilities as a function
of time.  If \code{par(mfrow=c())} is not already set to something besides
\code{c(1,1)}, \code{par(mfrow=c(2,1))} is set temporarily.
}
\value{
  A list of class \code{"ldBands"} with the following components.  When
  \code{summary} is used and \code{n} and either \code{pbar} or
  \code{sd} are given, \code{summary} returns the same object returned
  by \code{ldBands} but with possible components
  \code{diff.lower,diff.lower} (needed difference means or proportions) and
  \code{or.lower,or.upper} (needed odds ratios).
  \item{data}{a data frame containing the main calculations}
  \item{power}{power specified, if any}
  \item{drift}{drift calculated, if \code{power} specified}
  \item{type}{\code{"boundaries"} if \code{power} not specified,
	\code{"power"} otherwise}
  \item{n}{number of interim analyses, for \code{ldBands}.  For
	\code{summary} is the number of observations per arm, and it must be
	a vector with length equal to the number of looks if \code{pbar} or
	\code{sd} are given.}
  \item{alpha}{input value of \code{alpha}}
  \item{sided}{\code{1-3}}
  \item{alphaLower}{lower \eqn{\alpha}}
  \item{alphaUpper}{upper \eqn{\alpha}}
  \item{spending}{name of \eqn{\alpha} spending function used}
  \item{phi}{parameter for spending function}
  \item{spending2}{name of spending function for upper boundary.
	Defaults to \code{spending}}
  \item{phi2}{parameter for second spending function.  Defaults to
	\code{phi} but is ignored if \code{spending2} is the first or second
	type of spending function.}
  \item{truncate}{value at which \eqn{Z} statistics truncated (default
	is \code{Inf})}
  }
  \references{
	Reboussin DM, DeMets DL, Kim K-M, Lan KKG (1996): Programs
  for computing group sequential boundaries using the Lan-DeMets
  method.  \url{http://www.medsch.wisc.edu/landemets}

  Reboussin DM, DeMets DL, Kim K, Lan KKG (2000): Computations for group
  sequential boundaries using the Lan-DeMets spending function method.
  Controlled Clinical Trials 21:190-207.
  }
\author{Frank E Harrell Jr}
\seealso{\code{\link{gbayes}}}

\examples{
\dontrun{
# Get boundaries for O'Brien-Fleming spending function, 5 looks, alpha=.05
b <- ldBands(5, pr=FALSE)
plot(b)
# Same but times are irregular, and information times are different than
# test times.  Use Pocock spending function.
b <- ldBands(times=      c(.4, .6, .8, .9, .95),
             information=c(.42,.65,.83,.89,.94), spending='Pocock')

# Get power calculations
u <- ldBands(times=c(.4, .6, .8, .9, .95),  power=.9)
u$drift                  # standardize difference * sqrt(n per arm)
                         # needed to provide power=.9
summary(u, n=50)         # obtain detectable standardized difference
summary(u, p1=.4, p2=.5) # get sample size per arm, two-sample binomial
summary(u, hr=1.5)       # get number of events per arm needed
                         # to detect a hazard ratio of 1.5

# Asymmetric boundaries with different spending functions, truncate
b <- ldBands(5, sided=3, spending='alpha*t^phi', phi=1, phi2=1.5,
             alphaLower=.01, alphaUpper=.04, truncate=4)
b
plot(b)
# Compute differences in proportions and odds ratios needed to cross
# the boundaries, given a mean probability in two treatment arms of 0.1
# and given a vector of sample sizes per arm corresponding to the looks
s <- summary(b, n=seq(200,1000,by=200), pbar=.1)
s
d <- s$data
plot(d$n, d$or.lower, xlab='N Per Arm',
     ylab='Critical Odds Ratio', type='b',
     ylim=range(d$or.lower, d$or.upper), log='y')
lines(d$n, d$or.upper, type='b')
abline(h=1, lty=2)
}
}
\keyword{distribution}
\keyword{htest}
\keyword{design}
