\name{hyperbFit}
\alias{hyperbFit}
\alias{print.hyperbFit}
\alias{plot.hyperbFit}

\title{Fit the Hyperbolic Distribution to Data}
\description{
  Fits a hyperbolic distribution to data.  
  Displays the histogram, log-histogram (both with fitted densities),
  Q-Q plot and P-P plot for the fit which has the maximum likelihood.
}

\usage{
  hyperbFit(x, freq = NULL, breaks = NULL, ThetaStart = NULL,
            startMethod = "Nelder-Mead", startValues = "BN",
            method = "Nelder-Mead", hessian = FALSE,
            plots = FALSE, printOut = FALSE,
            controlBFGS = list(maxit=200),
            controlNLM = list(maxit=1000), maxitNLM = 1500, ...)

  \method{print}{hyperbFit}(x, digits = max(3, getOption("digits") - 3), ...)

  \method{plot}{hyperbFit}(x, which = 1:4,
       plotTitles = paste(c("Histogram of ","Log-Histogram of ",
                            "Q-Q Plot of ","P-P Plot of "), x$xName,
                          sep = ""),
       ask = prod(par("mfcol")) < length(which) && dev.interactive(), ...)
}

\arguments{
  \item{x}{Data vector for \code{hyperbFit}. Object of class
    \code{"hyperbFit"} for \code{print.hyperbFit}.}
  \item{freq}{A vector of weights with length equal to \code{length(x)}.}
  \item{breaks}{Breaks for histogram, defaults to those generated by 
    \code{hist(x, right = FALSE, plot = FALSE)}.}
  \item{ThetaStart}{A user specified starting parameter vector Theta taking 
    the form \code{c(pi,zeta,delta,mu)}.}
  \item{startMethod}{Method used by \code{hyperbFitStart} in calls to
    \code{\link{optim}}.}
  \item{startValues}{Code giving the method of determining starting
    values for finding the maximum likelihood estimate of Theta.}
  \item{method}{Different optimisation methods to consider.  
    See \bold{Details}.}
  \item{hessian}{Logical. If \code{TRUE} the value of the hessian is
    returned.}
  \item{plots}{Logical. If \code{FALSE} suppresses printing of the
    histogram, log-histogram, Q-Q plot and P-P plot.}
  \item{printOut}{Logical. If \code{FALSE} suppresses printing of
    results of fitting.}
  \item{controlBFGS}{A list of control parameters for \code{optim} when using 
    the \code{"BFGS"} optimisation.}
  \item{controlNLM}{A list of control parameters for \code{optim}
    when using the \code{"Nelder-Mead"} optimisation.}
  \item{maxitNLM}{A positive integer specifying the maximum number of
    iterations when using the \code{"nlm"} optimisation.}
  \item{digits}{Desired number of digits when the object is printed.}
   \item{which}{If a subset of the plots is required, specify a subset of
    the numbers \code{1:4}.}
  \item{plotTitles}{Titles to appear above the plots.}
  \item{ask}{Logical. If \code{TRUE}, the user is \emph{ask}ed before
    each plot, see \code{\link{par}(ask = .)}.}
  \item{\dots}{Passes arguments to \code{par}, \code{hist},
		\code{logHist}, \code{qqhyperb} and \code{pphyperb}.}
}
\details{
  \code{startMethod} can be either \code{"BFGS"} or
	\code{"Nelder-Mead"}.
	
  \code{startValues} can be one of the following:
  \itemize{
    \item{\code{"US"}}{User-supplied.}
    \item{\code{"BN"}}{Based on Barndorff-Nielsen (1977).}
    \item{\code{"FN"}}{A fitted normal distribution.}
    \item{\code{"SL"}}{Based on a fitted skew-Laplace distribution.}
    \item{\code{"MoM"}}{Method of moments.}
  }
  For the details concerning the use of \code{ThetaStart},
  \code{startMethod}, and \code{startValues}, see
  \code{\link{hyperbFitStart}}.
  
  The three optimisation methods currently available are:
  \itemize{
  \item{\code{"BFGS"}}{Uses the quasi-Newton method \code{"BFGS"} as
    documented in \code{\link{optim}}.}
  \item{\code{"Nelder-Mead"}}{Uses an implementation of the Nelder and
    Mead method as documented in \code{\link{optim}}.}
  \item{\code{"nlm"}}{Uses the \code{\link{nlm}} function in R.}
  }

  For details of how to pass control information for optimisation using
  \code{\link{optim}} and \code{\link{nlm}}, see \code{\link{optim}} and
  \code{\link{nlm}.}

	When \code{method = "nlm"} is used, warnings may be produced. These do
	not appear to be a problem.
}
\value{
  A list with components:
    \item{Theta}{A vector giving the maximum likelihood estimate of
      Theta, as \code{(pi,zeta,delta,mu)}.}
    \item{maxLik}{The value of the maximised log-likelihood.}
    \item{hessian}{If \code{hessian} was set to \code{TRUE}, the value
      of the hessian. Not present otherwise.}
    \item{method}{Optimisation method used.}
    \item{conv}{Convergence code. See the relevant documentation (either
      \code{\link{optim}} or \code{\link{nlm}}) for details on
      convergence.}
    \item{iter}{Number of iterations of optimisation routine.}
    \item{x}{The data used to fit the hyperbolic distribution.}
    \item{xName}{A character string with the actual \code{x} argument
      name.}
    \item{ThetaStart}{Starting value of Theta returned by call to
      \code{\link{hyperbFitStart}}.}
    \item{svName}{Descriptive name for the method finding start values.}
    \item{startValues}{Acronym for the method of finding start values.}
    \item{KNu}{Value of the Bessel function in the fitted density.}
    \item{breaks}{The cell boundaries found by a call to
      \code{\link{hist}}.}
    \item{midpoints}{The cell midpoints found by a call to
      \code{\link{hist}}.}
    \item{empDens}{The estimated density found by a call to
      \code{\link{hist}}.}

  }

\references{
  Barndorff-Nielsen, O. (1977)
  Exponentially decreasing distributions for the logarithm of particle size,
  \emph{Proc. Roy. Soc. Lond.},
  A\bold{353}, 401--419.
  
  Fieller, N. J., Flenley, E. C. and Olbricht, W. (1992)
  Statistics of particle size data.
  \emph{Appl. Statist.},
  \bold{41}, 127--146.

}

\author{David Scott \email{d.scott@auckland.ac.nz}, 
  Ai-Wei Lee, Jennifer Tso, Richard Trendall, Thomas Tran}


\seealso{
  \code{\link{optim}}, \code{\link{nlm}}, \code{\link{par}},
  \code{\link{hist}}, \code{\link{logHist}}, \code{\link{qqhyperb}},
  \code{\link{pphyperb}}, \code{\link{dskewlap}} and
  \code{\link{hyperbFitStart}}.

}

\examples{
Theta <- c(2,2,2,2)
dataVector <- rhyperb(500, Theta)
## See how well hyperbFit works
hyperbFit(dataVector)
hyperbFit(dataVector, plots = TRUE)
fit <- hyperbFit(dataVector)
par(mfrow = c(1,2))
plot(fit, which = c(1,3))

## Use nlm instead of default
hyperbFit(dataVector, method = "nlm")

}

\keyword{distribution}
